package ru.yandex.direct.binlogbroker.mysql;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.commons.lang3.tuple.Triple;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlog.model.ColumnType;
import ru.yandex.direct.binlog.model.CreateOrModifyColumn;
import ru.yandex.direct.binlog.model.CreateTable;
import ru.yandex.direct.binlog.model.DropColumn;
import ru.yandex.direct.binlog.model.DropTable;
import ru.yandex.direct.mysql.MySQLDataType;
import ru.yandex.direct.mysql.schema.ColumnSchema;
import ru.yandex.direct.mysql.schema.DatabaseSchema;
import ru.yandex.direct.mysql.schema.KeyColumn;
import ru.yandex.direct.mysql.schema.KeySchema;
import ru.yandex.direct.mysql.schema.ServerSchema;
import ru.yandex.direct.mysql.schema.TableSchema;
import ru.yandex.direct.utils.MySQLQuote;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.mysql.MySQLDataType.UNKNOWN;

@ParametersAreNonnullByDefault
public class MysqlUtil {
    private MysqlUtil() {
    }

    /**
     * Возвращает ColumnType, соответствующий заданному mysql-типу.
     * <p>
     * Методику получения columnType и dataType можно подглядеть в
     * {@link ru.yandex.direct.mysql.schema.ServerSchema#dump(Connection)}
     *
     * @param col Полное описание типа из mysql.
     *            Соответствует informational_schema.columns.column_type.
     * @return ColumnType, соответствующий mysql-типу
     */
    public static ColumnType mysqlDataTypeToColumnType(ColumnSchema col) {
        switch (MySQLDataType.byName(col.getDataType())) {
            case BIGINT:
                if (col.isUnsigned()) {
                    return ColumnType.UNSIGNED_BIGINT;
                } else {
                    return ColumnType.INTEGER;
                }

            case BLOB:
            case LONGBLOB:
            case MEDIUMBLOB:
            case VARBINARY:
                return ColumnType.BYTES;

            case CHAR:
            case ENUM:
            case JSON:
            case MEDIUMTEXT:
            case SET:
            case TEXT:
            case TINYTEXT:
            case VARCHAR:
                return ColumnType.STRING;

            case DATE:
                return ColumnType.DATE;

            case DATETIME:
            case TIMESTAMP:
                return ColumnType.TIMESTAMP;

            case DECIMAL:
                return ColumnType.FIXED_POINT;

            case DOUBLE:
            case FLOAT:
                return ColumnType.FLOATING_POINT;

            case BIT:
            case INT:
            case SMALLINT:
            case TINYINT:
                return ColumnType.INTEGER;

            default:
                throw new IllegalArgumentException("Can't handle " + col.getColumnType());
        }
    }

    @Nullable
    public static List<Pair<String, String>> looksLikeRenameTable(String sql) {
        List<String> splittedQuery = splitQuery(sql);
        if (splittedQuery.size() == 5
                && splittedQuery.get(0).equalsIgnoreCase("alter")
                && splittedQuery.get(1).equalsIgnoreCase("table")
                && splittedQuery.get(3).equalsIgnoreCase("rename")) {
            return ImmutableList.of(Pair.of(splittedQuery.get(2), splittedQuery.get(4)));
        } else if (splittedQuery.size() >= 5
                && splittedQuery.get(0).equalsIgnoreCase("rename")
                && splittedQuery.get(1).equalsIgnoreCase("table")) {
            int ind = 2;
            List<Pair<String, String>> result = new ArrayList<>();
            while (ind < splittedQuery.size()) {
                if (!splittedQuery.get(ind + 1).equalsIgnoreCase("to") || ind + 2 >= splittedQuery.size()) {
                    throw new IllegalStateException(String.format("can't handle sql %s", sql));
                }
                result.add(Pair.of(splittedQuery.get(ind), splittedQuery.get(ind + 2)));
                ind += 3;
            }
            return ImmutableList.copyOf(result);
        } else {
            return null;
        }
    }

    @Nullable
    public static List<Pair<String, String>> looksLikeRenameColumn(String sql) {

        List<String> splitQuery = splitQuery(sql);
        if (splitQuery.size() >= 6
                && splitQuery.get(0).equalsIgnoreCase("alter")
                && splitQuery.get(1).equalsIgnoreCase("table")) {

            String canonized = String.join(" ", splitQuery) + " ";
            Pattern rename = Pattern.compile("rename column (\\w+) to (\\w+) ", Pattern.CASE_INSENSITIVE);
            String dataTypeRegex = EnumSet.allOf(MySQLDataType.class).stream()
                    .filter(type -> type != UNKNOWN)
                    .map(MySQLDataType::getValue)
                    .collect(Collectors.joining("|"));
            Pattern changeWithRename = Pattern.compile("change column (\\w+) (\\w+) (" + dataTypeRegex + ")\\W",
                    Pattern.CASE_INSENSITIVE);
            List<Pair<String, String>> columnPairs = new ArrayList<>();
            for (Pattern pattern : Arrays.asList(rename, changeWithRename)) {
                Matcher matcher = pattern.matcher(canonized);
                while (matcher.find()) {
                    String firstColumn = matcher.group(1);
                    String secondColumn = matcher.group(2);
                    if (!firstColumn.equals(secondColumn)) {
                        columnPairs.add(Pair.of(firstColumn, secondColumn));
                    }
                }
            }
            return columnPairs.isEmpty() ? null : ImmutableList.copyOf(columnPairs);
        } else {
            return null;
        }
    }

    /**
     * Очень топорный парсинг sql-запросов, который много чего не учитывает и предназначен только для опознания самых
     * простых запросов. В будущем лучше воспользоваться нормальным парсером. Например, судя по документации неплох
     * https://github.com/JSQLParser/JSqlParser
     */
    @SuppressWarnings("squid:S3776")  // cognitive complexity
    public static List<String> splitQuery(String sql) {
        sql = sql
                .replaceAll("/\\*.*?\\*/", " ")
                .replaceAll(";.*$", "")
                .replaceAll(",", "");
        List<String> result = new ArrayList<>();
        boolean insideComment = false;
        String[] crudeSplitted = sql.split("\\s+");
        for (String part : crudeSplitted) {
            if (!part.isEmpty()) {
                if (part.equals("/*")) {
                    insideComment = true;
                }
                if (!insideComment) {
                    if (part.startsWith("`") && part.endsWith("`")) {
                        int lastIndex = part.lastIndexOf("`");
                        part = part.substring(0, lastIndex);
                        int preLastIndex = part.lastIndexOf("`");
                        part = part.substring(preLastIndex + 1);
                        result.add(part.replaceAll("``", "`"));
                    } else {
                        result.add(part);
                    }
                }
                if (part.equals("*/")) {
                    insideComment = false;
                }
            }
        }
        return result;
    }

    @Nullable
    public static String looksLikeTruncateTable(String sql) {
        List<String> splittedQuery = splitQuery(sql);
        if (splittedQuery.size() == 3
                && splittedQuery.get(0).equalsIgnoreCase("truncate")
                && splittedQuery.get(1).equalsIgnoreCase("table")) {
            return splittedQuery.get(2);
        } else {
            return null;
        }
    }

    public static int createTableEvent(BinlogEvent resultBinlogEvent, Map<String, CreateTable> oldTables,
                                       Map<String, CreateTable> newTables) {
        List<String> createdTables = newTables.keySet().stream()
                .filter(k -> !oldTables.containsKey(k))
                .collect(Collectors.toList());
        if (createdTables.isEmpty()) {
            return 0;
        } else if (createdTables.size() > 1) {
            throw new IllegalStateException(String.format("Got %d CREATE TABLE in one SQL-query: %s",
                    createdTables.size(),
                    createdTables.stream().map(MySQLQuote::quoteName).collect(Collectors.joining(", "))));
        } else {
            String tableName = createdTables.get(0);
            resultBinlogEvent
                    .withTable(tableName)
                    .withAddedSchemaChanges(newTables.get(tableName));
            return 1;
        }
    }

    public static int dropTableEvent(BinlogEvent resultBinlogEvent, Map<String, CreateTable> oldTables,
                                     Map<String, CreateTable> newTables) {
        List<String> droppedTables = oldTables.keySet().stream()
                .filter(k -> !newTables.containsKey(k))
                .sorted()
                .collect(Collectors.toList());
        if (droppedTables.isEmpty()) {
            return 0;
        } else {
            // по аналогии с rename - в table пишем первую таблицу
            resultBinlogEvent
                    .withTable(droppedTables.get(0));
            for (String table : droppedTables) {
                resultBinlogEvent
                        .withAddedSchemaChanges(new DropTable(table));
            }
            return droppedTables.size();
        }
    }

    public static void addColumnsEvent(BinlogEvent resultBinlogEvent, CreateTable oldTable,
                                       CreateTable newTable) {
        Set<String> oldColumnNames = oldTable.getColumns().stream()
                .map(CreateOrModifyColumn::getColumnName)
                .collect(toSet());
        for (CreateOrModifyColumn columnSchema : newTable.getColumns()) {
            if (!oldColumnNames.contains(columnSchema.getColumnName())) {
                resultBinlogEvent.addSchemaChanges(columnSchema);
            }
        }
    }

    public static void dropColumnsEvent(BinlogEvent resultBinlogEvent, CreateTable oldTable,
                                        CreateTable newTable) {
        Set<String> newColumnNames = newTable.getColumns().stream()
                .map(CreateOrModifyColumn::getColumnName)
                .collect(toSet());
        for (CreateOrModifyColumn columnSchema : oldTable.getColumns()) {
            if (!newColumnNames.contains(columnSchema.getColumnName())) {
                resultBinlogEvent.addSchemaChanges(new DropColumn()
                        .withColumnName(columnSchema.getColumnName()));
            }
        }
    }

    public static void modifyColumnsEvent(BinlogEvent resultBinlogEvent, CreateTable oldTable,
                                          CreateTable newTable) {
        // DIRECT-81584 Определение переименования колонки вызвала сложности и было оставлена на потом.
        // Используемая абстракция предоставляет только схему, которая была до изменения, и схему, которая стала после
        // изменения. При переименовании колонки в новой схеме не стало какой-то колонки, зато появилась какая-то другая
        // колонка. Непонятно, как отличить, то ли пришёл "ALTER TABLE CHANGE COLUMN old_name new_name TYPE", то ли
        // "ALTER TABLE DROP COLUMN old_name, ADD COLUMN new_name".
        Map<String, CreateOrModifyColumn> oldColumnSchemas =
                Maps.uniqueIndex(oldTable.getColumns(), CreateOrModifyColumn::getColumnName);
        for (CreateOrModifyColumn newColumnSchema : newTable.getColumns()) {
            String columnName = newColumnSchema.getColumnName();
            CreateOrModifyColumn oldColumnSchema = oldColumnSchemas.get(columnName);
            if (oldColumnSchema != null && !newColumnSchema.equals(oldColumnSchema)) {
                resultBinlogEvent.addSchemaChanges(newColumnSchema);
            }
        }
    }

    public static CreateTable tableSchemaAsCreateTable(TableSchema tableSchema) {
        CreateTable createTable = new CreateTable();
        for (ColumnSchema col : tableSchema.getColumns()) {
            createTable.addColumns(new CreateOrModifyColumn()
                    .withColumnName(col.getName())
                    .withColumnType(mysqlDataTypeToColumnType(col))
                    .withNullable(col.isNullable())
                    .withDefaultValue(col.getDefaultValue()));
        }
        createTable.setPrimaryKey(tableSchema
                .getPrimaryKey()
                .map(KeySchema::getColumns)
                .orElse(Collections.emptyList())
                .stream()
                .map(KeyColumn::getName)
                .collect(Collectors.toList()));
        return createTable;
    }

    public static int columnChangeEvent(BinlogEvent resultBinlogEvent, Map<String, CreateTable> oldTables,
                                        Map<String, CreateTable> newTables) {
        List<Triple<String, CreateTable, CreateTable>> changedTables = oldTables.keySet().stream()
                .filter(newTables::containsKey)
                .map(name -> Triple.of(name, oldTables.get(name), newTables.get(name)))
                .filter(triple -> !triple.getMiddle().equals(triple.getRight()))
                .collect(Collectors.toList());
        if (changedTables.isEmpty()) {
            return 0;
        }
        Preconditions.checkState(changedTables.size() == 1, "Changed several tables in one SQL-query");
        String tableName = changedTables.get(0).getLeft();
        CreateTable oldTable = changedTables.get(0).getMiddle();
        CreateTable newTable = changedTables.get(0).getRight();
        resultBinlogEvent.setTable(tableName);
        addColumnsEvent(resultBinlogEvent, oldTable, newTable);
        dropColumnsEvent(resultBinlogEvent, oldTable, newTable);
        modifyColumnsEvent(resultBinlogEvent, oldTable, newTable);
        return 1;
    }

    public static Map<String, CreateTable> tableByNameForDb(String db, ServerSchema serverSchema) {
        return StreamEx.of(serverSchema.getDatabases())
                .filter(d -> d.getName().equals(db))
                .map(DatabaseSchema::getTables)
                .flatMap(List::stream)
                .mapToEntry(TableSchema::getName, MysqlUtil::tableSchemaAsCreateTable)
                .toMap();
    }
}
