package ru.yandex.direct.binlogbroker.replicatetoyt;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.model.ColumnType;
import ru.yandex.direct.binlog.model.CreateOrModifyColumn;
import ru.yandex.direct.binlog.model.CreateTable;
import ru.yandex.direct.binlog.model.DropColumn;
import ru.yandex.direct.binlog.model.DropTable;
import ru.yandex.direct.binlog.model.RenameColumn;
import ru.yandex.direct.binlog.model.RenameTable;
import ru.yandex.direct.binlog.model.SchemaChange;
import ru.yandex.direct.binlog.model.Truncate;
import ru.yandex.yt.ytclient.tables.ColumnValueType;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class Alter {
    private static final Logger logger = LoggerFactory.getLogger(Alter.class);

    private static final Map<ColumnType, Object> DEFAULT_VALUES =
            ImmutableMap.<ColumnType, Object>builder()
                    .put(ColumnType.BYTES, new byte[0])
                    .put(ColumnType.DATE, "0000-00-00")
                    .put(ColumnType.FIXED_POINT, "0.0")
                    .put(ColumnType.FLOATING_POINT, 0.)
                    .put(ColumnType.INTEGER, 0L)
                    .put(ColumnType.STRING, "")
                    .put(ColumnType.TIMESTAMP, "0000-00-00T00:00:00")
                    .put(ColumnType.UNSIGNED_BIGINT, "0")
                    .build();

    private final List<RenameTable.RenameSingleTable> renames = new ArrayList<>();

    private final Map<String, Object> defaultValueByColumn = new HashMap<>();
    private final Map<String, Pair<ColumnValueType, ColumnValueType>> typeConversionByColumn = new HashMap<>();
    private final Map<String, String> newNameByColumn = new HashMap<>();
    private final Map<String, String> oldNameByColumn = new HashMap<>();
    private final Set<String> columnsToDelete = new HashSet<>();

    private boolean createTable = false;
    private boolean dropTable = false;
    private boolean renameTable = false;

    private final SchemaManager schemaManager;

    public Alter(SchemaManager schemaManager) {
        this.schemaManager = schemaManager;
    }

    @Nullable
    private Object getDefaultValue(CreateOrModifyColumn change, LocalDateTime eventTimestamp) {
        if (change.getDefaultValue() != null) {
            if (change.getDefaultValue().equals("CURRENT_TIMESTAMP")) {
                Preconditions.checkState(change.getColumnType().equals(ColumnType.TIMESTAMP));
                return eventTimestamp.toString();
            }
            switch (change.getColumnType()) {
                case BYTES:
                    return change.getDefaultValue().getBytes();
                case FLOATING_POINT:
                    return Double.parseDouble(change.getDefaultValue());
                case INTEGER:
                    return Long.parseLong(change.getDefaultValue());
                default:
                    return change.getDefaultValue();
            }
        }
        if (change.isNullable()) {
            return null;
        }
        Object result = DEFAULT_VALUES.get(change.getColumnType());
        if (result == null) {
            throw new IllegalArgumentException("could not find default value for " + change.getColumnType());
        }
        return result;
    }

    public boolean shouldCreateTable() {
        return createTable;
    }

    public boolean shouldDropTable() {
        return dropTable;
    }

    public boolean shouldRenameTable() {
        return renameTable;
    }

    public boolean shouldRunMap() {
        boolean noColumnsModified = defaultValueByColumn.isEmpty() && typeConversionByColumn.isEmpty()
                && newNameByColumn.isEmpty() && columnsToDelete.isEmpty();
        return !(shouldCreateTable() || noColumnsModified);
    }

    public void handle(SchemaChange change, LocalDateTime eventTimestamp, boolean tableExists, String table) {
        if (change instanceof CreateTable) {
            handleCreateTable((CreateTable) change, eventTimestamp);
        } else if (change instanceof DropTable) {
            if (tableExists) {
                handleDropTable((DropTable) change);
            } else {
                logger.info("Skip DropTable because table {} doesn't exist", table);
            }
        } else if (change instanceof RenameTable) {
            if (tableExists) {
                handleRenameTable((RenameTable) change);
            } else {
                logger.info("Skip RenameTable because table {} doesn't exist", table);
            }
        } else if (change instanceof CreateOrModifyColumn) {
            if (tableExists) {
                handleCreateOrModifyColumn((CreateOrModifyColumn) change, eventTimestamp);
            } else {
                logger.info("Skip CreateOrModifyColumn because table {} doesn't exist", table);
            }
        } else if (change instanceof DropColumn) {
            if (tableExists) {
                handleDropColumn((DropColumn) change);
            } else {
                logger.info("Skip DropColumn because table {} doesn't exist", table);
            }
        } else if (change instanceof RenameColumn) {
            if (tableExists) {
                handleRenameColumn((RenameColumn) change);
            } else {
                logger.info("Skip RenameColumn because table {} doesn't exist", table);
            }
        } else if (change instanceof Truncate) {
            // DIRECT-82929
            logger.error("Got TRUNCATE event for table, but truncates are not implemented yet");
        } else {
            throw new IllegalArgumentException("can't handle schema change " + change);
        }
    }

    private void handleCreateTable(CreateTable change, LocalDateTime eventTimestamp) {
        validateTableChange();

        Set<String> primaryKey = new HashSet<>(change.getPrimaryKey());
        Map<String, CreateOrModifyColumn> columnMap = listToMap(change.getColumns(), CreateOrModifyColumn::getColumnName);
        List<CreateOrModifyColumn> keyColumns = mapList(change.getPrimaryKey(), columnMap::get);
        List<CreateOrModifyColumn> valueColumns = filterList(change.getColumns(), c -> !primaryKey.contains(c.getColumnName()));
        StreamEx<CreateOrModifyColumn> columns = StreamEx.of(keyColumns).append(valueColumns);

        for (CreateOrModifyColumn columnChange : columns) {
            createTable = handleCreateOrModifyColumn(columnChange, eventTimestamp,
                    primaryKey.contains(columnChange.getColumnName()))
                    || createTable;
        }
        CreateOrModifyColumn sourceColumn = new CreateOrModifyColumn()
                .withColumnName(YtReplicator.SOURCE_COLUMN_NAME)
                .withColumnType(ColumnType.STRING)
                .withNullable(true);
        createTable = handleCreateOrModifyColumn(sourceColumn, eventTimestamp, false) || createTable;
    }

    @SuppressWarnings("unused")
    private void handleDropTable(DropTable ignored) {
        validateTableChange();

        dropTable = true;
    }

    private void handleRenameTable(RenameTable change) {
        validateTableChange();

        renameTable = true;
        renames.addAll(change.getRenames());
    }

    /**
     * возвращает false если нет изменений в схеме
     */
    private boolean handleCreateOrModifyColumn(
            CreateOrModifyColumn change, LocalDateTime eventTimestamp, boolean isKeyColumn
    ) {
        validateColumnChange(change);
        schemaManager.handleCreateOrModifyColumn(change, isKeyColumn);
        String columnName = change.getColumnName();
        boolean columnWasRenamed = schemaManager.getNewSchema().hasColumn(columnName)
                && newNameByColumn.containsValue(columnName);
        boolean createColumn = !schemaManager.getOldSchema().hasColumn(columnName) && !columnWasRenamed;
        if (createColumn) {
            defaultValueByColumn.put(columnName, getDefaultValue(change, eventTimestamp));
        } else if (columnWasRenamed) {
            ColumnValueType oldType = schemaManager.getOldSchema()
                    .getColumn(oldNameByColumn.get(columnName))
                    .orElseThrow(IllegalStateException::new)
                    .getType();
            ColumnValueType newType = SchemaManager.getColumnSchemaType(change.getColumnType());
            defaultValueByColumn.put(columnName, getDefaultValue(change, eventTimestamp));
            if (!oldType.equals(newType)) {
                typeConversionByColumn.put(columnName, Pair.of(oldType, newType));
            } else {
                return false;
            }
        } else {
            ColumnValueType oldType = schemaManager.getOldSchema()
                    .getColumn(columnName)
                    .orElseThrow(IllegalStateException::new)
                    .getType();
            ColumnValueType newType = schemaManager.getNewSchema()
                    .getColumn(columnName)
                    .orElseThrow(IllegalStateException::new)
                    .getType();
            if (!oldType.equals(newType)) {
                typeConversionByColumn.put(columnName, Pair.of(oldType, newType));
            } else {
                return false;
            }
        }
        return true;
    }

    private void handleCreateOrModifyColumn(CreateOrModifyColumn change, LocalDateTime eventTimestamp) {
        handleCreateOrModifyColumn(change, eventTimestamp, false);
    }

    private void handleDropColumn(DropColumn change) {
        validateColumnChange(change);
        schemaManager.handleDropColumn(change);
        columnsToDelete.add(change.getColumnName());
    }

    private void handleRenameColumn(RenameColumn change) {
        validateColumnChange(change);
        schemaManager.handleRenameColumn(change);
        newNameByColumn.put(change.getOldColumnName(), change.getNewColumnName());
        oldNameByColumn.put(change.getNewColumnName(), change.getOldColumnName());
    }

    private void validateColumnChange(SchemaChange change) {
        String firstColumnName = null;
        String secondColumnName = null;
        if (change instanceof CreateOrModifyColumn) {
            firstColumnName = ((CreateOrModifyColumn) change).getColumnName();
        } else if (change instanceof DropColumn) {
            firstColumnName = ((DropColumn) change).getColumnName();
        } else if (change instanceof RenameColumn) {
            RenameColumn renameColumn = (RenameColumn) change;
            firstColumnName = renameColumn.getNewColumnName();
            secondColumnName = renameColumn.getOldColumnName();
        }

        for (String columnName : new String[]{firstColumnName, secondColumnName}) {
            if (columnName == null) {
                continue;
            }
            if (defaultValueByColumn.containsKey(columnName)) {
                throw new IllegalStateException("column " + columnName + " already scheduled for creation");
            }
            if (typeConversionByColumn.containsKey(columnName)) {
                throw new IllegalStateException("column " + columnName + " already scheduled for modification");
            }
            if (newNameByColumn.containsKey(columnName)) {
                throw new IllegalStateException("column " + columnName + " already scheduled for renaming");
            }
            if (columnsToDelete.contains(columnName)) {
                throw new IllegalStateException("column " + columnName + " already scheduled for dropping");
            }
        }
    }

    private void validateTableChange() {
        Preconditions.checkState(!shouldCreateTable(), "create table already scheduled");
        Preconditions.checkState(!shouldRenameTable(), "rename table already scheduled");
        Preconditions.checkState(!shouldDropTable(), "drop table already scheduled");
    }

    public AlterMapper alterMapper() {
        return new AlterMapper(
                new HashMap<>(defaultValueByColumn),
                new HashMap<>(typeConversionByColumn),
                new HashMap<>(newNameByColumn),
                new HashSet<>(columnsToDelete));
    }

    public List<RenameTable.RenameSingleTable> getRenames() {
        return renames;
    }
}
