package ru.yandex.direct.binlogbroker.replicatetoyt;

import java.math.BigInteger;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlog.model.RenameTable;
import ru.yandex.direct.binlog.model.SchemaChange;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.DataSize;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.operations.map.Mapper;
import ru.yandex.inside.yt.kosher.operations.specs.JobIo;
import ru.yandex.inside.yt.kosher.operations.specs.MapSpec;
import ru.yandex.inside.yt.kosher.operations.specs.SortSpec;
import ru.yandex.inside.yt.kosher.tables.TableWriterOptions;
import ru.yandex.inside.yt.kosher.ytree.YTreeListNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

@ParametersAreNonnullByDefault
public class AlterHandler {
    private static final Duration TRANSACTION_TIMEOUT = Duration.ofMinutes(1);
    private static final Duration DYNAMIC_TABLE_OPERATION_TIMEOUT = Duration.ofMinutes(5);
    private static final DataSize BLOCK_SIZE = DataSize.fromKiloBytes(256);
    private static final DataSize DESIRED_CHUNK_SIZE = DataSize.fromMegaBytes(100);
    private static final Logger logger = LoggerFactory.getLogger(AlterHandler.class);
    private static final Set<String> YT_TABLE_COLUMN_ATTRIBUTE_NAMES_TO_COMPARE =
            Set.of("name", "type", "required", "expression");
    private final Yt yt;
    private final YPath shardsDir;
    private final YPath tmpDir;
    private final String tablesSubDir;
    private final boolean ignoreTableNotExists;
    private static final BigInteger FULL_PIVOT_HASH_RANGE = BigInteger.valueOf(1).shiftLeft(64);
    public static final int TABLETS_PER_TABLE = 16;
    public static final long TABLET_RANGE =
            FULL_PIVOT_HASH_RANGE.divide(BigInteger.valueOf(TABLETS_PER_TABLE)).longValue();


    /**
     * @param yt        HTTP клиент YT
     * @param shardsDir корневая директория с шардами на YT
     * @param tmpDir    директория с временными таблицами на YT
     */
    public AlterHandler(Yt yt, YPath shardsDir, YPath tmpDir) {
        this(yt, shardsDir, tmpDir, null, false);
    }

    /**
     * @param yt           HTTP клиент YT
     * @param shardsDir    корневая директория с шардами на YT
     * @param tmpDir       директория с временными таблицами на YT
     * @param tablesSubDir директория таблицами на YT
     */
    public AlterHandler(Yt yt, YPath shardsDir, YPath tmpDir, @Nullable String tablesSubDir,
                        boolean ignoreTableNotExists) {
        this.yt = yt;
        this.shardsDir = shardsDir;
        this.tmpDir = tmpDir;
        this.tablesSubDir = tablesSubDir;
        this.ignoreTableNotExists = ignoreTableNotExists;
    }

    /**
     * Создание таблицы с указанной схемой и динамичностью. Таблица привязывается к главному мастеру,
     * чтобы её можно было сделать динамической
     */
    private Map<String, YTreeNode> makeCreateTableAttributes(YTreeNode schema, boolean isDynamic) {
        Map<String, YTreeNode> result = new HashMap<>();

        result.put(YtUtils.SCHEMA_ATTR, schema);
        result.put("compression_codec", YTree.stringNode("lz4"));
        // todo: set primary medium
        result.put("dynamic", YTree.booleanNode(isDynamic));
        result.put("enable_tablet_balancer", YTree.booleanNode(false));
        result.put("disable_tablet_balancer", YTree.booleanNode(true));
        return result;
    }

    private List<List<YTreeNode>> pivotKeysFromAttribute(YTreeNode attribute) {
        List<List<YTreeNode>> result = Cf.arrayList();
        for (YTreeNode node : attribute.asList()) {
            result.add(node.asList());
        }
        return result;
    }

    public List<YPath> handle(BinlogEvent event, StateManager.ShardOffsetSaver shardOffsetSaver)
            throws InterruptedException, TimeoutException {
        Preconditions.checkState(!event.getSchemaChanges().isEmpty(),
                "event does not have any schema changes");

        if (event.getTable().equals("test1")) {
            return new ArrayList<>();
        }

        YPath shardDir = shardsDir.child(event.getSource());
        YPath tablesDir = tablesSubDir != null ? shardDir.child(tablesSubDir) : shardDir;
        YPath tablePath = tablesDir.child(event.getTable());
        boolean tableExists = yt.cypress().exists(tablePath);

        SchemaManager schemaManager;
        if (tableExists) {
            schemaManager = new SchemaManager(yt.cypress().get(tablePath.attribute(YtUtils.SCHEMA_ATTR)).listNode());
        } else {
            schemaManager = new SchemaManager();
        }
        Alter alter = new Alter(schemaManager);
        logger.info("parsing schema changes");
        for (SchemaChange change : event.getSchemaChanges()) {
            alter.handle(change, event.getUtcTimestamp(), tableExists, event.getTable());
        }

        GUID txId = yt.transactions().start(TRANSACTION_TIMEOUT);
        Thread pingerThread = new Thread(new TransactionPinger(
                TRANSACTION_TIMEOUT.dividedBy(4).toMillis(),
                txId,
                false));
        pingerThread.setDaemon(true);
        pingerThread.start();

        List<YPath> tablesToMount = new ArrayList<>();
        try {

            if (alter.shouldCreateTable()) {
                doCreateTable(txId, tablePath, schemaManager, tableExists);
                tablesToMount.add(tablePath);
            } else if (alter.shouldDropTable()) {
                if (tableExists || !ignoreTableNotExists) {
                    doDropTable(txId, tablePath, tableExists);
                }
            } else if (alter.shouldRenameTable()) {
                if (tableExists || !ignoreTableNotExists) {
                    tablesToMount.addAll(doRenameTable(txId, shardDir, alter.getRenames()));
                }
            } else {
                if (tableExists || !ignoreTableNotExists) {
                    if (doAlter(txId, alter, schemaManager, tablePath)) {
                        tablesToMount.add(tablePath);
                    }
                }
            }
            shardOffsetSaver.save(txId);
        } catch (Exception e) {
            try {
                yt.transactions().abort(txId);
            } catch (Exception e2) {
                e.addSuppressed(e2);
            }
            throw e;
        } finally {
            logger.info("stopping pinger thread");
            pingerThread.interrupt();
            pingerThread.join(); // IS-NOT-COMPLETABLE-FUTURE-JOIN
        }

        Set<YPath> renamedTablePaths = alter.getRenames().stream()
                .map(RenameTable.RenameSingleTable::getNewTableName)
                .map(tablesDir::child)
                .collect(Collectors.toSet());

        logger.info("committing transaction {}", txId);
        yt.transactions().commit(txId);
        for (YPath path : tablesToMount) {
            logger.info("Resharding and mounting table {}", path);

            // TODO облагородить этот костыль,
            List<YTreeListNode> pivotKeys = hashModPartitions(YtReplicator.DEFAULT_PARTITIONS_COUNT);
            List<List<YTreeNode>> list = pivotKeys.stream().map(YTreeNode::asList).collect(Collectors.toList());

            // не решардируем переименованные таблицы
            if (!renamedTablePaths.contains(path)) {
                TabletUtils.syncReshardTable(yt, path, list);
            }

            // Монтирование таблицы с ожиданием всех таблетов
            TabletUtils.syncMountTable(yt, path, DYNAMIC_TABLE_OPERATION_TIMEOUT);
        }
        return tablesToMount;
    }

    /**
     * Генерирует PivotKeys для farm_hash % partitions
     * Например для partitions = 3 ответ будет [[], [1], [2]]
     * TODO : убрать дублирующийся код (одинаковый с PivotKeys::hashModPartitions)
     */
    private static List<YTreeListNode> hashModPartitions(int partitions) {
        List<YTreeListNode> keys = new ArrayList<>();
        keys.add(YTree.builder().beginList().buildList());
        for (int i = 1; i < partitions; ++i) {
            keys.add(YTree.builder().beginList().value(i).buildList());
        }
        return keys;
    }

    /**
     * Реализация создания таблицы. Опрерация выполняется под транзакцией.
     */
    private void doCreateTable(GUID txId, YPath path, SchemaManager schemaManager, boolean tableExists) {
        if (tableExists) {
            throw new IllegalStateException(String.format("table %s already exists", path));
        }
        YTreeNode schema = schemaManager.getNewSchema().getSortedSchema();
        logger.info("creating table {} with schema {}", path, schema);
        yt.cypress().create(
                Optional.of(txId),
                false,
                path,
                CypressNodeType.TABLE,
                true,
                false,
                makeCreateTableAttributes(schema, true));
    }

    /**
     * Реализация удаления таблицы. Операция выполняется под транзакцией.
     */
    private void doDropTable(GUID txId, YPath path, boolean tableExists) {
        if (!tableExists) {
            throw new IllegalStateException(String.format("table %s doesn't exist", path));
        }
        logger.info("removing table {}", path);
        yt.cypress().remove(Optional.of(txId), false, path);
    }

    /**
     * Реализация переименования таблиц. В начале проверяем, что переименование корректно,
     * после этого выполняем переименования в том порядке, в котором они приходят. Для
     * переноса динамические таблицы размонтируются. Сам перенос выполняется под транзакцией.
     *
     * @return список таблиц, которые нужно подмонтировать после коммита транзакции.
     */
    private Collection<YPath> doRenameTable(GUID txId, YPath shardDir, List<RenameTable.RenameSingleTable> renames)
            throws TimeoutException, InterruptedException {
        Set<YPath> addedTables = new HashSet<>();
        Set<YPath> removedTables = new HashSet<>();
        // Отдельно проверяем, что операцию можно совершить. Нужно, т.к. в процессе выполнения
        // надо будет отмонтировать таблицы, и делать это хочется только, если в альтере всё ок.
        YPath tablesDir = tablesSubDir != null ? shardDir.child(tablesSubDir) : shardDir;
        for (RenameTable.RenameSingleTable rename : renames) {
            YPath oldTablePath = tablesDir.child(rename.getOldTableName());
            YPath newTablePath = tablesDir.child(rename.getNewTableName());

            Preconditions.checkState(
                    addedTables.contains(oldTablePath) || yt.cypress().exists(oldTablePath),
                    "table %s does not exist", oldTablePath);
            Preconditions.checkState(
                    removedTables.contains(newTablePath) || !yt.cypress().exists(newTablePath),
                    "table %s already exists", newTablePath);

            removedTables.add(oldTablePath);
            addedTables.add(newTablePath);
            addedTables.remove(oldTablePath);
        }

        Set<YPath> finalTables = new HashSet<>();
        for (RenameTable.RenameSingleTable rename : renames) {
            YPath oldTablePath = tablesDir.child(rename.getOldTableName());
            YPath newTablePath = tablesDir.child(rename.getNewTableName());
            // если мы ещё не двигали эту таблицу, её надо отмонтировать
            if (!finalTables.contains(oldTablePath)) {
                TabletUtils.syncUnmountTable(yt, oldTablePath, DYNAMIC_TABLE_OPERATION_TIMEOUT);
            }
            logger.info("moving {} to {}", oldTablePath, newTablePath);
            yt.cypress().move(
                    Optional.of(txId),
                    false,
                    oldTablePath,
                    newTablePath,
                    false,
                    false,
                    false);
            finalTables.add(newTablePath);
            finalTables.remove(oldTablePath);
        }
        return finalTables;
    }

    private boolean doAlter(GUID txId, Alter alter, SchemaManager schemaManager, YPath tablePath)
            throws TimeoutException, InterruptedException {
        YPath tmpTablePath = tmpDir.child(tablePath.name() + UUID.randomUUID().toString());
        try {
            return doAlterWork(txId, alter, schemaManager, tablePath, tmpTablePath);
        } catch (Exception e) {
            logger.info("removing table {}", tmpTablePath);
            try {
                yt.cypress().remove(
                        Optional.empty(),
                        false,
                        tmpTablePath,
                        true,
                        true);
                if (TabletUtils.getTableState(yt, tablePath) == TabletUtils.TabletState.FROZEN) {
                    TabletUtils.syncUnmountTable(yt, tablePath, DYNAMIC_TABLE_OPERATION_TIMEOUT);
                }
            } catch (Exception e2) {
                logger.warn("could not remove {} because of {}", tmpTablePath, e2);
            }
            throw e;
        }
    }

    /**
     * Реализация альтера. Алгоритм следующий:
     * - Выводим из {@link BinlogEvent} схему новой таблицы и схему маппера.
     * - Замораживаем исходную таблицу и ждём заморозки таблетов.
     * - Затем запускаем map и sort на временных таблицах, указав выведенную ранее схему.
     * - После этого делаем временную таблицу динамической и решардируем её.
     * - В конце отмонтируем старую таблицу, подменяем её новой и подмонтируем новую.
     * <p>
     * Почти весь альтер, кроме финальной подмены таблицы делается без транзакции. Это
     * нужно так как с создаваемыми таблицами нужно работать и как с динамическими, что
     * сделать до коммита транзакции не получится.
     */
    private boolean doAlterWork(
            GUID txId, Alter alter, SchemaManager schemaManager, YPath tablePath, YPath tmpTablePath
    ) throws TimeoutException, InterruptedException {
        if (!alter.shouldRunMap()) {
            YTreeNode realSchema = yt.cypress().get(tablePath.attribute(YtUtils.SCHEMA_ATTR));
            Set<MapF<String, YTreeNode>> realSchemaFiltered = realSchema.listNode().asList().stream()
                    .map(column -> Cf.wrap(column.getAttributes())
                            .filterKeys(YT_TABLE_COLUMN_ATTRIBUTE_NAMES_TO_COMPARE::contains))
                    .collect(Collectors.toSet());
            YTreeNode expectedSchema = schemaManager.getNewSchema().getSortedSchema();
            Set<MapF<String, YTreeNode>> expectedSchemaFiltered = expectedSchema.listNode().asList().stream()
                    .map(column -> Cf.wrap(column.getAttributes())
                            .filterKeys(YT_TABLE_COLUMN_ATTRIBUTE_NAMES_TO_COMPARE::contains))
                    .collect(Collectors.toSet());
            if (!expectedSchemaFiltered.equals(realSchemaFiltered)) {
                throw new IllegalStateException(String.format(
                        "alter should not run map, but there is mismatch between real schema: %s and altered schema %s",
                        realSchema,
                        expectedSchema));
            }
            logger.info("there is no effective change in the table schema");
            return false;
        }

        TabletUtils.TabletState currentTableState = TabletUtils.getTableState(yt, tablePath);
        switch (currentTableState) {
            case MOUNTED:
                TabletUtils.syncFreezeTable(yt, tablePath, DYNAMIC_TABLE_OPERATION_TIMEOUT);
                break;
            case FROZEN:
                TabletUtils.waitForTablets(yt, tablePath, TabletUtils.TabletState.FROZEN,
                        DYNAMIC_TABLE_OPERATION_TIMEOUT);
                break;
            case UNMOUNTED:
                TabletUtils.waitForTablets(yt, tablePath, TabletUtils.TabletState.UNMOUNTED,
                        DYNAMIC_TABLE_OPERATION_TIMEOUT);
                break;
            default:
                throw new IllegalStateException(String.format("can not handle state %s", currentTableState));
        }


        YTreeNode unsortedSchema = schemaManager.getNewSchema().getUnsortedSchema();
        logger.info("creating table {} with schema {}", tmpTablePath, unsortedSchema);
        yt.cypress().create(
                tmpTablePath,
                CypressNodeType.TABLE,
                makeCreateTableAttributes(unsortedSchema, false));

        Mapper<YTreeMapNode, YTreeMapNode> mapper = alter.alterMapper();
        JobIo jobIo = new JobIo(new TableWriterOptions()
                .withBlockSize(BLOCK_SIZE)
                .withDesiredChunkSize(DESIRED_CHUNK_SIZE));

        MapSpec mapSpec = MapSpec.builder()
          .setInputTables(tablePath)
          .setOutputTables(tmpTablePath)
          .setMapper(mapper)
          .setJobIo(jobIo)
          .build();
        logger.info("running alter map: mapper {}, input {}, output {}", mapper, tablePath, tmpTablePath);
        yt.operations().mapAndGetOp(mapSpec).awaitAndThrowIfNotSuccess();

        ListF<String> keyColumnNames = Cf.toList(schemaManager.getNewSchema().getKeyColumnNames());
        if (keyColumnNames.isNotEmpty()) {
            logger.info("running alter sort with key columns {}", keyColumnNames);
            YPath tmpSortedTablePath = tmpTablePath.parent().child(tmpTablePath.name() + "_sorted");
            YTreeNode sortedSchema = schemaManager.getNewSchema().getSortedSchema();
            SortSpec sortSpec = SortSpec.builder()
                    .setInputTables(Cf.list(tmpTablePath))
                    .setOutputTable(tmpSortedTablePath)
                    .setSortBy(keyColumnNames)
                    .setMergeJobIo(jobIo)
                    .build();
            try {
                logger.info("creating table {} with schema {}", tmpSortedTablePath, sortedSchema);
                yt.cypress().create(
                        tmpSortedTablePath,
                        CypressNodeType.TABLE,
                        makeCreateTableAttributes(sortedSchema, false));
                yt.operations().sortAndGetOp(sortSpec).awaitAndThrowIfNotSuccess();
                yt.cypress().move(tmpSortedTablePath, tmpTablePath, true);
            } finally {
                logger.info("removing table {}", tmpSortedTablePath);
                yt.cypress().remove(
                        Optional.empty(), false, tmpSortedTablePath, true, true);
            }
        } else {
            logger.warn("table {} will be ordered dynamic instead of sorted dynamic", tablePath);
        }

        logger.info("making table {} dynamic", tmpTablePath);
        yt.tables().alterTable(tmpTablePath, Optional.of(true), Optional.empty());
        YTreeNode pivotKeysAttribute = yt.cypress().get(tablePath.attribute("pivot_keys"));
        if (!pivotKeysAttribute.isEntityNode()) {
            logger.info("resharding table {}", tmpTablePath);
            TabletUtils.syncReshardTable(yt, tmpTablePath, pivotKeysFromAttribute(pivotKeysAttribute));
        }
        TabletUtils.syncUnmountTable(yt, tablePath, DYNAMIC_TABLE_OPERATION_TIMEOUT);
        // К сожалению, подвинуть динамическую таблицу с подмонтированными таблетами нельзя
        logger.info("replacing table {} with {}", tablePath, tmpTablePath);
        yt.cypress().move(
                Optional.of(txId),
                false,
                tmpTablePath,
                tablePath,
                false,
                true,
                false);
        logger.info("alter completed");
        return true;
    }

    /**
     * Класс, который в отдельном треде пингует транзакцию. Нужен, т.к. есть тяжелые
     * операции которые нельзя выполнять под транзакцией - например, сам
     * альтер таблицы.
     */
    class TransactionPinger implements Runnable {
        private final long pingTimeout;
        private final GUID txID;
        private final boolean pingAncestorTransactions;

        /**
         * @param pingTimeout              время в миллисекундах через которое нужно пинговать транзакцию
         * @param txID                     id транзакции
         * @param pingAncestorTransactions нужно ли пинговать родительские транзакции
         */
        TransactionPinger(long pingTimeout, GUID txID, boolean pingAncestorTransactions) {
            this.pingTimeout = pingTimeout;
            this.txID = txID;
            this.pingAncestorTransactions = pingAncestorTransactions;
        }

        @Override
        public void run() {
            while (!Thread.currentThread().isInterrupted()) {
                try {
                    yt.transactions().ping(txID, pingAncestorTransactions);
                } catch (Exception e) {
                    logger.warn(String.format("pinging of tx %s failed with %s", txID, e));
                }

                try {
                    Thread.sleep(pingTimeout);
                } catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    logger.warn("pinger thread is interrupted");
                }
            }
        }
    }
}
