package ru.yandex.direct.binlogbroker.replicatetoyt;

import java.util.HashMap;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.inside.yt.kosher.impl.operations.utils.YtSerializable;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.operations.Statistics;
import ru.yandex.inside.yt.kosher.operations.Yield;
import ru.yandex.inside.yt.kosher.operations.map.Mapper;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.tables.ColumnValueType;

import static java.util.function.Function.identity;

@ParametersAreNonnullByDefault
public class AlterMapper implements Mapper<YTreeMapNode, YTreeMapNode>, YtSerializable {
    private final Map<String, Object> defaultValueByColumn;
    private final Map<String, Pair<ColumnValueType, ColumnValueType>> typeConversionByColumn;
    private final Map<String, String> newNameByColumn;
    private final Set<String> columnsToDelete;
    private final Set<String> newColumnNames;
    private transient Map<String, Function<YTreeNode, YTreeNode>> modifyFnByColumn = null;

    private static final Map<Pair<ColumnValueType, ColumnValueType>, Function<YTreeNode, YTreeNode>>
            VALUE_CONVERSION_MAP =
            ImmutableMap.<Pair<ColumnValueType, ColumnValueType>, Function<YTreeNode,
                    YTreeNode>>builder()
                    .put(Pair.of(
                            ColumnValueType.STRING, ColumnValueType.DOUBLE),
                            e -> YTree.doubleNode(Double.parseDouble(e.stringValue())))
                    .put(Pair.of(
                            ColumnValueType.STRING, ColumnValueType.INT64),
                            e -> YTree.integerNode(Long.parseLong(e.stringValue())))
                    .put(Pair.of(
                            ColumnValueType.INT64, ColumnValueType.STRING),
                            e -> YTree.stringNode(String.valueOf(e.longValue())))
                    .put(Pair.of(
                            ColumnValueType.INT64, ColumnValueType.DOUBLE),
                            e -> YTree.doubleNode((double) e.longValue()))
                    .put(Pair.of(
                            ColumnValueType.DOUBLE, ColumnValueType.STRING),
                            e -> YTree.stringNode(String.valueOf(e.doubleValue())))
                    .put(Pair.of(
                            ColumnValueType.DOUBLE, ColumnValueType.INT64),
                            e -> YTree.integerNode((long) e.doubleValue()))
                    .build();

    public AlterMapper(
            Map<String, Object> defaultValueByColumn,
            Map<String, Pair<ColumnValueType, ColumnValueType>> typeConversionByColumn,
            Map<String, String> newNameByColumn,
            Set<String> columnsToDelete) {
        this.defaultValueByColumn = defaultValueByColumn;
        this.newColumnNames = ImmutableSet.copyOf(newNameByColumn.values());
        this.typeConversionByColumn = typeConversionByColumn;
        this.newNameByColumn = newNameByColumn;
        this.columnsToDelete = columnsToDelete;
    }

    private Function<YTreeNode, YTreeNode> getConvertionFn(ColumnValueType before, ColumnValueType after) {
        Function<YTreeNode, YTreeNode> result = VALUE_CONVERSION_MAP.get(Pair.of(before, after));
        if (result == null) {
            throw new IllegalArgumentException("no conversion function from " + before + " to " + after + " found");
        }
        return result;
    }

    @Override
    public void start(Yield<YTreeMapNode> yield, Statistics statistics) {
        Preconditions.checkState(modifyFnByColumn == null, "modify column map is already initialized");
        // это можно было бы делать не на ЫТе, но у сериализатора проблемы с функциями в хешмапе
        Map<String, Function<YTreeNode, YTreeNode>> map = new HashMap<>();

        for (Map.Entry<String, Pair<ColumnValueType, ColumnValueType>> entry : typeConversionByColumn.entrySet()) {
            map.put(
                    entry.getKey(),
                    getConvertionFn(entry.getValue().getLeft(), entry.getValue().getRight()));
        }
        modifyFnByColumn = map;
    }

    @Override
    public void map(YTreeMapNode entry, Yield<YTreeMapNode> yield, Statistics statistics) {
        Preconditions.checkState(modifyFnByColumn != null, "modify column map is not initialized");

        for (String name : columnsToDelete) {
            entry.remove(name);
        }
        for (Map.Entry<String, String> renameEntry : newNameByColumn.entrySet()) {
            YTreeNode field = entry
                    .remove(renameEntry.getKey())
                    .orElseThrow(() -> new NoSuchElementException("field " + renameEntry.getKey() + " is missing"));
            entry.put(renameEntry.getValue(), field);
        }
        for (Map.Entry<String, Object> addEntry : defaultValueByColumn.entrySet()) {
            Object value = addEntry.getValue();
            YTreeNode defaultValue = value != null ?
                    YTree.builder().value(value).build() :
                    YTree.builder().entity().build();
            String columnName = addEntry.getKey();
            if (newColumnNames.contains(columnName)) {
                YTreeNode currentValue = entry
                        .remove(columnName)
                        .orElseThrow(() -> new NoSuchElementException("field " + columnName + " is missing"));
                boolean currentValueIsNull = currentValue.isEntityNode();
                Function<YTreeNode, YTreeNode> typeConversionFunction =
                        modifyFnByColumn.getOrDefault(columnName, identity());
                entry.put(columnName, currentValueIsNull ? defaultValue : typeConversionFunction.apply(currentValue));
            } else {
                entry.put(columnName, defaultValue);
            }
        }
        for (Map.Entry<String, Function<YTreeNode, YTreeNode>> modifyEntry : modifyFnByColumn.entrySet()) {
            String columnName = modifyEntry.getKey();
            // для переименованных колонок с дефолтным значение конвертация типа уже проведена
            if (newColumnNames.contains(columnName) && defaultValueByColumn.containsKey(columnName)) {
                continue;
            }
            YTreeNode field = entry
                    .remove(columnName)
                    .orElseThrow(() -> new NoSuchElementException("field " + columnName + " is missing"));
            entry.put(columnName, modifyEntry.getValue().apply(field));
        }

        yield.yield(entry);
    }

    @Override
    public String toString() {
        return "AlterMapper{" +
                "defaultValueByColumn=" + defaultValueByColumn +
                ", typeConversionByColumn=" + typeConversionByColumn +
                ", newNameByColumn=" + newNameByColumn +
                ", columnsToDelete=" + columnsToDelete +
                '}';
    }
}
