package ru.yandex.direct.binlogbroker.replicatetoyt;

import java.time.Duration;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.TimeoutException;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.yt.ytclient.rpc.RpcError;

/**
 * Имплементация {@link YtReplicator}, которая повторяет через время вызовы
 * {@link YtReplicator#acceptDML(List, StateManager.ShardOffsetSaver)},
 * если он кидает {@link TimeoutException} или {@link RpcError}.
 * Ожидание перед повторением каждый раз удваивается, начиная с FIRST_TIMEOUT и заканчивая MAX_RETRY_TIMEOUT.
 * Если вызов кидает другие ошибки или ни один повтор не завершился удачно, ошибка прокидывается наверх.
 */
@ParametersAreNonnullByDefault
public class RetryingYtReplicator implements YtReplicator {
    private static final Logger logger = LoggerFactory.getLogger(RetryingYtReplicator.class);
    // значения таймаутов подобраны опытным путем, т.е. взяты почти с потолка.
    private static final long FIRST_TIMEOUT = Duration.ofSeconds(8).toMillis();
    private static final long MAX_RETRY_TIMEOUT = Duration.ofSeconds(512).toMillis();
    private final YtReplicator ytReplicator;

    public RetryingYtReplicator(YtReplicator ytReplicator) {
        Preconditions.checkState(
                !(ytReplicator instanceof RetryingYtReplicator),
                "trying to put RetryingYtReplicator inside of RetryingYtReplicator");
        this.ytReplicator = ytReplicator;
    }

    @Override
    public void acceptDDL(BinlogEvent binlogEvent, StateManager.ShardOffsetSaver shardOffsetSaver)
            throws InterruptedException {
        // Применение DDL - дорогостоящая и сложная операция. Повторная попытка сделать то же самое в случае багов
        // может сильно испортить базу.
        ytReplicator.acceptDDL(binlogEvent, shardOffsetSaver);
    }

    @Override
    public void acceptDML(List<BinlogEvent> binlogEvents, StateManager.ShardOffsetSaver shardOffsetSaver)
            throws InterruptedException {
        long timeout = FIRST_TIMEOUT;
        do {
            try {
                ytReplicator.acceptDML(binlogEvents, shardOffsetSaver);
                return;
            } catch (Exception e) {
                // Для идентификации пачки событий используем мапу tableName -> max(queryIndex)
                final HashMap<String, Integer> tablesQueries = new HashMap<>();
                for (BinlogEvent event : binlogEvents) {
                    final String table = event.getTable();
                    final int queryIndex = event.getQueryIndex();
                    tablesQueries.compute(table, (k, v) -> v == null ? queryIndex : Math.max(v, queryIndex));
                }
                if (!mayRetry(e) || timeout > MAX_RETRY_TIMEOUT) {
                    if (timeout > MAX_RETRY_TIMEOUT) {
                        logger.error("Operation {} have not completed normally after several delayed retries",
                                tablesQueries);
                    }
                    throw e;
                }
                logger.warn("Operation {} completed with error, but may be recovered in {} milliseconds",
                        tablesQueries, timeout, e);
                Thread.sleep(timeout);
            }
            timeout = timeout * 2;
        } while (!Thread.currentThread().isInterrupted());
    }

    /**
     * Если в дереве ошибок есть RpcError или TimeoutException, считаем что операцию можно повторить через время.
     * Если после какой то ошибки повторять не стоило, то через N неудачных попыток эта ошибка будет прокинута выше.
     */
    private boolean mayRetry(Throwable e) {
        final Deque<Throwable> stack = new ArrayDeque<>();
        stack.push(e);
        while (!stack.isEmpty()) {
            Throwable t = stack.pop();
            if (t instanceof RpcError || t instanceof TimeoutException) {
                return true;
            }
            if (t.getCause() != null) {
                stack.push(t.getCause());
            }
            for (Throwable s : t.getSuppressed()) {
                stack.push(s);
            }
        }
        return false;
    }
}
