package ru.yandex.direct.binlogbroker.replicatetoyt;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.binlog.model.ColumnType;
import ru.yandex.direct.binlog.model.CreateOrModifyColumn;
import ru.yandex.direct.binlog.model.DropColumn;
import ru.yandex.direct.binlog.model.RenameColumn;
import ru.yandex.inside.yt.kosher.ytree.YTreeListNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.tables.ColumnSchema;
import ru.yandex.yt.ytclient.tables.ColumnSortOrder;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.direct.binlogbroker.replicatetoyt.YtReplicator.HASH_COLUMN_NAME;
import static ru.yandex.direct.binlogbroker.replicatetoyt.YtReplicator.SOURCE_COLUMN_NAME;

@ParametersAreNonnullByDefault
public class SchemaManager {
    private static final Map<ColumnType, ColumnValueType> SCHEMA_CONVERSION_MAP =
            ImmutableMap.<ColumnType, ColumnValueType>builder()
                    .put(ColumnType.BYTES, ColumnValueType.STRING)
                    .put(ColumnType.DATE, ColumnValueType.STRING)
                    .put(ColumnType.FIXED_POINT, ColumnValueType.STRING)
                    .put(ColumnType.FLOATING_POINT, ColumnValueType.DOUBLE)
                    .put(ColumnType.INTEGER, ColumnValueType.INT64)
                    .put(ColumnType.STRING, ColumnValueType.STRING)
                    .put(ColumnType.TIMESTAMP, ColumnValueType.STRING)
                    .put(ColumnType.UNSIGNED_BIGINT, ColumnValueType.UINT64)
                    .build();


    private final Schema oldSchema;
    private final Schema newSchema;

    public SchemaManager(YTreeListNode inputSchema) {
        oldSchema = new Schema(inputSchema);
        newSchema = new Schema(inputSchema);
    }

    public SchemaManager() {
        oldSchema = new Schema(true, true);
        newSchema = new Schema(true, true);
    }

    public Schema getOldSchema() {
        return oldSchema;
    }

    public Schema getNewSchema() {
        return newSchema;
    }

    public void handleCreateOrModifyColumn(CreateOrModifyColumn change, boolean isKeyColumn) {
        ColumnSchema columnSchema =
                new ColumnSchema.Builder(change.getColumnName(), getColumnSchemaType(change.getColumnType()))
                        .setSortOrder(isKeyColumn ? ColumnSortOrder.ASCENDING : null)
                        .build();
        newSchema.addOrModifyColumn(columnSchema);
    }

    public void handleDropColumn(DropColumn change) {
        newSchema.removeColumn(change.getColumnName());
    }

    public void handleRenameColumn(RenameColumn change) {
        newSchema.renameColumn(change.getOldColumnName(), change.getNewColumnName());
    }


    public static ColumnValueType getColumnSchemaType(ColumnType type) {
        ColumnValueType result = SCHEMA_CONVERSION_MAP.get(type);
        if (result == null) {
            throw new IllegalArgumentException("type conversion is unknown for " + type);
        }
        return result;
    }

    /**
     * Генерирует выражение для вычисляемой колонки __hash__
     * Используем farm_hash с делением по модулю, чтобы можно было делать эффективные запросы по диапазонам
     * (where PK >= x and PK < y and __hash__ in (0, 1, 2, ...))
     * Это важно для вычисления контрольных сумм данных
     */
    public static String getHashColumnExpression(List<String> keyColumnNames, int partitions) {
        return String.format("int64(farm_hash(%s) %% %d)", keyColumnNames.get(0), partitions);
    }

    class Schema {
        private final Map<String, ColumnSchema> keyColumns = new HashMap<>();
        private final Map<String, ColumnSchema> valueColumns = new HashMap<>();
        // Для этих переменных надо проверять вхождение элементов, это можно делать за O(1)
        // вместо O(n). Но сабж используется редко, и размеры списков небольшие.
        private final List<String> keyColumnNames = new ArrayList<>();
        private final List<String> valueColumnNames = new ArrayList<>();

        private final boolean strict;
        private final boolean uniqueKeys;

        Schema(boolean strict, boolean uniqueKeys) {
            this.strict = strict;
            this.uniqueKeys = uniqueKeys;
        }

        Schema(YTreeListNode inputSchema) {
            strict = inputSchema.getAttributeOrThrow("strict").boolValue();
            uniqueKeys = inputSchema.getAttributeOrThrow("unique_keys").boolValue();
            for (YTreeNode node : inputSchema) {
                ColumnSchema columnSchema = ColumnSchema.fromYTree(node);
                addColumn(columnSchema);
            }
        }

        public List<String> getKeyColumnNames() {
            return keyColumnNames;
        }

        private void addColumn(ColumnSchema columnSchema) {
            String columnName = columnSchema.getName();
            if (keyColumnNames.contains(columnName) || valueColumnNames.contains(columnName)) {
                throw new IllegalArgumentException(columnName + " already exists in the table schema");
            }
            if (columnSchema.getSortOrder() == null) {
                valueColumns.put(columnName, columnSchema);
                valueColumnNames.add(columnName);
                // Колонку __source__ всегда держим последней
                if (valueColumns.containsKey(SOURCE_COLUMN_NAME)) {
                    valueColumnNames.remove(SOURCE_COLUMN_NAME);
                    valueColumnNames.add(SOURCE_COLUMN_NAME);
                }
            } else {
                keyColumns.put(columnName, columnSchema);
                keyColumnNames.add(columnName);
            }
        }

        private void onKeyColumnListChange() {
            // rebuild __hash__ expression on key column set change
            if (keyColumnNames.contains(HASH_COLUMN_NAME)) {
                keyColumnNames.remove(HASH_COLUMN_NAME);
                keyColumns.remove(HASH_COLUMN_NAME);
            }
            if (!keyColumns.isEmpty()) {
                keyColumns.put(HASH_COLUMN_NAME, new ColumnSchema.Builder(HASH_COLUMN_NAME, ColumnValueType.INT64)
                        .setExpression(getHashColumnExpression(keyColumnNames, YtReplicator.DEFAULT_PARTITIONS_COUNT))
                        .setSortOrder(ColumnSortOrder.ASCENDING)
                        .build());
                // __hash__ should always go first
                keyColumnNames.add(0, HASH_COLUMN_NAME);
            }
        }

        private void modifyColumn(ColumnSchema columnSchema) {
            String columnName = columnSchema.getName();
            if (columnSchema.getSortOrder() == null) {
                valueColumns.put(columnName, columnSchema);
            } else {
                keyColumns.put(columnName, columnSchema);
            }
        }

        private void removeColumn(String columnName) {
            if (keyColumns.containsKey(columnName)) {
                keyColumns.remove(columnName);
                keyColumnNames.remove(columnName);
                onKeyColumnListChange();
            } else if (valueColumns.containsKey(columnName)) {
                valueColumns.remove(columnName);
                valueColumnNames.remove(columnName);
            } else {
                throw new IllegalArgumentException(
                        "Trying to delete column " + columnName + " which is not present in the table schema.");
            }
        }

        private void addOrModifyColumn(ColumnSchema columnSchema) {
            if (hasColumn(columnSchema.getName())) {
                modifyColumn(columnSchema);
            } else {
                addColumn(columnSchema);
                if (columnSchema.getSortOrder() != null) {
                    onKeyColumnListChange();
                }
            }
        }

        private void renameColumn(String oldName, String newName) {
            if (!hasColumn(oldName)) {
                throw new IllegalArgumentException("column " + oldName + " not present in the table schema");
            }
            if (hasColumn(newName)) {
                throw new IllegalArgumentException("column " + newName + " already present in the table schema");
            }
            ColumnSchema columnSchema = getColumn(oldName).orElseThrow(IllegalStateException::new);
            removeColumn(oldName);
            ColumnSchema newColumn = columnToBuilder(columnSchema, newName, columnSchema.getType()).build();
            addColumn(newColumn);
            if (newColumn.getSortOrder() != null) {
                onKeyColumnListChange();
            }
        }

        boolean hasColumn(String columnName) {
            return keyColumnNames.contains(columnName) || valueColumnNames.contains(columnName);
        }

        Optional<ColumnSchema> getColumn(String columnName) {
            if (keyColumnNames.contains(columnName)) {
                return Optional.of(keyColumns.get(columnName));
            }
            if (valueColumnNames.contains(columnName)) {
                return Optional.of(valueColumns.get(columnName));
            }
            return Optional.empty();
        }

        private ColumnSchema.Builder columnToBuilder(ColumnSchema columnSchema, String name, ColumnValueType type) {
            return new ColumnSchema.Builder(name, type, columnSchema.isRequired())
                    .setAggregate(columnSchema.getAggregate())
                    .setExpression(columnSchema.getExpression())
                    .setGroup(columnSchema.getGroup())
                    .setLock(columnSchema.getLock())
                    .setSortOrder(columnSchema.getSortOrder());
        }

        private ColumnSchema.Builder columnToBuilder(ColumnSchema columnSchema) {
            return columnToBuilder(columnSchema, columnSchema.getName(), columnSchema.getType());
        }

        public YTreeNode getUnsortedSchema() {
            TableSchema.Builder builder = new TableSchema.Builder()
                    .setStrict(strict)
                    .setUniqueKeys(false);
            for (String name : keyColumnNames) {
                ColumnSchema columnSchema = keyColumns.get(name);
                builder.add(columnToBuilder(columnSchema).setSortOrder(null).build());
            }
            for (String name : valueColumnNames) {
                builder.add(valueColumns.get(name));
            }
            return builder.build().toYTree();
        }

        public YTreeNode getSortedSchema() {
            TableSchema.Builder builder = new TableSchema.Builder()
                    .setStrict(strict)
                    .setUniqueKeys(uniqueKeys);
            // TODO найти правильное место для костыля
            if (!keyColumnNames.contains(HASH_COLUMN_NAME)) {
                builder.add(new ColumnSchema.Builder(HASH_COLUMN_NAME, ColumnValueType.INT64)
                        // По умолчанию для новых таблиц, создаваемых через CREATE TABLE,
                        // устанавливаем кол-во таблетов = 32, так как нет возможности заранее узнать,
                        // сколько партиций может понадобиться. Может быть, таблица совсем новая, и тогда
                        // окажется, что достаточно 1 партиции, а может быть, таблица создана в процессе
                        // тяжёлого альтера через pt-osc, и сейчас туда будут переложены все данные из основной таблицы
                        // Возможно, в будущем мы сможем такие случаи в фоновом режиме оптимизировать
                        .setExpression(getHashColumnExpression(keyColumnNames, YtReplicator.DEFAULT_PARTITIONS_COUNT))
                        .setSortOrder(ColumnSortOrder.ASCENDING)
                        .build());
            }
            for (String name : keyColumnNames) {
                builder.add(keyColumns.get(name));
            }
            for (String name : valueColumnNames) {
                builder.add(valueColumns.get(name));
            }
            return builder.build().toYTree();
        }
    }
}
