package ru.yandex.direct.binlogbroker.replicatetoyt;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.Iterables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.inside.yt.kosher.impl.common.YtException;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.ModifyRowsRequest;
import ru.yandex.yt.ytclient.proxy.YtClient;
import ru.yandex.yt.ytclient.tables.TableSchema;

import static ru.yandex.direct.binlogbroker.replicatetoyt.YtReplicator.SOURCE_COLUMN_NAME;

/**
 * Пишет данные из {@link BinlogEvent} в заранее указанную таблицу в YT
 * TODO DIRECT-82694: Корректно обрабатывать изменение primary key
 */
@ParametersAreNonnullByDefault
class TableDataYtReplicator {

    private static final Logger logger = LoggerFactory.getLogger(TableDataYtReplicator.class);

    private final YtClient ytClient;
    private final String tableNode;
    private final boolean skipErroneousEvents;
    private volatile TableSchema tableSchema;

    TableDataYtReplicator(YtClient ytClient, String tableNode, boolean skipErroneousEvents) {
        this.ytClient = ytClient;
        this.tableNode = tableNode;
        this.skipErroneousEvents = skipErroneousEvents;
        updateTableSchema();
    }

    void processEventBatch(List<BinlogEvent> binlogEvents,
                           ApiServiceTransactionOptions transactionOptions) {
        try (ApiServiceTransaction transaction =
                     ytClient.startTransaction(transactionOptions).join()) { // IGNORE-BAD-JOIN DIRECT-149116
            processEventBatch(transaction, binlogEvents)
                    .thenCompose(x -> transaction.commit())
                    .join(); // IGNORE-BAD-JOIN DIRECT-149116
        }
    }

    CompletableFuture<Void> processEventBatch(ApiServiceTransaction transaction, List<BinlogEvent> binlogEvents) {
        logger.debug("Handling {} events for {}", binlogEvents.size(), tableNode);
        int queryIndex = -1;
        CompletableFuture[] futures = new CompletableFuture[binlogEvents.size()];
        int i = 0;
        for (BinlogEvent event : binlogEvents) {
            queryIndex = Math.max(queryIndex, event.getQueryIndex());
            if (event.getOperation() == Operation.UPDATE && updatesPrimaryKey(event)) {
                futures[i] = processPrimaryKeyUpdate(transaction, event);
            } else {
                futures[i] = processEvent(transaction, tableNode, event);
            }
            i++;
        }
        final int logQueryIndex = queryIndex;
        return CompletableFuture.allOf(futures)
                .exceptionally(exc -> {
                    throw new IllegalStateException(String.format("While handling batch of %d events for path %s",
                            binlogEvents.size(), tableNode), exc);
                })
                .thenAccept(x -> logger.debug("Completed query #{} for table {}", logQueryIndex, tableNode));
    }

    private CompletableFuture<Void> processEvent(
            ApiServiceTransaction transaction, String tableNode, BinlogEvent event
    ) {
        ModifyRowsRequest request = new ModifyRowsRequest(tableNode, tableSchema);
        final String eventSource = event.getSource();
        try {
            switch (event.getOperation()) {
                case INSERT:
                    for (BinlogEvent.Row row : event.getRows()) {
                        request.addInsert(convertRowToYt(eventSource, row));
                    }
                    break;
                case DELETE:
                    for (BinlogEvent.Row row : event.getRows()) {
                        request.addDelete(convertRowToYt(eventSource, row));
                    }
                    break;
                case UPDATE:
                    for (BinlogEvent.Row row : event.getRows()) {
                        // здесь имеем гарантию что row.getPrimaryKey() и row.getAfter() не пересекаются по ключам
                        // assert получается слишком длинный
                        request.addUpdate(convertRowToYt(eventSource, row));
                    }
                    break;
                default:
                    logger.warn("Operation {} is not supported", event.getOperation());
            }
            return transaction.modifyRows(request).exceptionally(exc -> {
                throw new IllegalStateException("While handling " + event, exc);
            });
        } catch (YtException exc) {
            throw exc;
        } catch (RuntimeException exc) {
            String message = "While handling " + event;
            if (skipErroneousEvents) {
                logger.warn("{}", event, exc);
                return CompletableFuture.completedFuture(null);
            } else {
                throw new IllegalStateException(message, exc);
            }
        }
    }

    private static CompletableFuture<Void> processPrimaryKeyUpdate(
            ApiServiceTransaction transaction,
            BinlogEvent event
    ) {
        logger.warn("Event {} modifies primary key, this update is not supported yet", event);
        return CompletableFuture.completedFuture(null);
    }

    /**
     * возвращает true, если event изменяет значение primary key какой-либо строки
     */
    private static boolean updatesPrimaryKey(BinlogEvent event) {
        for (BinlogEvent.Row row : event.getRows()) {
            var after = row.getAfter();

            var updatedPrimaryKey = row.getPrimaryKey().keySet().stream()
                    .filter(after::containsKey)
                    .findAny();

            if (updatedPrimaryKey.isPresent()) {
                return true;
            }
        }
        return false;
    }

    void updateTableSchema() {
        this.tableSchema = TableSchema.fromYTree(ytClient.getNode(tableNode + "/@schema")
                .join()).toWrite(); // IGNORE-BAD-JOIN DIRECT-149116
    }

    Map<String, Object> convertRowToYt(String eventSource, BinlogEvent.Row row) {
        // используем оптимистичный подход:
        // проверяем только наличие соответствующей колонки в таблице YT для каждого значения
        // проверку соответствия типов (значения и колонки) оставляем на усмотрение YT
        Map<String, Object> result = new HashMap<>();
        for (Map.Entry<String, Object> me : Iterables.concat(row.getAfter().entrySet(),
                row.getPrimaryKey().entrySet())) {
            final String columnName = me.getKey();
            Preconditions.checkState(tableSchema.findColumn(columnName) >= 0,
                    "Column " + columnName + " is missing in table " + tableNode);
            result.put(columnName, convertValueToYt(me.getValue()));
        }
        result.put(SOURCE_COLUMN_NAME, eventSource);
        return result;
    }

    @Nullable
    private Object convertValueToYt(@Nullable Object value) {
        if (value instanceof BigDecimal || value instanceof BigInteger) {
            return value.toString();
        } else if (value == null
                || value instanceof String || value instanceof byte[]
                || value instanceof Number) {
            return value;
        } else if (value instanceof LocalDate) {
            return ((LocalDate) value).format(DateTimeFormatter.ISO_LOCAL_DATE);
        } else if (value instanceof LocalDateTime) {
            return ((LocalDateTime) value).format(DateTimeFormatter.ISO_LOCAL_DATE_TIME);
        }
        throw new IllegalArgumentException(
                "The value " + value + " (type: " + value.getClass().getName() + ") can not be stored in YT table");
    }
}
