package ru.yandex.direct.binlogbroker.replicatetoyt;

import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeoutException;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;
import ru.yandex.direct.utils.ThreadUtils;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static java.util.stream.Collectors.toList;

@ParametersAreNonnullByDefault
class TabletUtils {
    private static final Duration SLEEP_BETWEEN_TABLET_CHECKS = Duration.ofSeconds(5);
    private static final Logger logger = LoggerFactory.getLogger(TabletUtils.class);

    private TabletUtils() {
    }

    /**
     * Ожидание того, что все таблеты таблицы придут в состояние state.
     */
    static void waitForTablets(Yt yt, YPath path, TabletState state, Duration timeout)
            throws TimeoutException, InterruptedException {
        NanoTimeClock clock = new NanoTimeClock();
        MonotonicTime deadline = clock.getTime().plus(timeout);
        while (clock.getTime().isBefore(deadline)) {
            boolean allTabletsReady = areAllTabletsInState(yt, path, state);
            if (allTabletsReady) {
                return;
            }
            clock.sleep(SLEEP_BETWEEN_TABLET_CHECKS);
        }
        throw new TimeoutException(String.format(
                "tablets of table %s did not reach state %s in %s",
                path,
                state,
                timeout));
    }

    static boolean areAllTabletsInState(Yt yt, YPath path, TabletState state) {
        String actualState = yt.cypress().get(path.child("@tablet_state")).stringValue();
        return state.name.equals(actualState);
    }

    static TabletState getTableState(Yt yt, YPath path) {
        return TabletState.fromName(yt.cypress().get(path.child("@tablet_state")).stringValue());
    }

    static List<List<YTreeNode>> getPivotKeys(Yt yt, YPath path) {
        return yt.cypress().get(path.child("@pivot_keys")).asList().stream().map(YTreeNode::asList).collect(toList());
    }

    /**
     * Заморозка таблицы с ожиданием всех таблетов.
     */
    static void syncFreezeTable(Yt yt, YPath path, Duration timeout) throws TimeoutException, InterruptedException {
        logger.info("freezing table {}", path);
        NanoTimeClock clock = new NanoTimeClock();
        MonotonicTime deadline = clock.getTime().plus(timeout);
        ThreadUtils.execWithRetries(attempt -> {
            if (attempt > 1) {
                // Возможно, freeze уже сработал, и повторный запрос не требуется
                if (areAllTabletsInState(yt, path, TabletState.FROZEN)) {
                    return;
                }
            }
            yt.tables().freeze(path);
        }, 10, 1000, 1.4, logger);
        waitForTablets(yt, path, TabletState.FROZEN, deadline.minus(clock.getTime()));
        logger.info("table {} frozen", path);
    }

    /**
     * Монтирование таблицы с ожиданием всех таблетов.
     */
    static void syncMountTable(Yt yt, YPath path, Duration timeout) throws TimeoutException, InterruptedException {
        logger.info("mounting table {}", path);
        NanoTimeClock clock = new NanoTimeClock();
        MonotonicTime deadline = clock.getTime().plus(timeout);
        ThreadUtils.execWithRetries(attempt -> {
            if (attempt > 1) {
                // Возможно, mount уже сработал, и повторный запрос не требуется
                if (areAllTabletsInState(yt, path, TabletState.MOUNTED)) {
                    return;
                }
            }
            yt.tables().mount(path);
        }, 10, 1000, 1.4, logger);
        waitForTablets(yt, path, TabletState.MOUNTED, deadline.minus(clock.getTime()));
        logger.info("table {} mounted", path);
    }

    /**
     * Решардирование таблицы с повторами.
     */
    static void syncReshardTable(Yt yt, YPath path, List<List<YTreeNode>> pivotKeys) {
        logger.info("resharding table {}", path);
        ThreadUtils.execWithRetries(attempt -> {
            if (attempt > 1) {
                // Возможно, reshard уже сработал, и повторный запрос не требуется
                if (pivotKeys.equals(getPivotKeys(yt, path))) {
                    return;
                }
            }
            yt.tables().reshard(Optional.empty(), false, path, pivotKeys, Optional.empty(), Optional.empty());
        }, 10, 1000, 1.4, logger);
        logger.info("table {} resharded", path);
    }

    static void syncUnmountTable(Yt yt, YPath path, Duration timeout) throws TimeoutException, InterruptedException {
        logger.info("unmounting table {}", path);
        NanoTimeClock clock = new NanoTimeClock();
        MonotonicTime deadline = clock.getTime().plus(timeout);
        ThreadUtils.execWithRetries(attempt -> {
            if (attempt > 1) {
                // Возможно, unmount уже сработал, и повторный запрос не требуется
                if (areAllTabletsInState(yt, path, TabletState.UNMOUNTED)) {
                    return;
                }
            }
            yt.tables().unmount(path);
        }, 10, 1000, 1.4, logger);
        waitForTablets(yt, path, TabletState.UNMOUNTED, deadline.minus(clock.getTime()));
        logger.info("table {} unmounted", path);
    }


    enum TabletState {
        MOUNTED("mounted"),
        FROZEN("frozen"),
        UNMOUNTED("unmounted");

        private final String name;

        TabletState(String name) {
            this.name = name;
        }

        static TabletState fromName(String name) {
            return TabletState.valueOf(name.toUpperCase());
        }
    }
}
