package ru.yandex.direct.binlogbroker.replicatetoyt;

import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeoutException;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.NotThreadSafe;

import com.google.common.base.Preconditions;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlog.model.DropTable;
import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.binlog.model.RenameTable;
import ru.yandex.direct.utils.InterruptedRuntimeException;
import ru.yandex.direct.utils.exception.RuntimeTimeoutException;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.yt.rpcproxy.EAtomicity;
import ru.yandex.yt.rpcproxy.ETransactionType;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransaction;
import ru.yandex.yt.ytclient.proxy.ApiServiceTransactionOptions;
import ru.yandex.yt.ytclient.proxy.YtClient;


/**
 * Повторяет в YT либо один DDL {@link YtReplicatorImpl#acceptDDL(BinlogEvent, StateManager.ShardOffsetSaver)},
 * либо пачку DML {@link YtReplicatorImpl#acceptDML(List, StateManager.ShardOffsetSaver)}.
 * <p>
 * Для каждой БД и каждой таблицы из BinlogEvent идёт запись в соответствующий YT-подкаталог БД/таблица.
 */
@NotThreadSafe
@ParametersAreNonnullByDefault
public class YtReplicatorImpl implements YtReplicator {
    private final YtClient ytClient;
    private final YPath rootPath;
    private final boolean skipErroneousEvents;
    private final AlterHandler alterHandler;
    private final ConcurrentHashMap<String, TableDataYtReplicator> tableDataYtReplicators = new ConcurrentHashMap<>();

    /**
     * @param ytClient            RPC клиент для работы с YT
     * @param yt                  HTTP клиент для работы с YT
     * @param rootPath            Корневая нода на YT, в которой находятся БД-каталоги с таблицами назначения.
     * @param tmpPath             Корневая нода на YT, в которой хранятся временные таблицы
     * @param skipErroneousEvents Если по каким-то причинам обработка какого-то события вызвала ошибку, то
     *                            проигнорировать это событие и продолжить писать дальше.
     */
    public YtReplicatorImpl(YtClient ytClient, Yt yt, YPath rootPath, YPath tmpPath, boolean skipErroneousEvents) {
        this.ytClient = ytClient;
        this.rootPath = rootPath;
        this.skipErroneousEvents = skipErroneousEvents;
        this.alterHandler = new AlterHandler(yt, rootPath, tmpPath);
    }

    private TableDataYtReplicator getTableDataYtReplicator(String path) {
        return tableDataYtReplicators.computeIfAbsent(path, p -> new TableDataYtReplicator(ytClient, path,
                skipErroneousEvents));
    }

    private String getEventPath(BinlogEvent binlogEvent) {
        return rootPath.child(binlogEvent.getSource()).child(binlogEvent.getTable()).toString();
    }

    /**
     * Применить DDL событие на динамических таблицах YT в одной YT-транзакции.
     */
    @Override
    public void acceptDDL(BinlogEvent binlogEvent, StateManager.ShardOffsetSaver shardOffsetSaver) {
        Preconditions.checkState(binlogEvent.getOperation() == Operation.SCHEMA,
                "expected schema event, got %s", binlogEvent.getOperation());

        try {
            alterHandler.handle(binlogEvent, shardOffsetSaver);
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
            throw new InterruptedRuntimeException();
        } catch (TimeoutException ex) {
            throw new RuntimeTimeoutException("While handling " + binlogEvent);
        }
        if (binlogEvent.getSchemaChanges().stream()
                .noneMatch(c -> (c instanceof RenameTable) || (c instanceof DropTable))) {
            // в случае переименования или удаления таблицы нет смысла обновлять ее схему
            getTableDataYtReplicator(getEventPath(binlogEvent)).updateTableSchema();
        }
    }

    /**
     * Повторить все DML события на динамических таблицах YT в одной YT-транзакции.
     */
    @Override
    public void acceptDML(List<BinlogEvent> eventWithOffsetList, StateManager.ShardOffsetSaver shardOffsetSaver) {
        Preconditions.checkState(
                eventWithOffsetList.stream()
                        .allMatch(e -> e.getOperation() != Operation.SCHEMA),
                "found DDL event inside list of DML events");
        ApiServiceTransactionOptions transactionOptions =
                new ApiServiceTransactionOptions(ETransactionType.TT_TABLET)
                        // Все настройки, кроме sticky, transaction type и atomicity взяты с потолка.
                        // Любой желающий может вынести в поля и/или переопределить.
                        // fix RpcError: Error 1: Invalid atomicity mode: "none" instead of "full"
                        .setAtomicity(EAtomicity.A_FULL)
                        .setPing(true)
                        .setPingPeriod(Duration.ofSeconds(2))
                        .setSticky(true)
                        .setTimeout(Duration.ofSeconds(30));
        Map<String, List<BinlogEvent>> eventsForPath = eventWithOffsetList.stream()
                .collect(Collectors.groupingBy(
                        this::getEventPath,
                        Collectors.toList()));
        try (ApiServiceTransaction transaction =
                     ytClient.startTransaction(transactionOptions).join()) { // IGNORE-BAD-JOIN DIRECT-149116
            eventsForPath.forEach(
                    (path, events) -> getTableDataYtReplicator(path)
                            .processEventBatch(transaction, events)
                            .join() // IGNORE-BAD-JOIN DIRECT-149116
            );
            transaction.commit().join(); // IGNORE-BAD-JOIN DIRECT-149116
        } catch (RuntimeException exc) {
            String message = String.format("While handling batch for tables: %s, GTIDs: %s",
                    EntryStream.of(eventsForPath)
                            .mapValues(Collection::size)
                            .mapKeyValue((k, v) -> String.format("%s (%d events)", k, v))
                            .collect(Collectors.joining(", ")),
                    eventWithOffsetList.stream()
                            .collect(Collectors.groupingBy(
                                    BinlogEvent::getServerUuid,
                                    Collectors.reducing(
                                            Pair.of(Long.MAX_VALUE, Long.MIN_VALUE),
                                            event -> Pair.of(event.getTransactionId(), event.getTransactionId()),
                                            (minMax1, minMax2) -> Pair.of(
                                                    Math.min(minMax1.getLeft(), minMax2.getLeft()),
                                                    Math.max(minMax1.getRight(), minMax2.getRight())))))
                            .entrySet()
                            .stream()
                            .map(e -> String.format("%s:%d-%d",
                                    e.getKey(), e.getValue().getLeft(), e.getValue().getRight()))
                            .sorted()
                            .collect(Collectors.joining(",")));
            throw new IllegalStateException(message, exc);
        }
        shardOffsetSaver.save();
    }
}
