package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import java.time.Duration;
import java.util.Iterator;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.ImmutableSourceState;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

/**
 * Отложенная запись состояния. Состояние записывается в отдельном потоке асинхронно. Если попробовать записать новоое
 * состояние для одного источника, пока старое не успело записаться, то сначала дождётся завершения старой записи, а
 * потом асинхронно запустит новую запись. Если при записи состояния возникнет ошибка, она будет переброшена при
 * следующей записи.
 */
@ParametersAreNonnullByDefault
@ThreadSafe
public class AsyncSaveStateRepository implements SourceStateRepository {
    /**
     * Отдельный пул потоков для асинхронного сохранения состояния. Вероятность одновременного сохранения двух и более
     * состояний низка, но если вдруг начнёт наполняться очередь, не будет создано огромное количество потоков.
     */
    private final ExecutorService asyncStateSavers = new ThreadPoolExecutor(1, 4, 1, TimeUnit.MINUTES,
            new LinkedBlockingQueue<>(),
            new ThreadFactoryBuilder()
                    .setDaemon(true)  // Сохранение состояния не должно препятствовать завершению приложения
                    .setNameFormat("async-save-state-%d")
                    .build());
    private final Map<SourceType, CompletableFuture<Void>> previousSaves = new ConcurrentHashMap<>();
    private final SourceStateRepository forward;
    private final Duration saveTimeout;
    private volatile boolean closing;

    public AsyncSaveStateRepository(SourceStateRepository forward, Duration saveTimeout) {
        this.forward = forward;
        this.saveTimeout = saveTimeout;
    }

    @Override
    public ImmutableSourceState loadState(SourceType source) {
        return forward.loadState(source);
    }

    @Override
    public void saveState(SourceType source, ImmutableSourceState sourceState) {
        Preconditions.checkState(!closing, "State saver is closing now");
        CompletableFuture<Void> oldFuture = previousSaves.remove(source);
        if (oldFuture != null) {
            try {
                oldFuture.get(saveTimeout.toMillis(), MILLISECONDS);
            } catch (InterruptedException | ExecutionException | TimeoutException e) {
                throw new RuntimeException(
                        "Can't save logbroker writer state for source " + source + ". Timeout was " + saveTimeout
                                + ". Message - " + e.getMessage(), e);
            }
        }
        AtomicBoolean inRace = new AtomicBoolean(true);
        previousSaves.computeIfAbsent(source, s -> {
            Preconditions.checkState(!closing, "State saver is closing now");
            inRace.set(false);
            return CompletableFuture.runAsync(() -> forward.saveState(s, sourceState), asyncStateSavers);
        });
        Preconditions.checkState(!inRace.get(), "Race. Two or more threads writes state for %s", source);
    }

    @Override
    public String getClusterName() {
        return forward.getClusterName();
    }

    @Override
    public void close() {
        closing = true;
        Iterator<CompletableFuture<Void>> futureIter = previousSaves.values().iterator();
        while (futureIter.hasNext()) {
            CompletableFuture<Void> future;
            try {
                future = futureIter.next();
                futureIter.remove();
            } catch (NoSuchElementException ignored) {
                break;
            }
            future.join(); // IGNORE-BAD-JOIN DIRECT-149116
        }
    }
}
