package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.ImmutableSourceState;

/**
 * Размножающий SourceStateRepository — пишем стейт и в primary и в secondary, читаеем из primary.
 * Ошибки записи в запасной — игнорируем.
 */
@ParametersAreNonnullByDefault
public class AsyncSaveStateWithReplicaRepository implements SourceStateRepository {
    private static final Logger logger = LoggerFactory.getLogger(AsyncSaveStateWithReplicaRepository.class);
    private final SourceStateRepository primary;
    private final SourceStateRepository replica;

    public AsyncSaveStateWithReplicaRepository(SourceStateRepository primary, SourceStateRepository replica) {
        this.primary = primary;
        this.replica = replica;
    }

    @Override
    public ImmutableSourceState loadState(SourceType source) {
        return primary.loadState(source);
    }

    @Override
    public String getClusterName() {
        return primary.getClusterName();
    }

    @Override
    public void saveState(SourceType source, ImmutableSourceState sourceState) {
        primary.saveState(source, sourceState);
        try {
            replica.saveState(source, sourceState);
        } catch (RuntimeException e) {
            logger.warn("Saving state into backup repository failed", e);
        }
    }

    @Override
    public void close() throws Exception {
        RuntimeException primaryException = null;

        try {
            primary.close();
        } catch (RuntimeException ex) {
            primaryException = ex;
        }
        try {
            replica.close();
        } catch (RuntimeException secondaryException) {
            if (primaryException != null) {
                primaryException.addSuppressed(secondaryException);
                throw primaryException;
            } else {
                throw secondaryException;
            }
        }
    }
}
