package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriter;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.BinlogWithSeqId;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.LogbrokerWriterConfig;
import ru.yandex.direct.utils.AutoCloseableList;
import ru.yandex.direct.utils.Transient;
import ru.yandex.kikimr.persqueue.LogbrokerClientAsyncFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;
import ru.yandex.kikimr.persqueue.producer.async.AsyncProducerConfig;

import static java.nio.charset.StandardCharsets.UTF_8;
import static ru.yandex.direct.utils.Interrupts.failingGet;

@Component
@ParametersAreNonnullByDefault
public class BinlogEventConsumerFactory implements AutoCloseable {

    private final LogbrokerWriterConfig logbrokerWriterConfig;
    private final LogbrokerWriterConfig logbrokerQueryWriterConfig;
    private final Supplier<Credentials> logbrokerCredentialsSupplier;
    private final LogbrokerClientAsyncFactory logbrokerClientFactory;
    private final String logbrokerSourceIdPrefix;
    private DataFormat logbrokerDataFormat;

    private final AutoCloseableList<LogbrokerWriter<BinlogWithSeqId>> logbrokerWriters = new AutoCloseableList<>();

    @Autowired
    public BinlogEventConsumerFactory(
            LogbrokerWriterConfig logbrokerWriterConfig,
            LogbrokerWriterConfig logbrokerQueryWriterConfig,
            Supplier<Credentials> logbrokerCredentialsSupplier,
            LogbrokerClientAsyncFactory logbrokerClientFactory,
            String logbrokerSourceIdPrefix,
            DataFormat logbrokerDataFormat
    ) {
        this.logbrokerWriterConfig = logbrokerWriterConfig;
        this.logbrokerQueryWriterConfig = logbrokerQueryWriterConfig;
        this.logbrokerCredentialsSupplier = logbrokerCredentialsSupplier;
        this.logbrokerClientFactory = logbrokerClientFactory;
        this.logbrokerSourceIdPrefix = logbrokerSourceIdPrefix;
        this.logbrokerDataFormat = logbrokerDataFormat;
    }

    public LogbrokerWriter<BinlogWithSeqId> getForSource(SourceType source) {
        Supplier<CompletableFuture<AsyncProducer>> asyncProducerSupplier = getAsyncProducerSupplier(source, logbrokerWriterConfig);

        LogbrokerWriter<BinlogWithSeqId> logbrokerWriter = new BinlogLogbrokerWriter(asyncProducerSupplier,
                Duration.ofSeconds(logbrokerWriterConfig.getLogbrokerTimeout()),
                logbrokerWriterConfig.getLogbrokerRetries(), logbrokerDataFormat);
        logbrokerWriters.add(new Transient<>(logbrokerWriter));
        return logbrokerWriter;
    }

    // опциональный, чтобы поддержать обратную совместимость с текущим продакшеновым процессом, который пишет protobuf и не пишет запросов
    // если в будущем перейдём на json, можно сделать обязательным и немного упростить
    @Nullable
    public LogbrokerWriter<BinlogWithSeqId> getQueryWriterForSource(SourceType source) {
        if (logbrokerQueryWriterConfig.getTopics().isEmpty()) {
            return null;
        }
        Supplier<CompletableFuture<AsyncProducer>> asyncProducerSupplier =
                getAsyncProducerSupplier(source, logbrokerQueryWriterConfig);

        LogbrokerWriter<BinlogWithSeqId> logbrokerQueryWriter =
                new BinlogLogbrokerQueryWriter(asyncProducerSupplier,
                        Duration.ofSeconds(logbrokerQueryWriterConfig.getLogbrokerTimeout()),
                        logbrokerQueryWriterConfig.getLogbrokerRetries(), logbrokerDataFormat);
        logbrokerWriters.add(new Transient<>(logbrokerQueryWriter));
        return logbrokerQueryWriter;
    }

    private Supplier<CompletableFuture<AsyncProducer>> getAsyncProducerSupplier(SourceType source,
            LogbrokerWriterConfig writerConfig) {
        String sourceName = source.getSourceName();
        String sourceId = logbrokerSourceIdPrefix + sourceName;
        Map<String, String> meta = new HashMap<>();
        if (writerConfig.getWriteMetadataForLogshatter()) {
            meta.put("ident", writerConfig.getIdentForSource(sourceName));
            meta.put("logtype", writerConfig.getLogTypeForSource(sourceName));
        }
        AsyncProducerConfig asyncProducerConfig =
                AsyncProducerConfig.builder(writerConfig.getTopicForSource(sourceName), sourceId.getBytes(UTF_8))
                        .setCodec(writerConfig.getCodec())
                        .setGroup(source.getLogbrokerGroupId())
                        .setCredentialsProvider(logbrokerCredentialsSupplier)
                        .setExtraMetadataSingleValue(meta)
                        .build();
        return () -> failingGet(() -> logbrokerClientFactory.asyncProducer(asyncProducerConfig));
    }

    @Override
    @PreDestroy
    public void close() {
        logbrokerWriters.close();
    }
}
