package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.AbstractLogbrokerWriterImpl;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriterRetryConfig;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.BinlogWithSeqId;
import ru.yandex.direct.utils.json.LocalDateTimeDeserializer;
import ru.yandex.direct.utils.json.LocalDateTimeSerializer;
import ru.yandex.kikimr.persqueue.producer.AsyncProducer;

@ParametersAreNonnullByDefault
public class BinlogLogbrokerWriter extends AbstractLogbrokerWriterImpl<BinlogWithSeqId> {
    /**
     * Добавляет формат времени из {@link LocalDateTimeSerializer} и {@link LocalDateTimeDeserializer}
     */
    private static final ObjectMapper DATETIME_FORMAT_MAPPER = new ObjectMapper()
            .registerModule(new JavaTimeModule()
                    .addSerializer(LocalDateTime.class, new LocalDateTimeSerializer())
                    .addDeserializer(LocalDateTime.class, new LocalDateTimeDeserializer()))
            .disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);

    private DataFormat dataFormat;

    public BinlogLogbrokerWriter(Supplier<CompletableFuture<AsyncProducer>> asyncProducerSupplier,
                                 Duration logbrokerTimeout, int retries, DataFormat dataFormat) {
        super(asyncProducerSupplier, logbrokerTimeout, new LogbrokerWriterRetryConfig(retries), false);
        this.dataFormat = dataFormat;
    }

    public DataFormat getDataFormat() {
        return dataFormat;
    }


    static byte[] convertToJson(BinlogEvent event) {
        try {
            return DATETIME_FORMAT_MAPPER.writeValueAsBytes(event);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("can not serialize object to json", e);
        }
    }

    @Override
    protected LogbrokerWriteRequest makeRequest(BinlogWithSeqId record) {
        byte[] bytesToWrite;
        switch (dataFormat) {
            case PROTOBUF:
                bytesToWrite = record.event.toProtobuf().toByteArray();
                break;
            case JSON:
                bytesToWrite = convertToJson(record.event);
                break;
            default:
                throw new IllegalArgumentException(String.format("unsupported data format: %s", dataFormat));
        }
        return new LogbrokerWriteRequest(bytesToWrite, record.seqId);
    }
}
