package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import java.io.Serializable;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneOffset;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Deque;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Supplier;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.Maps;
import com.google.common.primitives.UnsignedInteger;
import com.google.common.primitives.UnsignedLong;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlog.model.ColumnType;
import ru.yandex.direct.binlog.model.CreateTable;
import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.binlog.model.RenameTable;
import ru.yandex.direct.binlog.model.Truncate;
import ru.yandex.direct.binlog.reader.BinlogReader;
import ru.yandex.direct.binlog.reader.BinlogSource;
import ru.yandex.direct.binlog.reader.BinlogStateOptimisticSnapshotter;
import ru.yandex.direct.binlog.reader.EnrichedDeleteRow;
import ru.yandex.direct.binlog.reader.EnrichedEvent;
import ru.yandex.direct.binlog.reader.EnrichedInsertRow;
import ru.yandex.direct.binlog.reader.EnrichedRow;
import ru.yandex.direct.binlog.reader.EnrichedUpdateRow;
import ru.yandex.direct.binlog.reader.MySQLSimpleRowIndexed;
import ru.yandex.direct.binlog.reader.StateBound;
import ru.yandex.direct.binlog.reader.StatefulBinlogSource;
import ru.yandex.direct.binlog.reader.Transaction;
import ru.yandex.direct.binlog.reader.TransactionReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.binlogbroker.logbroker_utils.writer.LogbrokerWriter;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.BinlogWithSeqId;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.ImmutableSourceState;
import ru.yandex.direct.binlogbroker.mysql.MysqlUtil;
import ru.yandex.direct.db.config.DbConfig;
import ru.yandex.direct.db.config.DbConfigEvent;
import ru.yandex.direct.db.config.DbConfigListener;
import ru.yandex.direct.mysql.BinlogEventData;
import ru.yandex.direct.mysql.MySQLBinlogState;
import ru.yandex.direct.mysql.MySQLColumnData;
import ru.yandex.direct.mysql.MySQLColumnType;
import ru.yandex.direct.mysql.MySQLServerBuilder;
import ru.yandex.direct.mysql.MySQLSimpleConnector;
import ru.yandex.direct.mysql.MySQLUtils;
import ru.yandex.direct.mysql.schema.KeyColumn;
import ru.yandex.direct.mysql.schema.KeySchema;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceGuard;
import ru.yandex.direct.tracing.TraceHelper;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.tracing.data.DirectTraceInfo;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.MonotonicTime;
import ru.yandex.direct.utils.NanoTimeClock;
import ru.yandex.direct.utils.db.MySQLConnector;
import ru.yandex.direct.utils.exception.RuntimeTimeoutException;
import ru.yandex.misc.concurrent.TimeoutRuntimeException;
import ru.yandex.misc.thread.ExecutionRuntimeException;

import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Чтение бинлога из одного mysql-сервера и преобразование событий в {@link BinlogEvent}.
 */
@ParametersAreNonnullByDefault
public class Binlogbroker implements Runnable, Checked.CheckedRunnable<RuntimeException>, DbConfigListener {
    private static final Logger logger = LoggerFactory.getLogger(Binlogbroker.class);
    /**
     * Писать в лог о залитых событиях не чаще, чем указанный интервал времени.
     */
    private static final Duration LOG_MESSAGE_WITH_COUNT_PERIOD = Duration.ofSeconds(60);
    /**
     * Защитная задержка при смене кластера хранения стейта/взятия локов.
     * <p>
     * Если прочитанный стейт записан на другом кластере (не текущем) и последнее событие в нём произошло за последние
     * CLUSTER_CHANGE_PROTECTIVE_DELAY секунд — будет выброшено исключение.
     * Это сделано для предотвращения параллельной работы бинлогброкера на обоих кластерах.
     * <p>
     * Значение должно быть больше таймаутов на локи, под которым ведется работа.
     */
    private static final Duration CLUSTER_CHANGE_PROTECTIVE_DELAY = Duration.ofMinutes(3);
    private final SourceType source;
    private final Supplier<DbConfig> sourceDbConfigSupplier;
    private final LogbrokerWriter<BinlogWithSeqId> binlogEventConsumer;
    private final LogbrokerWriter<BinlogWithSeqId> binlogQueryConsumer;
    private final SourceStateRepository sourceStateRepository;
    private final Integer initialServerId;
    private final Duration keepAliveTimeout;
    private final int maxBufferedEvents;
    private final int consumerChunkSize;
    private final int maxEventsPerTransaction;
    private final Duration consumerChunkDuration;
    private final MySQLServerBuilder mysqlServerBuilder;
    private final Duration flushEventsTimeout;
    /**
     * table -> fields
     */
    private final Map<String, Set<String>> dontTruncateFields;
    private Deque<BinlogEvent> sendQueue;
    private final LogbrokerWriterMonitoring monitoring;
    private final TraceHelper traceHelper;
    private MonotonicTime flushDeadline;
    private int logMessageSentCount;
    private MonotonicTime lastLogMessageCountWasAt;
    private volatile MySQLConnector connector;
    /**
     * Флаг взводится при обновлении DBConfig {@link Binlogbroker#update(DbConfigEvent)}
     */
    private volatile Boolean dbConfigReset = false;

    private final ExecutorService flushEventsAndSaveStateExecutorService;
    private CompletableFuture<ImmutableSourceState> flushEventsAndSaveStateFuture;

    private ImmutableSourceState sourceState;

    @SuppressWarnings({"squid:S00107", "checkstyle:parameternumber"})
    private Binlogbroker(SourceType source,
                         Supplier<DbConfig> sourceDbConfigSupplier,
                         LogbrokerWriter<BinlogWithSeqId> binlogEventConsumer,
                         @Nullable LogbrokerWriter<BinlogWithSeqId> binlogQueryConsumer,
                         SourceStateRepository sourceStateRepository,
                         MySQLServerBuilder mysqlServerBuilder,
                         @Nullable Integer initialServerId,
                         Duration keepAliveTimeout,
                         int maxBufferedEvents,
                         int consumerChunkSize,
                         int maxEventsPerTransaction,
                         Duration consumerChunkDuration,
                         Map<String, Set<String>> dontTruncateFields,
                         LogbrokerWriterMonitoring monitoring,
                         TraceHelper traceHelper,
                         Duration flushEventsTimeout
    ) {
        this.source = source;
        this.sourceDbConfigSupplier = sourceDbConfigSupplier;
        this.binlogEventConsumer = binlogEventConsumer;
        this.binlogQueryConsumer = binlogQueryConsumer;
        this.sourceStateRepository = sourceStateRepository;
        this.mysqlServerBuilder = mysqlServerBuilder;
        this.initialServerId = initialServerId;
        this.keepAliveTimeout = keepAliveTimeout;
        this.maxBufferedEvents = maxBufferedEvents;
        this.consumerChunkSize = consumerChunkSize;
        this.maxEventsPerTransaction = maxEventsPerTransaction;
        this.consumerChunkDuration = consumerChunkDuration;
        this.dontTruncateFields = dontTruncateFields;
        this.sendQueue = new ArrayDeque<>(consumerChunkSize);
        this.monitoring = monitoring;
        this.traceHelper = traceHelper;
        this.flushEventsTimeout = flushEventsTimeout;
        this.lastLogMessageCountWasAt = NanoTimeClock.now();
        String threadNameFormat =
                "flush-events-and-save-state-" + source + "-%02d";
        ThreadFactory threadFactory =
                new ThreadFactoryBuilder().setNameFormat(threadNameFormat).build();
        this.flushEventsAndSaveStateExecutorService = Executors.newSingleThreadExecutor(threadFactory);
    }

    public static Builder builder() {
        return new Builder();
    }

    private boolean writingQueriesEnabled() {
        return binlogQueryConsumer != null;
    }

    private void transactionToBinlogEvents(Collection<BinlogEvent> result, String source, String db,
                                           boolean needWriteQueryOnFirstQuery, Transaction transaction) {
        StreamEx.of(transaction.getEnrichedEvents())
                .filter(this::filterRowEventsByTable)
                .flatMap(EnrichedEvent::rowsStream)
                .groupRuns((before, after) -> before.getClass() == after.getClass()
                        && before.getGtid().equals(after.getGtid())
                        && before.getEventSerial() == after.getEventSerial()
                        && before.getDbName().equals(after.getDbName())
                        && before.getTableName().equals(after.getTableName()))
                // транзакция не может затрагивать несколько БД, поэтому можно фильтровать по БД первой строки
                .filter(enrichedRows -> db.equals(enrichedRows.get(0).getDbName()))
                .map(this::enrichedRowsToBinlogEvent)
                .filter(Objects::nonNull)
                .map(e -> e.withSource(source).validate())
                .groupRuns((before, after) -> before.getQueryIndex() == after.getQueryIndex())
                // События и запросы пишутся из одной очереди.
                // На один запрос может приходиться несколько событий, так что помечаем какое-то одно событие,
                // для которого нужно записать ещё и запрос
                // Отбор по eventIndex == 0 не подходит, т. к. такое событие может быть отфильтровано выше.
                .flatMap(events -> {
                    // Не отправляем запрос, если это не первый запрос транзакции (его уже отправили в прошлый раз)
                    if (transaction.getQueryStartIndex() == 0 && transaction.getEventStartIndex() == 0 ||
                            (transaction.getQueryStartIndex() != events.get(0).getQueryIndex() || needWriteQueryOnFirstQuery)) {
                        events.get(0).setWriteQuery(true);
                    }
                    return StreamEx.of(events);
                })
                .forEachOrdered(event -> {
                    // и protobuf-, и json-инстансы отправляют одинаковые метрики (различаются только меткой binlogbroker_format)
                    // экономнее было бы отправлять из какого-то одного, но это деталь, которую легко потерять
                    monitoring.addEventByTableAndOperation(this.source, event.getTable(), event.getOperation());
                    result.add(event);
                });
    }

    private boolean filterRowEventsByTable(EnrichedEvent enrichedEvent) {
        if (!enrichedEvent.isRowsEvent()) {
            return true;
        }
        String rowsTable = enrichedEvent.getRowsTable();
        if (rowsTable != null && rowsTable.startsWith("_")) {
            // во время альтеров через pt-osc идёт большой поток событий в таблицы вида _TBL_new
            // нам эти события не нужны, поэтому отфильтровываем
            monitoring.addSkippedByTableCount(source, 1);
            return false;
        }
        return true;
    }

    @Nullable
    private BinlogEvent enrichedRowsToBinlogEvent(List<? extends EnrichedRow> enrichedRows) {
        Preconditions.checkState(!enrichedRows.isEmpty());
        EnrichedRow firstRow = enrichedRows.get(0);
        String gtid = firstRow.getGtid();
        BinlogEvent result = new BinlogEvent()
                .withOperation(operationOfRow(firstRow));
        for (EnrichedRow enrichedRow : enrichedRows) {
            Map<String, Object> before;
            Map<String, Object> after;
            switch (result.getOperation()) {
                case INSERT:
                    before = Maps.newLinkedHashMapWithExpectedSize(0);
                    after = fieldsToObjectMap(((EnrichedInsertRow) enrichedRow).getFields());
                    break;
                case UPDATE:
                    before = fieldsToObjectMap(((EnrichedUpdateRow) enrichedRow).getFields().getBefore());
                    after = fieldsToObjectMap(((EnrichedUpdateRow) enrichedRow).getFields().getAfter());
                    break;
                case DELETE:
                    before = fieldsToObjectMap(((EnrichedDeleteRow) enrichedRow).getFields());
                    after = Maps.newLinkedHashMapWithExpectedSize(0);
                    break;
                default:
                    throw new IllegalStateException(result.getOperation().toString());
            }
            Map<String, Object> primaryKey = extractPrimaryKey(enrichedRow, before, after);

            Set<String> dontTruncate = dontTruncateFields.getOrDefault(enrichedRow.getTableName(), emptySet());
            deduplicate(before, after, dontTruncate);

            if (result.getOperation() == Operation.DELETE) {
                // При удалении оставляется исключительно primary key, остальные поля выбрасываются (кроме dontTruncate)
                removeAllExcept(before, dontTruncate);
                result.addRows(new BinlogEvent.Row()
                        .withAfter(Map.of())
                        .withBefore(before.isEmpty() ? Map.of() : before)
                        .withPrimaryKey(primaryKey)
                        .withRowIndex(enrichedRow.getRowSerial()));
            } else if (!after.isEmpty()) {
                // primary key удаляется из before, но если он изменился, то оставляется в after
                primaryKey.keySet().forEach(before::remove);
                result.addRows(new BinlogEvent.Row()
                        .withAfter(after)
                        .withBefore(before)
                        .withPrimaryKey(primaryKey)
                        .withRowIndex(enrichedRow.getRowSerial()));
            }
        }
        if (result.getRows().isEmpty()) {
            return null;
        } else {
            DirectTraceInfo directTraceInfo = firstRow.getDirectTraceInfo();

            BinlogEvent res = result
                    .withDb(firstRow.getDbName())
                    .withGtid(gtid)
                    .withQueryIndex(firstRow.getQuerySerial())
                    .withEventIndex(firstRow.getEventSerial())
                    .withQuery(firstRow.getQuery())
                    .withTable(firstRow.getTableName())
                    // На момент написания этого кода EnrichedRow.getDateTime гарантированно
                    // возвращал время в UTC.
                    .withUtcTimestamp(firstRow.getDateTime());

            addDirectTraceInfoToEvent(res, directTraceInfo);

            return res;
        }
    }

    private static void ddlToBinlogEvent(Collection<BinlogEvent> result, String source, String desiredDb,
                                         ru.yandex.direct.mysql.BinlogEvent sourceBinlogEvent) {
        Preconditions.checkState(sourceBinlogEvent.getData() instanceof BinlogEventData.DDL);
        BinlogEventData.DDL ddl = sourceBinlogEvent.getData();
        try {
            String gtid = sourceBinlogEvent.getData().getGtid();

            BinlogEvent resultBinlogEvent = new BinlogEvent()
                    .withDb(desiredDb)
                    .withGtid(gtid)
                    .withOperation(Operation.SCHEMA)
                    // Каждый DDL-запрос делается в отдельном gtid.
                    .withQueryIndex(0)
                    .withEventIndex(0)
                    .withWriteQuery(true)
                    .withQuery(ddl.getData().getSql())
                    .withSource(source)
                    .withUtcTimestamp(Instant.ofEpochMilli(sourceBinlogEvent.getTimestamp()).atZone(ZoneOffset.UTC)
                            .toLocalDateTime());

            DirectTraceInfo directTraceInfo = DirectTraceInfo.extractIgnoringErrors(ddl.getData().getSql());

            addDirectTraceInfoToEvent(resultBinlogEvent, directTraceInfo);

            // rename table нельзя определить по схеме, он или не определится вовсе (например, если две таблицы
            // меняются местами). Или (что ещё хуже) определится как drop table + create table. Поэтому эта
            // проверка должна быть в начале.
            List<Pair<String, String>> renames = MysqlUtil.looksLikeRenameTable(ddl.getData().getSql());
            if (renames != null) {
                RenameTable renameTable = new RenameTable();
                boolean firstRename = true;
                for (Pair<String, String> rename : renames) {
                    if (firstRename) {
                        firstRename = false;
                        // на самом деле не используется, вставляется, чтобы механизм валидации корректно работал
                        resultBinlogEvent.setTable(rename.getLeft());
                    }
                    renameTable.withAddRename(rename.getLeft(), rename.getRight());
                }
                result.add(resultBinlogEvent.withAddedSchemaChanges(renameTable).validate());
                return;
            }

            Map<String, CreateTable> oldTables = MysqlUtil.tableByNameForDb(desiredDb, ddl.getBefore());
            Map<String, CreateTable> newTables = MysqlUtil.tableByNameForDb(desiredDb, ddl.getAfter());

            int changedTables = MysqlUtil.createTableEvent(resultBinlogEvent, oldTables, newTables)
                    + MysqlUtil.dropTableEvent(resultBinlogEvent, oldTables, newTables)
                    + MysqlUtil.columnChangeEvent(resultBinlogEvent, oldTables, newTables);
            if (changedTables == 0) {
                // truncate table невозможно определить через diff схемы
                // Эта проверка проводится после проверки diff'а, т.к. реализованный парсер ненадёжен,
                // чем меньше шансов ложного срабатывания, тем лучше.
                String truncatedTableName = MysqlUtil.looksLikeTruncateTable(ddl.getData().getSql());
                if (truncatedTableName != null) {
                    resultBinlogEvent
                            .withAddedSchemaChanges(new Truncate())
                            .withTable(truncatedTableName);
                    changedTables = 1;
                }
            }
            if (changedTables == 1 && !resultBinlogEvent.getSchemaChanges().isEmpty()) {
                result.add(resultBinlogEvent.validate());
            } else {
                logger.warn("Got alter that changes {} tables. Skipped it. SQL: {}",
                        changedTables, ddl.getData().getSql());
            }
        } catch (RuntimeException exc) {
            throw new IllegalStateException("While handling SQL-query: " + ddl.getData().getSql(), exc);
        }
    }

    /**
     * Если в двух мапах есть одинаковые пары ключ-значение, то они удаляются.
     *
     * @param except -- имена колонок, которые не следует удалять
     */
    private static void deduplicate(Map<String, Object> before, Map<String, Object> after, Set<String> except) {
        Iterator<Map.Entry<String, Object>> beforeEntryIter = before.entrySet().iterator();
        while (beforeEntryIter.hasNext()) {
            Map.Entry<String, Object> beforeEntry = beforeEntryIter.next();
            if (!except.contains(beforeEntry.getKey())
                    && Objects.equals(beforeEntry.getValue(), after.get(beforeEntry.getKey()))) {
                after.remove(beforeEntry.getKey());
                beforeEntryIter.remove();
            }
        }
    }

    private static void removeAllExcept(Map<String, Object> before, Set<String> except) {
        before.entrySet().removeIf(entry -> !except.contains(entry.getKey()));
    }

    private static Map<String, Object> extractPrimaryKey(EnrichedRow enrichedRow, Map<String, Object> before,
                                                         Map<String, Object> after) {
        Map<String, Object> result = new LinkedHashMap<>();
        Optional<KeySchema> primaryKey = enrichedRow.getTableSchema().getPrimaryKey();
        if (primaryKey.isPresent()) {
            for (KeyColumn keyColumn : primaryKey.get().getColumns()) {
                String name = keyColumn.getName();
                if (after.containsKey(name)) {
                    result.put(name, after.get(name));
                } else {
                    result.put(name, before.get(name));
                }
            }
        }
        return result;
    }

    private static Map<String, Object> fieldsToObjectMap(MySQLSimpleRowIndexed fields) {
        LinkedHashMap<String, Object> result = Maps.newLinkedHashMapWithExpectedSize(fields.size());
        for (MySQLColumnData columnData : fields) {
            // mysql присылает все char/text/binary/blob-типы в виде byte[]
            // При этом получатель желает видеть разницу между utf8-строкой и набором байт.
            ColumnType.Normalized normalized = calcNormalized(columnData);
            result.put(columnData.getSchema().getName(), normalized == null ? null : normalized.getObject());
        }
        return result;
    }

    private static ColumnType.Normalized calcNormalized(MySQLColumnData columnData) {
        if (columnData.getRawValue() == null) {
            return null;
        }
        Object preNormalized;
        switch (columnData.getSchema().mysqlDataType()) {
            case BIGINT:
                ColumnType columnType = MysqlUtil.mysqlDataTypeToColumnType(columnData.getSchema());
                if (columnType == ColumnType.UNSIGNED_BIGINT) {
                    preNormalized = UnsignedLong.fromLongBits((long) columnData.getRawValue()).bigIntegerValue();
                } else {
                    preNormalized = columnData.getRawValue();
                }
                break;
            case INT:
                if (columnData.getSchema().isUnsigned()) {
                    int value = getRawValueAsInt(columnData);
                    preNormalized = UnsignedInteger.fromIntBits(value).longValue();
                } else {
                    preNormalized = columnData.getRawValue();
                }
                break;
            case SMALLINT:
                if (columnData.getSchema().isUnsigned()) {
                    int value = getRawValueAsInt(columnData);
                    if (value < 0) {
                        preNormalized = 65536 + value;
                    } else {
                        preNormalized = columnData.getRawValue();
                    }
                } else {
                    preNormalized = columnData.getRawValue();
                }
                break;
            case TINYINT:
                if (columnData.getSchema().isUnsigned()) {
                    int value = getRawValueAsInt(columnData);
                    if (value < 0) {
                        preNormalized = 256 + value;
                    } else {
                        preNormalized = columnData.getRawValue();
                    }
                } else {
                    preNormalized = columnData.getRawValue();
                }
                break;
            case CHAR:
            case JSON:
            case LONGTEXT:
            case MEDIUMTEXT:
            case SET:
            case TEXT:
            case TINYTEXT:
            case VARCHAR:
                preNormalized = columnData.getValueAsString();
                break;
            case ENUM:
                final Integer rawValue = (Integer) columnData.getRawValue();
                // https://dev.mysql.com/doc/refman/5.7/en/enum.html#enum-nulls
                // MySQL всегда добавляет enum-элемент для пустой строки на позицию 0, даже если это не
                // запрашивалось. При этом MySQL отличает NULL от пустой строки.
                // Код ниже трактует невалидное значение как пустую строку
                // (так это выглядит в SQL результатах и binlog'ах)
                // NULL как NULL. Приер SQL - direct-sql ts:ppc:15 "SELECT count(*), day_budget_show_mode
                // FROM campaigns GROUP by day_budget_show_mode"
                if (rawValue == null) {
                    preNormalized = null;
                } else if (rawValue == 0) {
                    preNormalized = "";
                } else {
                    MySQLColumnType mySQLColumnType =
                            MySQLColumnType.getCached(columnData.getSchema().getColumnType());
                    preNormalized = mySQLColumnType.getValue(rawValue - 1);
                }
                break;
            default:
                preNormalized = columnData.getRawValue();
        }

        return ColumnType.normalize(preNormalized);
    }

    private static int getRawValueAsInt(MySQLColumnData columnData) {
        Serializable rawValue = columnData.getRawValue();
        if (rawValue instanceof Integer) {
            return (int) rawValue;
        } else if (rawValue instanceof Byte) {
            return ((Byte) rawValue).intValue();
        } else if (rawValue instanceof Short) {
            return ((Short) rawValue).intValue();
        } else {
            throw new IllegalArgumentException("Can't get rawValue as int for " + columnData);
        }
    }

    private static Operation operationOfRow(EnrichedRow row) {
        if (row instanceof EnrichedInsertRow) {
            return Operation.INSERT;
        } else if (row instanceof EnrichedUpdateRow) {
            return Operation.UPDATE;
        } else if (row instanceof EnrichedDeleteRow) {
            return Operation.DELETE;
        } else {
            throw new IllegalArgumentException(row.getClass().getCanonicalName());
        }
    }

    private static MySQLConnector dbConfigToConnector(DbConfig dbConfig) {
        return new MySQLSimpleConnector(dbConfig.getHosts().get(0),
                dbConfig.getPort(), dbConfig.getUser(), dbConfig.getPass());
    }

    @Override
    public void run() {
        try {
            connector = dbConfigToConnector(sourceDbConfigSupplier.get());
            sourceState = sourceStateRepository.loadState(source);
            checkStateForRace();
            monitoring.setSourceHighwatermark(source, sourceState.getHwmTimestamp());
            if (sourceState.isMysqlStateEmpty()) {
                BinlogStateOptimisticSnapshotter snapshotter =
                        new BinlogStateOptimisticSnapshotter(MySQLUtils.pickServerId(initialServerId));
                MySQLBinlogState mySQLBinlogState = snapshotter.snapshot(connector);
                // Если начинаем с нуля, то initialSeqId нужно взять из логброкера,
                // иначе logbroker-writer будет пропускать все события до тех пор, пока не догонит это число
                long seqId = binlogEventConsumer.getInitialMaxSeqNo();
                sourceState = new ImmutableSourceState(seqId, sourceState.getHwmTimestamp(),
                        mySQLBinlogState.getGtidSet(), 0, mySQLBinlogState.getSerializedServerSchema(),
                        sourceStateRepository.getClusterName()
                );
                sourceStateRepository.saveState(source, sourceState);
                logger.warn("saved new state for {}: gtid={}, seqno={}",
                        source.getDbName(), sourceState.getGtid(), sourceState.getSeqId());
            }

            String db = sourceDbConfigSupplier.get().getDb();
            flushDeadline = NanoTimeClock.now().plus(consumerChunkDuration);
            while (!Thread.currentThread().isInterrupted()) {
                try (BinlogReader binlogReader = new BinlogReader(
                        new StatefulBinlogSource(
                                new BinlogSource(source.getDbName(), connector),
                                new MySQLBinlogState(sourceState.getSerializedServerSchema(), sourceState.getGtid())),
                        mysqlServerBuilder.copy(),
                        null,
                        keepAliveTimeout,
                        MySQLUtils.pickServerId(initialServerId),
                        maxBufferedEvents)) {
                    logger.info("created new BinlogReader {}", binlogReader.getSource().getConnector());
                    insideBinlogReader(db, binlogReader);
                } catch (InterruptedException e) {
                    // Предполагается, что после этого кода ничего работать не будет.
                    Thread.currentThread().interrupt();
                } catch (BinlogReader.BadServerIdException exc) {
                    // Если задан константный server id, значит подключился новый slave с таким же server id.
                    // Попытка повторного соединения приведёт к тому, что два slave будут по кругу дисконнектить
                    // друг друга.
                    // Если задан выбор случайного server id, то стоит переподключиться с новым server id.
                    if (initialServerId != null) {
                        throw exc;
                    }
                }
            }
        } finally {
            flushEventsAndSaveStateExecutorService.shutdown();
        }
    }

    /**
     * Прорверить подгруженный стейт, то его не пишут активно с другого кластера.
     * Пытаемся косвенно защищаться от кофликтов при смене кластера хранения локов/стейта
     */
    private void checkStateForRace() {
        if (sourceState.getClusterName() == null) {
            // стейта не было или храним его не в ыте
            return;
        }
        if (sourceState.getClusterName().equals(sourceStateRepository.getClusterName())) {
            // все ок, стейт записан при работе с этим же кластером (значит StateGuard — могут нас защитить)
            return;
        }

        long stateTimestamp = sourceState.getHwmTimestamp();
        long now = Instant.now().getEpochSecond();
        long window = CLUSTER_CHANGE_PROTECTIVE_DELAY.getSeconds();
        if (now - window < stateTimestamp) {
            logger.error("State for {} was written on cluster {} less than {} seconds ago — probably old Binlogbroker" +
                    " process still working", source, sourceState.getClusterName(), window);
            throw new IllegalStateException("Detected possible concurrent process on another cluster");
        }
    }

    private void insideBinlogReader(String db, BinlogReader binlogReader)
            throws InterruptedException {
        TransactionReader transactionReader = new TransactionReader(binlogReader)
                .withMaxEventsPerTransaction(maxEventsPerTransaction);

        boolean firstRead = true;
        try (TraceGuard g = traceHelper.guard("insideBinlogReader", singleton(source.getDbName()))) {
            while (!Thread.currentThread().isInterrupted()) {
                //если поменялся конфиг выходим, внейшний цикл переинициализирует коннекшны к базе и продолжит
                if (dbConfigReset) {
                    waitPreviousFutureAndUpdateState();
                    dbConfigReset = false;
                    logger.debug("Conf reset, break");
                    break;
                }
                StateBound<TransactionReader.TransactionOrDdl> transactionOrDdl = null;
                try {
                    Duration duration = flushDeadline.minus(NanoTimeClock.now());
                    if (!duration.isNegative() && !duration.isZero()) {
                        try (TraceProfile p = Trace.current().profile("readBinlogs")) {
                            if (firstRead && sourceState.getEventsCount() > 0) {
                                transactionReader.skipFirstEventsFromTransaction(duration,
                                        sourceState.getEventsCount());
                            }
                            transactionOrDdl = transactionReader.readTransactionOrDdl(duration);
                            firstRead = false;
                        }
                    }
                } catch (RuntimeTimeoutException ignored) {
                    // Долго не было новых событий. Стоит попробовать ещё раз.
                    // Если соединение порвалось, то BinlogReader должен его переустановить в фоне.
                }
                if (transactionOrDdl != null) {
                    if (transactionOrDdl.getData().getTransaction() != null) {
                        transactionToBinlogEvents(sendQueue, source.getSourceName(), db,
                                transactionReader.getNeedWriteQueryOnFirstQuery(),
                                transactionOrDdl.getData().getTransaction());
                    } else {
                        ddlToBinlogEvent(sendQueue, source.getSourceName(), db,
                                Objects.requireNonNull(transactionOrDdl.getData().getDdl()));
                    }
                }
                if (isTimeToFlush()) {
                    flushDeadline = NanoTimeClock.now().plus(consumerChunkDuration);
                    if (sendQueue.isEmpty()) {
                        logger.debug("Time to flush events but no new events seen.");
                        continue;
                    }
                    waitPreviousFutureAndUpdateState();
                    MySQLBinlogState currentBinlogState = transactionReader.getState().getState();
                    flushEventsAndSaveStateFuture = flushEventsAndSaveState(currentBinlogState.getGtidSet(),
                            transactionReader.getEventsCount(), currentBinlogState.getSerializedServerSchema());
                }
            }
        }
    }

    private CompletableFuture<Integer> writeToLogbroker(List<BinlogWithSeqId> binlogEventsWithId) {
        CompletableFuture<Integer> writeEventsToLogbrokerFuture = supplyWithTrace("writingBinlogEvents",
                () -> binlogEventConsumer.write(binlogEventsWithId));
        CompletableFuture<Integer> writeQueriesToLogbrokerFuture;
        if (writingQueriesEnabled()) {
            writeQueriesToLogbrokerFuture = supplyWithTrace("writingBinlogQueries",
                    () -> binlogQueryConsumer.write(StreamEx.of(binlogEventsWithId)
                            .filter(eventWithId -> eventWithId.event.getWriteQuery()).toList()
                    ));
        } else {
            writeQueriesToLogbrokerFuture = CompletableFuture.completedFuture(0);
        }
        // количество отправленных сообщений можно считать по топикам, а не суммировать, но непонятно, насколько нужно
        return writeEventsToLogbrokerFuture.thenCombine(writeQueriesToLogbrokerFuture, Integer::sum);
    }

    private CompletableFuture<ImmutableSourceState> flushEventsAndSaveState(String gtidToSave,
                                                                            int eventsCount,
                                                                            byte[] serializedServerSchema) {
        List<BinlogEvent> binlogEvents =
                new ArrayList<>(sendQueue);

        sendQueue.clear();
        AtomicLong atomicSeqNo = new AtomicLong(sourceState.getSeqId());
        AtomicLong atomicHwmTimestamp = new AtomicLong(sourceState.getHwmTimestamp());


        Trace currentTrace = Trace.current();

        CompletableFuture<Void> startFuture = new CompletableFuture<>();

        CompletableFuture<ImmutableSourceState> resultFuture = startFuture
                .thenRun(() -> Trace.push(currentTrace))
                .thenApply(unused -> supplyWithTrace("collectEventsToWrite", () -> collectEventsToWrite(binlogEvents,
                        atomicSeqNo, atomicHwmTimestamp)))
                .thenCompose(this::writeToLogbroker)
                // Запись в логброкер происходит в другом executor service, а текущий trace добавлен в
                // flushEventsAndSaveStateExecutorService
                // чтобы в следюущих методах можно было исопльзовать trace,  нужно вернуть выполнение в наш
                // thread pool
                .thenApplyAsync(writingMessagesCnt -> supplyWithTrace("logWritingResult",
                        () -> {
                            logWritingResult(writingMessagesCnt);
                            return supplyWithTrace("saveState", () -> saveState(atomicSeqNo,
                                    atomicHwmTimestamp, gtidToSave, eventsCount, serializedServerSchema));
                        }), flushEventsAndSaveStateExecutorService)
                .thenApply(logbrokerState -> {
                    Trace.pop();
                    return logbrokerState;
                });

        startFuture.completeAsync(() -> null, flushEventsAndSaveStateExecutorService);
        return resultFuture;
    }


    private ImmutableSourceState saveState(AtomicLong atomicSeqNo, AtomicLong atomicHwmTimestamp, String gtidToSave,
                                           int eventsCount, byte[] serializedServerSchema) {
        ImmutableSourceState immutableSourceState = new ImmutableSourceState(atomicSeqNo.get(),
                atomicHwmTimestamp.get(), gtidToSave, eventsCount, serializedServerSchema,
                sourceStateRepository.getClusterName());
        sourceStateRepository.saveState(
                source,
                immutableSourceState
        );
        monitoring.setSourceHighwatermark(source, atomicHwmTimestamp.get());
        return immutableSourceState;
    }

    private boolean isTimeToFlush() {
        return sendQueue.size() >= consumerChunkSize || NanoTimeClock.now().isAtOrAfter(flushDeadline);
    }

    private List<BinlogWithSeqId> collectEventsToWrite(List<BinlogEvent> binlogEvents, AtomicLong seqNo,
                                                       AtomicLong hwmTimestamp) {
        // события из "будущего" не должны апдейтить hwm
        long maxPossibleEventTime = System.currentTimeMillis() / 1000 + 3660;
        List<BinlogWithSeqId> binlogWithQueryWithSeqIds =
                new ArrayList<>(Math.min(consumerChunkSize, sendQueue.size()));

        AtomicInteger queryCnt = new AtomicInteger();
        List<BinlogWithSeqId> binlogWithSeqIds = binlogEvents.stream()
                .peek(event -> {
                            long eventTimestamp = event.getUtcTimestamp().toEpochSecond(ZoneOffset.UTC);
                            if (eventTimestamp > hwmTimestamp.get() && eventTimestamp <= maxPossibleEventTime) {
                                hwmTimestamp.set(eventTimestamp);
                            }
                            if (writingQueriesEnabled() && event.getWriteQuery()) {
                                queryCnt.incrementAndGet();
                            }
                        }
                )
                .map(binlogEvent -> new BinlogWithSeqId(
                        seqNo.incrementAndGet(),
                        binlogEvent))
                .collect(toList());

        if (writingQueriesEnabled()) {
            logger.debug("Writing {} binlog events, {} queries to logbroker", binlogWithSeqIds.size(), queryCnt);
        } else {
            logger.debug("Writing {} binlog events to logbroker", binlogWithSeqIds.size());
        }
        return binlogWithSeqIds;
    }

    private void logWritingResult(int writingMessagesCnt) {
        monitoring.addMessagesCount(source, writingMessagesCnt);
        MonotonicTime now = NanoTimeClock.now();
        logMessageSentCount += writingMessagesCnt;
        if (lastLogMessageCountWasAt.plus(LOG_MESSAGE_WITH_COUNT_PERIOD).isAtOrBefore(now)) {
            logger.info("Written {} messages in {}", logMessageSentCount,
                    now.minus(lastLogMessageCountWasAt));
            lastLogMessageCountWasAt = now;
            logMessageSentCount = 0;
        }
    }

    @SuppressWarnings("unused")
    private <R> R supplyWithTrace(String funcName, Supplier<R> supplier) {
        R res;
        try (TraceProfile profile = Trace.current().profile(funcName)) {
            res = supplier.get();
        }
        return res;
    }

    private void waitPreviousFutureAndUpdateState() throws InterruptedException {
        if (Objects.isNull(flushEventsAndSaveStateFuture)) {
            return;
        }
        try {
            sourceState = flushEventsAndSaveStateFuture.get(flushEventsTimeout.getSeconds(), TimeUnit.SECONDS);
        } catch (ExecutionException e) {
            throw new ExecutionRuntimeException(e);
        } catch (TimeoutException e) {
            throw new TimeoutRuntimeException(e);
        }
    }

    private static void addDirectTraceInfoToEvent(BinlogEvent event, DirectTraceInfo directTraceInfo) {
        event.withTraceInfoReqId(directTraceInfo.getReqId().orElse(0))
                .withTraceInfoService(directTraceInfo.getService().orElse(""))
                .withTraceInfoMethod(directTraceInfo.getMethod().orElse(""))
                .withTraceInfoOperatorUid(directTraceInfo.getOperatorUid().orElse(0))
                .withEssTag(directTraceInfo.getEssTag().orElse(""))
                .withResharding(directTraceInfo.getResharding());
    }

    @Override
    public void update(DbConfigEvent event) {
        var newConnector = dbConfigToConnector(sourceDbConfigSupplier.get());
        logger.debug("new: {}, old: {}", newConnector, connector);
        if(!newConnector.equals(connector)) {
            logger.info("Db config changed {}", newConnector);
            connector = newConnector;
            dbConfigReset = true;
        }
    }

    public static class Builder {
        private SourceType source;
        private Supplier<DbConfig> sourceDbConfigSupplier;
        private LogbrokerWriter<BinlogWithSeqId> binlogEventConsumer;
        private LogbrokerWriter<BinlogWithSeqId> binlogQueryConsumer;
        private SourceStateRepository sourceStateRepository;
        private LogbrokerWriterMonitoring monitoring;
        private boolean initialServerIdSet;
        private Integer initialServerId;
        private Duration keepAliveTimeout;
        private Integer maxBufferedEvents;
        private Integer consumerChunkSize;
        private Integer maxEventsPerTransaction;
        private Duration consumerChunkDuration;
        private MySQLServerBuilder mysqlServerBuilder;
        private Map<String, Set<String>> dontTruncateFields;
        private TraceHelper traceHelper;
        private Duration flushEventsTimeout;

        private Builder() {
        }

        /**
         * Название источника. Например, "ppc:1".
         */
        public Builder withSource(SourceType source) {
            this.source = source;
            return this;
        }

        /**
         * Делегат, который возвращает DbConfig для источника бинлога.
         */
        public Builder withSourceDbConfigSupplier(Supplier<DbConfig> sourceDbConfigSupplier) {
            this.sourceDbConfigSupplier = sourceDbConfigSupplier;
            return this;
        }

        /**
         * Обработчик, который принимает пачки {@link BinlogEvent}.
         */
        public Builder withBinlogEventConsumer(LogbrokerWriter<BinlogWithSeqId> binlogEventConsumer) {
            this.binlogEventConsumer = binlogEventConsumer;
            return this;
        }

        public Builder withBinlogQueryConsumer(@Nullable LogbrokerWriter<BinlogWithSeqId> binlogQueryConsumer) {
            this.binlogQueryConsumer = binlogQueryConsumer;
            return this;
        }

        /**
         * Репозиторий для чтения/записи позиции в бинлоге и иной метаинформации.
         */
        public Builder withSourceStateRepository(SourceStateRepository sourceStateRepository) {
            this.sourceStateRepository = sourceStateRepository;
            return this;
        }

        public Builder withLogbrokerWriterMonitoring(LogbrokerWriterMonitoring monitoring) {
            this.monitoring = monitoring;
            return this;
        }

        /**
         * Явный mysql server id или null, если следует выбрать случайный id при каждом подключении.
         */
        public Builder withInitialServerId(@Nullable Integer initialServerId) {
            initialServerIdSet = true;
            this.initialServerId = initialServerId;
            return this;
        }

        /**
         * Если за указанное время не получено ни одного бинлог-события из источника, соединение с сервером
         * устанавливается повторно.
         */
        public Builder withKeepAliveTimeout(Duration keepAliveTimeout) {
            this.keepAliveTimeout = keepAliveTimeout;
            return this;
        }

        /**
         * Размер буфера для хранения необработанных событий от бинлога.
         */
        public Builder withMaxBufferedEvents(int maxBufferedEvents) {
            this.maxBufferedEvents = maxBufferedEvents;
            return this;
        }

        /**
         * Сколько сообщений за раз передавать на отправку в обработчик.
         */
        public Builder withConsumerChunkSize(int logbrokerChunkSize) {
            this.consumerChunkSize = logbrokerChunkSize;
            return this;
        }

        /**
         * Какое максимальное количество событий читаем из транзакции за раз.
         * Большие транзакции будут отправляться по частям.
         */
        public Builder withMaxEventsPerTransaction(int maxEventsPerTransaction) {
            this.maxEventsPerTransaction = maxEventsPerTransaction;
            return this;
        }

        /**
         * Сколько времени максимум собирать события для отправки в обработчик.
         */
        public Builder withConsumerChunkDuration(Duration logbrokerChunkDuration) {
            this.consumerChunkDuration = logbrokerChunkDuration;
            return this;
        }

        /**
         * Параметры для создания временного mysqld
         */
        public Builder withMysqlServerBuilder(MySQLServerBuilder mysqlServerBuilder) {
            this.mysqlServerBuilder = mysqlServerBuilder;
            return this;
        }

        /**
         * Какие поля в таблицах не следует убирать из бинлога
         */
        public Builder withDontTruncateFields(Set<String> dontTruncateFields) {
            this.dontTruncateFields =
                    dontTruncateFields.stream()
                            .map(String::trim)
                            .filter(s -> !s.isEmpty())
                            .map(pair -> {
                                String[] parts = pair.split("\\.");

                                if (parts.length != 2 || parts[0].isEmpty() || parts[1].isEmpty()) {
                                    throw new IllegalStateException(
                                            "Incorrect dontTruncateFields: " + dontTruncateFields);
                                }

                                return new TableFieldPair(parts[0], parts[1]);
                            })
                            .collect(groupingBy(pair -> pair.table, collectingAndThen(
                                    toList(), list -> list.stream().map(pair -> pair.field).collect(toSet())
                            )));
            return this;
        }

        public Builder withTraceHelper(TraceHelper traceHelper) {
            this.traceHelper = traceHelper;
            return this;
        }

        public Builder withFlushEventsTimeout(Duration flushEventsTimeout) {
            this.flushEventsTimeout = flushEventsTimeout;
            return this;
        }

        private static class TableFieldPair {
            final String table;
            final String field;

            TableFieldPair(String table, String field) {
                this.table = table;
                this.field = field;
            }
        }

        public Binlogbroker build() {
            Objects.requireNonNull(source, "Forgotten source");
            Objects.requireNonNull(sourceDbConfigSupplier, "Forgotten sourceDbConfigSupplier");
            Objects.requireNonNull(binlogEventConsumer, "Forgotten binlogEventConsumer");
            Objects.requireNonNull(sourceStateRepository, "Forgotten sourceStateRepository");
            Objects.requireNonNull(keepAliveTimeout, "Forgotten keepAliveTimeout");
            Objects.requireNonNull(maxBufferedEvents, "Forgotten maxBufferedEvents");
            Objects.requireNonNull(maxEventsPerTransaction, "Forgotten maxEventsPerTransaction");
            Objects.requireNonNull(consumerChunkSize, "Forgotten consumerChunkSize");
            Objects.requireNonNull(consumerChunkDuration, "Forgotten consumerChunkDuration");
            Objects.requireNonNull(mysqlServerBuilder, "Forgotten mysqlServerBuilder");
            Objects.requireNonNull(flushEventsTimeout, "Forgotten flushEventsTimeout");
            Preconditions.checkState(initialServerIdSet, "Forgotten initialServerId");
            Objects.requireNonNull(traceHelper);

            return new Binlogbroker(source,
                    sourceDbConfigSupplier,
                    binlogEventConsumer,
                    binlogQueryConsumer,
                    sourceStateRepository,
                    mysqlServerBuilder,
                    initialServerId,
                    keepAliveTimeout,
                    maxBufferedEvents,
                    consumerChunkSize,
                    maxEventsPerTransaction,
                    consumerChunkDuration,
                    nvl(dontTruncateFields, emptyMap()),
                    monitoring,
                    traceHelper,
                    flushEventsTimeout);
        }
    }
}
