package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Duration;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.SourceDbConfigs;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.mysql.MySQLServerBuilder;
import ru.yandex.direct.tracing.TraceHelper;
import ru.yandex.direct.utils.Completer;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.utils.io.TempDirectory;

/**
 * Запускает отдельный поток для каждого источника бинлога.
 * Если установить у текущего потока флаг interrupted, то все запущенные потоки корректно завершаются.
 * Если хотя бы один из потоков падает с ошибкой, все остальные потоки корректно завершаются, ошибка бросается наружу.
 */
@Component("binlogbrokerSupervisor")
@ParametersAreNonnullByDefault
public class BinlogbrokerSupervisor implements Interrupts.InterruptibleCheckedRunnable<InterruptedException> {
    private static final Logger logger = LoggerFactory.getLogger(BinlogbrokerSupervisor.class);
    private final SourceDbConfigs sourceDbConfigs;
    private final BinlogEventConsumerFactory binlogEventConsumerFactory;
    private final SourceStateRepository sourceStateRepository;
    private final SourceGuardFactory sourceGuardFactory;
    private final UrgentAppDestroyer urgentAppDestroyer;
    private final List<String> dontTruncateFields;
    private final LogbrokerWriterMonitoring monitoring;
    private boolean useTmpfsForMysql;
    private List<String> mysqlBinaryPath;
    private Duration consumerChunkDuration;
    private final TraceHelper traceHelper;
    private final Duration writeTimeout;

    @Autowired
    @SuppressWarnings("checkstyle:parameternumber")
    public BinlogbrokerSupervisor(SourceDbConfigs sourceDbConfigs,
                                  BinlogEventConsumerFactory binlogEventConsumerFactory,
                                  SourceStateRepository sourceStateRepository,
                                  SourceGuardFactory sourceGuardFactory,
                                  UrgentAppDestroyer urgentAppDestroyer,
                                  @Qualifier("logbrokerDontTruncateFields") List<String> dontTruncateFields,
                                  LogbrokerWriterMonitoring monitoring,
                                  @Value("${binlogbroker.use_tmpfs_for_mysql}") boolean useTmpfsForMysql,
                                  @Value("${binlogbroker.write_timeout_sec}") int writeTimeoutSec,
                                  @Qualifier("logbrokerMysqldBinaryPath") List<String> mysqldBinaryPath,
                                  DirectConfig directConfig,
                                  TraceHelper traceHelper
    ) {
        this.sourceDbConfigs = sourceDbConfigs;
        this.binlogEventConsumerFactory = binlogEventConsumerFactory;
        this.sourceStateRepository = sourceStateRepository;
        this.sourceGuardFactory = sourceGuardFactory;
        this.urgentAppDestroyer = urgentAppDestroyer;
        this.dontTruncateFields = dontTruncateFields;
        this.monitoring = monitoring;
        this.useTmpfsForMysql = useTmpfsForMysql;
        this.mysqlBinaryPath = mysqldBinaryPath;
        this.consumerChunkDuration = directConfig.getDuration("binlogbroker.consumer_chunk_duration");
        this.traceHelper = traceHelper;
        this.writeTimeout = Duration.ofSeconds(writeTimeoutSec);
    }

    /**
     * Создаёт временную директорию для работы mysqld
     */
    private TempDirectory createMysqlTempDirectory(String prefix) {
        Path tmpfs = Paths.get("/dev/shm");
        if (useTmpfsForMysql && tmpfs.toFile().isDirectory() && tmpfs.toFile().canWrite()) {
            // Если у нас есть права записи в /dev/shm, то создаём директорию там
            return new TempDirectory(tmpfs, "mysql-binlogbroker-" + prefix);
        } else {
            return new TempDirectory("mysql-binlogbroker-" + prefix);
        }
    }

    @Override
    public void run() throws InterruptedException {
        // Тайм-ауты и прочие настройки взяты с потолка, любой желающий может их поменять или сделать настраиваемыми
        Completer.Builder builder = new Completer.Builder(Duration.ofSeconds(60));
        // Проглатывание InterruptedException в silentRun подходит лучше всего:
        // * Будет меньше больших стектрейсов при получении sigint. Надо ещё позволить классу Binlogbroker бросать
        //   наверх InterruptedException.
        // * Полученный Runnable - единственное, что будет работать внутри потока.
        builder.submitVoid("urgent-app-destroyer", () -> Interrupts.silentRun(urgentAppDestroyer));
        for (SourceType source : sourceDbConfigs.getSources()) {
            Binlogbroker binlogbroker = Binlogbroker.builder()
                    .withSource(source)
                    .withSourceDbConfigSupplier(sourceDbConfigs.getMysqlConnector(source))
                    .withBinlogEventConsumer(binlogEventConsumerFactory.getForSource(source))
                    .withBinlogQueryConsumer(binlogEventConsumerFactory.getQueryWriterForSource(source))
                    .withSourceStateRepository(sourceStateRepository)
                    .withLogbrokerWriterMonitoring(monitoring)
                    .withTraceHelper(traceHelper)
                    .withMysqlServerBuilder(new MySQLServerBuilder()
                            .setMysqldBinary(mysqlBinaryPath)
                            .setGracefulStopTimeout(Duration.ZERO)
                            .addExtraArgs(Collections.singletonList("--skip-innodb-use-native-aio"))
                            .withTempDirectoryProvider(this::createMysqlTempDirectory)
                            .withNoSync(true))
                    .withInitialServerId(null)
                    .withKeepAliveTimeout(Duration.ofSeconds(30))
                    .withMaxBufferedEvents(100)
                    .withConsumerChunkSize(1000)
                    .withMaxEventsPerTransaction(50)
                    .withConsumerChunkDuration(consumerChunkDuration)
                    .withDontTruncateFields(new HashSet<>(dontTruncateFields))
                    .withFlushEventsTimeout(writeTimeout)
                    .build();
            builder.submitVoid("binlogbroker-" + source.getSourceName(), () -> Interrupts.silentRun(
                    new AwaitGuardAndRun(source, sourceGuardFactory, binlogbroker)));
            sourceDbConfigs.addListener(binlogbroker);
        }
        try (Completer completer = builder.build()) {
            completer.waitAll();
        }
    }

    /**
     * Запускает процесс только если успешно удалось взять лок для требуемого {@link SourceType}.
     * Если лок взять не удалось, ждёт какое-то время и пробует заново, так до тех пор, пока поток не прервут.
     */
    private static class AwaitGuardAndRun implements Interrupts.InterruptibleCheckedRunnable<RuntimeException> {
        private static final Duration MIN_AWAIT_BETWEEN_LOCK_ATTEMPTS = Duration.ofSeconds(5);
        private static final Duration MAX_AWAIT_BETWEEN_LOCK_ATTEMPTS = Duration.ofSeconds(20);
        private final SourceType sourceType;
        private final SourceGuardFactory sourceGuardFactory;
        private final Runnable runnable;

        private AwaitGuardAndRun(SourceType source,
                                 SourceGuardFactory sourceGuardFactory,
                                 Runnable runnable) {
            this.sourceType = source;
            this.sourceGuardFactory = sourceGuardFactory;
            this.runnable = runnable;
        }

        @Override
        public void run() throws InterruptedException {
            Optional<SourceGuard> guard;
            do {
                guard = sourceGuardFactory.guard(sourceType);
                if (guard.isPresent()) {
                    logger.info("Successfully got lock for {}", sourceType);
                } else {
                    Duration await = Duration.ofMillis(ThreadLocalRandom.current().nextLong(
                            MIN_AWAIT_BETWEEN_LOCK_ATTEMPTS.toMillis(),
                            MAX_AWAIT_BETWEEN_LOCK_ATTEMPTS.toMillis() + 1));
                    logger.info("Failed to get lock for {}. Will try again after {}", sourceType, await);
                    Thread.sleep(await.toMillis());
                }
            } while (!guard.isPresent());
            runnable.run();
        }
    }
}

