package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import org.apache.commons.io.IOUtils;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.ImmutableSourceState;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.io.FileUtils;

/**
 * Тривиальная реализация загрузки/сохранения стейта путём чтения/записи json-файла в текущем каталоге.
 */
@ParametersAreNonnullByDefault
public class FileSourceStateRepository implements SourceStateRepository {
    private static final ObjectMapper MAPPER = new ObjectMapper()
            .enable(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS)
            .enable(SerializationFeature.INDENT_OUTPUT);

    private static Path makePath(SourceType source) {
        return Paths.get("logbrokerwriter-state." + source.getSourceName() + ".json").toAbsolutePath();
    }

    @Override
    public ImmutableSourceState loadState(SourceType source) {
        Path path = makePath(source);
        try (InputStream inputStream = Files.newInputStream(path)) {
            String raw = IOUtils.toString(inputStream, StandardCharsets.UTF_8);
            if (!raw.trim().isEmpty()) {
                return MAPPER.readValue(raw, ImmutableSourceState.class);
            }
        } catch (NoSuchFileException ignored) {
            // fallthrough
        } catch (IOException exc) {
            throw new IllegalStateException("Error while read " + path, exc);
        }
        return new ImmutableSourceState();
    }

    @Override
    public void saveState(SourceType source, ImmutableSourceState sourceState) {
        String raw = Checked.get(() -> MAPPER.writeValueAsString(sourceState));
        FileUtils.atomicWrite(raw, makePath(source));
    }

    @Override
    public void close() {
        // nothing to close
    }
}
