package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import java.time.Duration;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.juggler.JugglerAsyncMultiSender;
import ru.yandex.direct.juggler.JugglerStatus;
import ru.yandex.direct.juggler.JugglerSubject;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Counter;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.binlogbroker.logbrokerwriter.configuration.BinlogbrokerConfiguration.WORK_MODE_METRIC_REGISTRY;

/**
 * Для каждого источника отправляет в graphite отставание бинлогов в логброкере от реального времени. Актуальным
 * временем в бинлогах считается {@link BinlogEvent#getUtcTimestamp()} из последнего успешно записанного в
 * логброкер события. Название источника берётся из {@link BinlogEvent#getSource()}.
 */
@Component
@Lazy
@ParametersAreNonnullByDefault
@ThreadSafe
public class LogbrokerWriterMonitoring {
    private static final Logger logger = LoggerFactory.getLogger(LogbrokerWriterMonitoring.class);
    private static final Duration MONITORING_INTERVAL = Duration.ofSeconds(15);
    private static final Duration PING_INTERVAL = Duration.ofSeconds(10);
    private static final Duration CRITICAL_LAG = Duration.ofMinutes(10);

    private final String appNameForMonitoring;
    private final Map<SourceType, Long> sourceHwm = new ConcurrentHashMap<>();
    private final Map<SourceType, Long> sourceMessagesCount = new ConcurrentHashMap<>();
    private final Map<SourceType, Counter> skippedByTableCount = new ConcurrentHashMap<>();
    private final Map<BinlogEventForMonitoring, Rate> eventsByTableAndOperationCount = new ConcurrentHashMap<>();
    private final ScheduledExecutorService executer;
    private final JugglerAsyncMultiSender jugglerClient;
    private final MetricRegistry metricRegistry;

    private final JugglerSubject pingSubject;

    private String getJugglerSubjectPrefix() {
        return String.join(".", "app", this.appNameForMonitoring, "lag.");
    }

    @Autowired
    public LogbrokerWriterMonitoring(String appNameForMonitoring, JugglerAsyncMultiSender jugglerClient,
                                     @Qualifier(WORK_MODE_METRIC_REGISTRY) MetricRegistry metricRegistry) {
        this.appNameForMonitoring = appNameForMonitoring;
        this.jugglerClient = jugglerClient;
        this.metricRegistry = metricRegistry;
        this.pingSubject = new JugglerSubject("app." + this.appNameForMonitoring + ".freshness");
        executer = createDefaultExecutor();
    }

    @PostConstruct
    public void init() {
        logger.info("init");
        executer.scheduleAtFixedRate(
                this::jugglerMonitoring,
                MONITORING_INTERVAL.toMillis(),
                MONITORING_INTERVAL.toMillis(),
                TimeUnit.MILLISECONDS
        );

        executer.scheduleAtFixedRate(
                this::ping,
                PING_INTERVAL.toMillis(),
                PING_INTERVAL.toMillis(),
                TimeUnit.MILLISECONDS
        );
    }

    private void ping() {
        jugglerClient.sendEvent(pingSubject.makeEvent(JugglerStatus.OK, "OK"), PING_INTERVAL);
    }

    @PreDestroy
    public void close() {
        logger.info("close");
        if (!executer.isShutdown()) {
            executer.shutdown();
        }
    }

    public void setSourceHighwatermark(SourceType source, long timestamp) {
        if (!sourceHwm.containsKey(source)) {
            sourceHwm.put(source, timestamp);
            metricRegistry.lazyGaugeInt64(
                    "logbrokerwriter_delay_seconds",
                    Labels.of("db", source.getDbName()),
                    () -> calcDelay(sourceHwm.get(source))
            );
        } else {
            sourceHwm.put(source, timestamp);
        }
    }

    void addSkippedByTableCount(SourceType source, int cnt) {
        skippedByTableCount
                .computeIfAbsent(source,
                        s -> metricRegistry.counter("logbrokerwriter_skipped_by_table", Labels.of("db",
                                source.getDbName())))
                .add(cnt);
    }

    public void addMessagesCount(SourceType source, int cnt) {
        if (!sourceMessagesCount.containsKey(source)) {
            metricRegistry.lazyCounter(
                    "logbrokerwriter_written_messages",
                    Labels.of("db", source.getDbName()),
                    () -> sourceMessagesCount.getOrDefault(source, 0L)
            );
        }
        sourceMessagesCount.compute(source, (s, c) -> c == null ? cnt : c + cnt);
    }

    public void addEventByTableAndOperation(SourceType source, String table, Operation operation) {
        // стоит добавить ещё service и method, но в method пока может попадаться мусор (в частности в direct.canvas), который сильно увеличит кардинальность метрик
        var key = new BinlogEventForMonitoring(source, table, operation);
        eventsByTableAndOperationCount
                .computeIfAbsent(key,
                        k -> metricRegistry.rate("logbrokerwriter_binlog_events",
                                Labels.of("db", k.getSource().getDbName(), "table", k.getTable(), "operation", k.getOperation().toString())))
                .inc();
    }

    private void jugglerMonitoring() {
        sourceHwm.forEach((src, hwm) -> {
            JugglerSubject subj = new JugglerSubject(getJugglerSubjectPrefix() + src.getDbName().replace(':', '_'));

            long delay = calcDelay(hwm);
            if (delay > CRITICAL_LAG.getSeconds()) {
                jugglerClient.sendEvent(
                        subj.makeCrit(String.format("Lag for db %s: %ds > %ds", src.getDbName(), delay,
                                CRITICAL_LAG.getSeconds())),
                        MONITORING_INTERVAL
                );
            } else {
                jugglerClient.sendEvent(subj.makeOk(), MONITORING_INTERVAL);
            }
        });
    }


    static long calcDelay(long hwm) {
        return Long.max(0, System.currentTimeMillis() / 1000 - hwm);
    }

    private ScheduledExecutorService createDefaultExecutor() {
        ThreadFactory threadFactory =
                new ThreadFactoryBuilder().setNameFormat("logbrokerwriter-monitoring-%02d").build();
        return Executors.newScheduledThreadPool(2, threadFactory);
    }

    private class BinlogEventForMonitoring {
        private SourceType source;

        private String table;

        private Operation operation;

        public BinlogEventForMonitoring(SourceType source, String table, Operation operation) {
            this.source = source;
            this.table = table;
            this.operation = operation;
        }

        public SourceType getSource() {
            return source;
        }

        public String getTable() {
            return table;
        }

        public Operation getOperation() {
            return operation;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            BinlogEventForMonitoring that = (BinlogEventForMonitoring) o;
            return Objects.equals(source, that.source) && Objects.equals(table, that.table) && operation == that.operation;
        }

        @Override
        public int hashCode() {
            return Objects.hash(source, table, operation);
        }

        @Override
        public String toString() {
            return "BinlogEventForMonitoring{" +
                    "source=" + source +
                    ", table='" + table + '\'' +
                    ", operation=" + operation +
                    '}';
        }
    }
}
