package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import java.time.Duration;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.ImmutableSourceState;
import ru.yandex.direct.utils.NamedThreadFactory;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.direct.binlogbroker.logbrokerwriter.components.LogbrokerWriterMonitoring.calcDelay;

/**
 * Резервное сохранение стейтов по таймеру в отдельном треде.
 * Ошибки сохранения игнорируются, логи доступны с уровнем DEBUG.
 */
@ParametersAreNonnullByDefault
public class ReplicaStateRepository implements SourceStateRepository {
    private static final Logger logger = LoggerFactory.getLogger(ReplicaStateRepository.class);

    private static final Duration SAVE_STATE_TIMEOUT = Duration.ofSeconds(50);
    private static final Duration BACKUP_INTERVAL = Duration.ofMinutes(1);


    private final Map<SourceType, ImmutableSourceState> states;
    private final Map<SourceType, Long> sourceHwm;
    private final SourceStateRepository forward;
    private final ExecutorService executorService;
    private final Timer timer;
    private final MetricRegistry metricRegistry;


    public ReplicaStateRepository(SourceStateRepository sourceStateRepository, MetricRegistry metricRegistry) {
        this.forward = sourceStateRepository;
        this.metricRegistry = metricRegistry;

        states = new ConcurrentHashMap<>();
        sourceHwm = new ConcurrentHashMap<>();
        executorService = Executors.newSingleThreadExecutor(new NamedThreadFactory("replica-state-saver"));
        timer = new Timer(getClass().getSimpleName() + "-Timer", true);

        logger.debug("Start timer to periodic save");
        timer.schedule(new AsyncSaver(), 0, BACKUP_INTERVAL.toMillis());
    }

    @Override
    public ImmutableSourceState loadState(SourceType source) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void saveState(SourceType source, ImmutableSourceState sourceState) {
        states.put(source, sourceState);
    }

    @Override
    public void close() throws Exception {
        timer.cancel();
        executorService.shutdownNow();
    }

    @Override
    public String getClusterName() {
        return null;
    }

    private class AsyncSaver extends TimerTask {
        @Override
        public void run() {
            if (states.isEmpty()) {
                return;
            }

            var futures = EntryStream.of(states)
                    .mapKeyValue(this::saveStateAsync)
                    .toArray(CompletableFuture[]::new);
            CompletableFuture.allOf(futures).join(); // IGNORE-BAD-JOIN DIRECT-149116
        }

        CompletableFuture<Void> saveStateAsync(SourceType source, ImmutableSourceState state) {
            return CompletableFuture.runAsync(() -> forward.saveState(source, state), executorService)
                    // работает как общий таймаут на всю пачку сохранений
                    .orTimeout(SAVE_STATE_TIMEOUT.toMillis(), TimeUnit.MILLISECONDS)
                    .handle((r, ex) -> {
                        if (ex == null) {
                            long timestamp = state.getHwmTimestamp();
                            if (!sourceHwm.containsKey(source)) {
                                sourceHwm.put(source, timestamp);
                                metricRegistry.lazyGaugeInt64(
                                        "replica_delay_seconds",
                                        Labels.of("db", source.getDbName()),
                                        () -> calcDelay(sourceHwm.get(source))
                                );
                            } else {
                                sourceHwm.put(source, timestamp);
                            }
                            logger.debug("Successfully backed up state for {}", source);
                        } else {
                            logger.warn("Failed to back up state for {}: {}", source, ex);
                        }
                        return null;
                    });
        }
    }
}
