package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.ExecutionException;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.ytwrapper.chooser.AppDestroyer;
import ru.yandex.misc.sunMiscSignal.SunMiscSignal;

/**
 * Способ незамедлительно завершить приложение. То же самое можно достичь через ApplicationEvent из spring,
 * но с используемым инструментарием (всё работает в одном {@link ru.yandex.direct.utils.Completer}) такой вариант
 * удобнее.
 * <p>
 * Метод {@link #run()} следует запускать одним из потоков {@link ru.yandex.direct.utils.Completer Completer},
 * а немедленное завершение приложения выполняется ручкой {@link #destroy(Exception)}.
 */
@Component
@Lazy
@ParametersAreNonnullByDefault
@ThreadSafe
public class UrgentAppDestroyer implements Interrupts.InterruptibleCheckedRunnable<ExecutionException>, AppDestroyer {
    private static final Logger logger = LoggerFactory.getLogger(UrgentAppDestroyer.class);
    private final CompletableFuture<RuntimeException> exceptionHolder = new CompletableFuture<>();

    UrgentAppDestroyer() {
        SunMiscSignal.handle("TERM", signalName -> {
            logger.info("Signal {} catched, let's try to shutdown", signalName);
            try {
                destroy(new RuntimeException("Signal TERM received"));
            } catch (Exception e) {
                logger.error("Shutdown error", e);
            }
        });
    }

    /**
     * Завершить все потоки.
     */
    @Override
    public void destroy(Exception exception) {
        if (!exceptionHolder.isDone()) {
            exceptionHolder.completeExceptionally(exception);
            logger.warn("Thread {} urgently terminates application.", Thread.currentThread());
        } else {
            logger.warn("Thread {} tried to interrupt application, but it was terminated earlier. New exception here.",
                    Thread.currentThread(), exception);
        }
    }

    @Override
    public void run() throws ExecutionException, InterruptedException {
        exceptionHolder.get();
    }

    @Override
    public boolean hasError() {
        return exceptionHolder.isCompletedExceptionally();
    }

    @Override
    @Nullable
    public Throwable getError() {
        try {
            exceptionHolder.getNow(null);
        } catch (CompletionException exc) {
            return exc.getCause();
        }
        return null;
    }
}
