package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import net.jcip.annotations.ThreadSafe;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.LockMode;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.common.YtErrorMapping;

import static ru.yandex.direct.binlogbroker.logbrokerwriter.configuration.BinlogbrokerConfiguration.LOCKING_YT_CLIENT;
import static ru.yandex.direct.binlogbroker.logbrokerwriter.configuration.BinlogbrokerConfiguration.LOCKING_YT_CONFIG;

/**
 * Распределённая блокировка через YT.
 */
@Component
@Lazy
@ParametersAreNonnullByDefault
@ThreadSafe
public class YtSourceGuardFactory implements SourceGuardFactory {
    private static final Duration TRANSACTION_TIMEOUT = Duration.ofSeconds(120);
    private static final Duration HEARTBEAT_PERIOD = Duration.ofSeconds(10);
    private final Yt yt;
    private final YPath lockRootPath;
    private final UrgentAppDestroyer urgentAppDestroyer;
    private final Map<SourceType, YtSourceGuard> guards;
    private volatile boolean destroying;
    @Nullable
    private Thread transactionPingerThread;

    @Autowired
    public YtSourceGuardFactory(
            @Qualifier(LOCKING_YT_CONFIG) YtClusterConfig lockingYtConfig,
            @Qualifier(LOCKING_YT_CLIENT) Yt lockingYtClient,
            String ytLocksSubpath,
            EnvironmentType environmentType,
            UrgentAppDestroyer urgentAppDestroyer) {
        this(lockingYtClient,
                YPath.simple(lockingYtConfig.getHome())
                        .child(ytLocksSubpath)
                        .child(environmentType.name().toLowerCase()),
                urgentAppDestroyer);
    }

    public YtSourceGuardFactory(Yt yt, YPath lockRootPath, UrgentAppDestroyer urgentAppDestroyer) {
        this.yt = yt;
        this.lockRootPath = lockRootPath;
        this.urgentAppDestroyer = urgentAppDestroyer;

        // ConcurrentHashMap не подходит, т.к. часто использует оптимистичные блокировки.
        // Например, вызов computeIfAbsent/computeIfPresent при конкурентном доступе может вызвать обработчик
        // несколько раз.
        // Collections.synchronizedMap берёт один общий монитор на всю коллекцию, что намного лучше подходит
        // для нужд этого класса.
        guards = Collections.synchronizedMap(new HashMap<>());
    }

    private YPath lockPath(SourceType source) {
        return lockRootPath.child(source.getSourceName() + "-lock");
    }

    @PostConstruct
    public void init() {
        if (transactionPingerThread == null) {
            // Почему-то этот бин инициализируется два раза, что не является показателем хорошего кода.
            transactionPingerThread = new TransactionPingerThread();
            transactionPingerThread.start();
        }
    }

    @Override
    public Optional<SourceGuard> guard(SourceType source) {
        if (destroying) {
            return Optional.empty();
        }
        Preconditions.checkState(transactionPingerThread != null, "Not initialized");
        YPath lockPath = lockPath(source);
        try {
            yt.cypress().create(lockPath, CypressNodeType.DOCUMENT);
        } catch (YtErrorMapping.AlreadyExists ignored) {
            // Нода уже есть, ничего не надо создавать
        }
        GUID candidateTransaction = yt.transactions().start(TRANSACTION_TIMEOUT);
        YtSourceGuard candidateGuard = new YtSourceGuard(source, candidateTransaction);
        Optional<RuntimeException> lockingExc = Optional.empty();
        guards.computeIfPresent(source, (src, g) -> g.isClosed() ? null : g);
        if (guards.putIfAbsent(source, candidateGuard) == null) {
            try {
                yt.cypress().lock(candidateTransaction, lockPath, LockMode.EXCLUSIVE, false);
                return Optional.of(candidateGuard);
            } catch (YtErrorMapping.ConcurrentTransactionLockConflict ignored) {
                // Лок не был взят, нормальная ситуация.
            } catch (RuntimeException exc) {
                lockingExc = Optional.of(exc);
            }
            // Элемент в guards кладётся через putIfAbsent и это единственное место, где в guards что-то добавляется.
            YtSourceGuard removedGuard = guards.remove(source);
            Preconditions.checkState(candidateGuard == removedGuard,
                    "Bug. Someone replaced guard for %s. Expected that guard is %s, but it was %s",
                    source, candidateGuard, removedGuard);
        }
        try {
            yt.transactions().abort(candidateTransaction);
        } catch (YtErrorMapping.ConcurrentTransactionLockConflict exc) {
            if (!destroying) {
                lockingExc.ifPresent(exc::addSuppressed);
                throw exc;
            }
        } catch (RuntimeException exc) {
            lockingExc.ifPresent(exc::addSuppressed);
            throw exc;
        }
        return Optional.empty();
    }

    @PreDestroy
    public void destroy() {
        if (destroying) {
            return;
        }
        destroying = true;
        Objects.requireNonNull(transactionPingerThread);
        transactionPingerThread.interrupt();
        transactionPingerThread = null;
        RuntimeException rootExc = null;
        synchronized (guards) {
            for (YtSourceGuard guard : guards.values()) {
                try {
                    guard.close();
                } catch (RuntimeException exc) {
                    if (rootExc == null) {
                        rootExc = exc;
                    } else {
                        rootExc.addSuppressed(exc);
                    }
                }
            }
        }
        if (rootExc != null) {
            throw rootExc;
        }
    }

    public class YtSourceGuard implements SourceGuard {
        private final SourceType source;
        private final GUID transaction;
        private volatile boolean closed = false;

        YtSourceGuard(SourceType source, GUID transaction) {
            this.source = source;
            this.transaction = transaction;
        }

        @Override
        public void close() {
            yt.transactions().abort(transaction);
            closed = true;
        }

        public boolean isClosed() {
            return closed;
        }
    }

    /**
     * Поток, который периодически пингует транзакцию, чтобы она не порвалась. Если всё-таки сеть пропала,
     * то поток через {@link UrgentAppDestroyer} побыстрее завершает приложение.
     * Учитывая, что интервал между пингами в несколько раз меньше времени жизни транзакции,
     * времени на завершение приложения должно хватить.
     */
    private class TransactionPingerThread extends Thread {
        TransactionPingerThread() {
            setName(getClass().getSimpleName());
            setDaemon(true);
        }

        @Override
        public void run() {
            try {
                while (!Thread.currentThread().isInterrupted()) {
                    Set<SourceType> sourceTypes = guards.keySet();
                    List<SourceType> sourceTypeList;
                    synchronized (guards) {
                        sourceTypeList = ImmutableList.copyOf(sourceTypes);
                    }

                    for (SourceType sourceType : sourceTypeList) {
                        guards.computeIfPresent(sourceType, (ignored, guard) -> {
                            yt.transactions().ping(guard.transaction);
                            return guard;
                        });
                    }
                    Thread.sleep(HEARTBEAT_PERIOD.toMillis());
                }
            } catch (InterruptedException ignored) {
                Thread.currentThread().interrupt();
            } catch (RuntimeException exc) {
                urgentAppDestroyer.destroy(exc);
            }
        }
    }
}
