package ru.yandex.direct.binlogbroker.logbrokerwriter.components;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PostConstruct;
import javax.annotation.concurrent.ThreadSafe;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.ImmutableSourceState;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.CypressNodeType;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.common.YtErrorMapping;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeIntegerNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeMapNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.YTreeStringNodeImpl;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.transactions.Transaction;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

import static ru.yandex.direct.binlogbroker.logbrokerwriter.configuration.BinlogbrokerConfiguration.LOCKING_YT_CLIENT;
import static ru.yandex.direct.binlogbroker.logbrokerwriter.configuration.BinlogbrokerConfiguration.LOCKING_YT_CONFIG;

/**
 * Хранение {@link ImmutableSourceState} на YT.
 * <p>
 * Маленькие и часто меняющиеся seqNo+gtidSet хранятся в атрибуте ноды. Большая и редко меняющаяся схема хранится в
 * файле. К пути до ноды со стейтом автоматически добавляется тип среды, что даёт возможность хранить на одном сервере
 * без дополнительных настроек стейты для продакшна и тестовых сред.
 */
@Component
@Lazy
@ParametersAreNonnullByDefault
@ThreadSafe
public class YtSourceStateRepository implements SourceStateRepository {
    private static final Logger logger = LoggerFactory.getLogger(YtSourceStateRepository.class);
    private static final String COUNTERS_ATTRIBUTE = "counters";
    private static final String COUNTERS_GTID_SET_FIELD = "set";
    private static final String COUNTERS_SEQ_NO_FIELD = "seq_no";
    private static final String COUNTERS_HWM_TIMESTAMP = "hwm_timestamp";
    private static final String COUNTERS_EVENTS_COUNT_FIELD = "events_count";
    private static final String COUNTERS_CLUSTER_NAME_FIELD = "cluster_name";

    private final Yt yt;
    private final boolean allowEmptyState;
    private final YPath stateRootPath;
    private final Map<SourceType, byte[]> lastSavedSchema;
    private final String clusterName;

    @Autowired
    public YtSourceStateRepository(
            @Qualifier(LOCKING_YT_CONFIG) YtClusterConfig lockingYtConfig,
            @Qualifier(LOCKING_YT_CLIENT) Yt lockingYtClient,
            String ytLocksSubpath,
            EnvironmentType environmentType,
            @Value("${binlogbroker.allow_empty_state}") boolean allowEmptyState) {
        this(lockingYtConfig.getCluster().getName(),
                lockingYtClient,
                YPath.simple(lockingYtConfig.getHome())
                        .child(ytLocksSubpath)
                        .child(environmentType.name().toLowerCase()),
                allowEmptyState);
    }

        private YtSourceStateRepository(@Nullable  String clusterName, Yt yt, YPath stateRootPath, boolean allowEmptyState){
        this.clusterName = clusterName;
        this.yt = yt;
        this.stateRootPath = stateRootPath;
        this.allowEmptyState = allowEmptyState;
        lastSavedSchema = new ConcurrentHashMap<>();
    }

    // для тестов
    YtSourceStateRepository(Yt yt, YPath stateRootPath, boolean allowEmptyState) {
        this(null, yt, stateRootPath, allowEmptyState);
    }

    @Override
    public String getClusterName() {
        return clusterName;
    }

    @PostConstruct
    public void init() {
        try {
            yt.cypress().create(stateRootPath, CypressNodeType.MAP, true);
        } catch (YtErrorMapping.AlreadyExists ignored) {
            // does not matter if node is already created
        }
    }

    private YPath statePath(SourceType source) {
        return stateRootPath.child(source.getSourceName() + "-state");
    }

    @Override
    public ImmutableSourceState loadState(SourceType source) {
        // Загрузка состояния - редкая операция, которая в норме делается один раз при старте приложения.
        // Нечего пытаться оптимизировать секунды на микросекунды.
        YPath path = statePath(source);
        YTreeMapNode countersNode;
        ByteArrayOutputStream serializedSchema = new ByteArrayOutputStream();
        if (allowEmptyState && !yt.cypress().exists(path)) {
            logger.warn("Can't found state for {}. Returning empty state.", source);
            return new ImmutableSourceState();
        }
        yt.files().read(path, s -> IOUtils.copy(s, serializedSchema));
        countersNode = yt.cypress().get(path.attribute(COUNTERS_ATTRIBUTE)).mapNode();

        long seqId;
        long hwmTimestamp;
        int eventNo;
        String gtidSet;
        seqId = countersNode.getOrThrow(COUNTERS_SEQ_NO_FIELD).integerNode().getLong();
        hwmTimestamp = countersNode.getLongO(COUNTERS_HWM_TIMESTAMP).orElse(0L);
        eventNo = countersNode.getIntO(COUNTERS_EVENTS_COUNT_FIELD).orElse(0);
        gtidSet = new String(
                countersNode.getOrThrow(COUNTERS_GTID_SET_FIELD).stringValue().getBytes(),
                StandardCharsets.US_ASCII);
        String clusterName = countersNode.getStringO(COUNTERS_CLUSTER_NAME_FIELD).orElse(null);
        byte[] serializedServerSchema = serializedSchema.toByteArray();
        lastSavedSchema.put(source, serializedServerSchema);
        return new ImmutableSourceState(seqId, hwmTimestamp, gtidSet, eventNo, serializedServerSchema, clusterName);
    }

    @Override
    public void saveState(SourceType source, ImmutableSourceState immutableSourceState) {
        YTreeMapNode counters = new YTreeMapNodeImpl(Cf.map());
        counters.put(COUNTERS_GTID_SET_FIELD,
                new YTreeStringNodeImpl(immutableSourceState.getGtid(), Cf.map()));
        counters.put(COUNTERS_EVENTS_COUNT_FIELD,
                new YTreeIntegerNodeImpl(false, immutableSourceState.getEventsCount(), Cf.map()));
        counters.put(COUNTERS_SEQ_NO_FIELD,
                new YTreeIntegerNodeImpl(false, immutableSourceState.getSeqId(), Cf.map()));
        counters.put(COUNTERS_HWM_TIMESTAMP,
                new YTreeIntegerNodeImpl(false, immutableSourceState.getHwmTimestamp(), Cf.map()));
        counters.put(COUNTERS_CLUSTER_NAME_FIELD, YTree.stringOrNullNode(immutableSourceState.getClusterName()));
        // Если схема не изменилась, то стоит попытаться сохранить маленькие счётчики по-быстрому.
        if (Arrays.equals(immutableSourceState.getSerializedServerSchema(), lastSavedSchema.get(source))) {
            try {
                saveStateQuick(source, counters);
                return;
            } catch (YtErrorMapping.ResolveError ignored) {
                // Ноды нет, её нужно создать. Требуется полное сохранение стейта.
            }
        }
        saveStateFull(source, immutableSourceState.getSerializedServerSchema(), counters);
    }

    /**
     * Быстрый способ - обновить один маленький атрибут с seqNo и gtidSet, который часто меняется.
     * Не тратит время на попытку создать ноду.
     */
    private void saveStateQuick(SourceType source, YTreeMapNode counters) {
        YPath path = statePath(source).attribute(COUNTERS_ATTRIBUTE);
        yt.cypress().set(path, counters);
    }

    /**
     * Медленный способ - обновить маленький атрибут и большой атрибут со схемой, который редко меняется.
     * Предварительно пытается создать ноду.
     */
    @SuppressWarnings("squid:S1141")  // nested try-catch
    private void saveStateFull(SourceType source, @Nullable byte[] serializedSchema, YTreeMapNode counters) {
        YPath path = statePath(source);
        Transaction trx = yt.transactions().startAndGet();
        try {
            Optional<GUID> trxId = Optional.of(trx.getId());
            yt.cypress().create(trxId, true, statePath(source), CypressNodeType.FILE, true, true, Cf.map());
            yt.cypress().set(trxId, true, path.attribute(COUNTERS_ATTRIBUTE), counters);
            try (ByteArrayInputStream stream = new ByteArrayInputStream(serializedSchema)) {
                yt.files().write(trxId, true, path, stream);
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        } catch (RuntimeException exc) {
            trx.abort();
            throw exc;
        }
        trx.commit();

        lastSavedSchema.put(source, serializedSchema);
    }

    @Override
    public void close() {
        // Нечего закрывать
    }
}
