package ru.yandex.direct.binlogbroker.logbrokerwriter.configuration;

import java.time.Duration;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;

import ru.yandex.direct.binlogbroker.logbroker_utils.configuration.LogbrokerCommonConfiguration;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceTypeHelper;
import ru.yandex.direct.binlogbroker.logbrokerwriter.components.AsyncSaveStateRepository;
import ru.yandex.direct.binlogbroker.logbrokerwriter.components.AsyncSaveStateWithReplicaRepository;
import ru.yandex.direct.binlogbroker.logbrokerwriter.components.DataFormat;
import ru.yandex.direct.binlogbroker.logbrokerwriter.components.ReplicaStateRepository;
import ru.yandex.direct.binlogbroker.logbrokerwriter.components.SourceGuardFactory;
import ru.yandex.direct.binlogbroker.logbrokerwriter.components.SourceStateRepository;
import ru.yandex.direct.binlogbroker.logbrokerwriter.components.UrgentAppDestroyer;
import ru.yandex.direct.binlogbroker.logbrokerwriter.components.YtSourceGuardFactory;
import ru.yandex.direct.binlogbroker.logbrokerwriter.components.YtSourceStateRepository;
import ru.yandex.direct.binlogbroker.logbrokerwriter.jcommander.BinlogbrokerToolParameters;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.LogbrokerWriterConfig;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.SourceDbConfigs;
import ru.yandex.direct.binlogbroker.logbrokerwriter.models.WorkMode;
import ru.yandex.direct.common.configuration.MetricsConfiguration;
import ru.yandex.direct.common.configuration.MonitoringHttpServerConfiguration;
import ru.yandex.direct.common.jetty.JettyLauncher;
import ru.yandex.direct.config.DirectConfig;
import ru.yandex.direct.config.EssentialConfiguration;
import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.dbutil.configuration.DbUtilConfiguration;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.juggler.check.configuration.JugglerCheckConfiguration;
import ru.yandex.direct.logging.GlobalCustomMdc;
import ru.yandex.direct.tracing.TraceMdcAdapter;
import ru.yandex.direct.ytwrapper.chooser.WorkModeChooser;
import ru.yandex.direct.ytwrapper.client.YtClusterConfig;
import ru.yandex.direct.ytwrapper.client.YtClusterConfigProvider;
import ru.yandex.direct.ytwrapper.client.YtClusterTypesafeConfigProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.kikimr.persqueue.LogbrokerClientAsyncFactory;
import ru.yandex.kikimr.persqueue.compression.CompressionCodec;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static com.google.common.base.MoreObjects.firstNonNull;
import static java.lang.String.format;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@Configuration
@ComponentScan(
        basePackages = "ru.yandex.direct.binlogbroker",
        excludeFilters = @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION)
)
@Import({
        DbUtilConfiguration.class,
        EssentialConfiguration.class,
        LogbrokerCommonConfiguration.class,
        MetricsConfiguration.class,
        JugglerCheckConfiguration.class,
        MonitoringHttpServerConfiguration.class
})
@ParametersAreNonnullByDefault
public class BinlogbrokerConfiguration {
    public static final String LOCKING_YT_CONFIG = "lockingYtConfig";
    public static final String LOCKING_YT_CLIENT = "lockingYtClient";
    public static final String WORK_MODE_METRIC_REGISTRY = "workModeMetricRegistry";
    private static final Logger logger = LoggerFactory.getLogger(BinlogbrokerConfiguration.class);

    private static BinlogbrokerToolParameters parameters;

    public static void initializeStaticData(BinlogbrokerToolParameters parameters) {
        BinlogbrokerConfiguration.parameters = parameters;
    }

    private static void checkParams() {
        Preconditions.checkState(parameters != null,
                "You should explicitly call `%s`.initializeStaticData`"
                        + " before initialization of spring context.",
                BinlogbrokerConfiguration.class.getCanonicalName());
    }

    @Bean(destroyMethod = "close")
    public SourceStateRepository sourceStateRepository(
            YtSourceStateRepository ytSourceStateRepository,
            ReplicaStateRepository replicaStateRepository) {
        var primary = new AsyncSaveStateRepository(ytSourceStateRepository, Duration.ofMinutes(2));
        return new AsyncSaveStateWithReplicaRepository(primary, replicaStateRepository);
    }

    @Bean
    public SourceGuardFactory sourceGuardFactory(YtSourceGuardFactory ytSourceGuardFactory) {
        return ytSourceGuardFactory;
    }

    @Bean
    public WorkMode WorkMode(WorkModeChooser<WorkMode> workModeChooser) {
        WorkMode workMode = workModeChooser.getOrRequestWorkMode();
        GlobalCustomMdc.put(TraceMdcAdapter.METHOD_KEY, format("binlogbroker-%s", workMode.toString()));
        return workMode;
    }

    @Bean
    public SourceDbConfigs sourceDbConfigs(
            DbConfigFactory dbConfigFactory,
            SourceTypeHelper sourceTypeHelper,
            ShardHelper shardHelper,
            WorkMode workMode
    ) {
        checkParams();

        var totalShards = shardHelper.dbShards();

        List<Integer> shards;
        // если в параметрах явно задан список шардов - берём его
        if (parameters.shards != null && !parameters.shards.isEmpty()) {
            shards = parameters.shards.stream()
                    .filter(totalShards::contains)
                    .distinct()
                    .collect(Collectors.toList());
        } else {
            shards = workMode.getShards(shardHelper.dbShards());
        }

        if (shards.isEmpty() && !workMode.getWritePpcDict()) {
            throw new IllegalStateException("No shard or ppcdict specified in work mode.");
        }
        Set<SourceType> requiredSourceTypes = sourceTypeHelper.getPpcSources(shards);
        if (workMode.getWritePpcDict()) {
            requiredSourceTypes.add(sourceTypeHelper.getPpcdictSource());
        }
        logger.info("setup sourceDbConfigs for workMode {} and sources {}", workMode, requiredSourceTypes);
        return new SourceDbConfigs(requiredSourceTypes, dbConfigFactory);
    }

    /**
     * Конфигурация YT-клиента для хранения небольших атрибутов и локов.
     * На кластере locke нет grpc-прокси, он поддерживает только HTTP-интерфейс.
     */
    @Bean(name = LOCKING_YT_CONFIG)
    public YtClusterConfig lockingYtConfig(
            DirectConfig directConfig,
            @Value("${binlogbroker.yt_lock.cluster_name}") String clusterName) {
        return createYtConfig(directConfig, clusterName);
    }

    private static YtClusterConfig createYtConfig(DirectConfig directConfig, String clusterName) {
        YtClusterConfigProvider provider =
                new YtClusterTypesafeConfigProvider(directConfig.getConfig().getConfig("yt"));
        YtCluster cluster = YtCluster.clusterFromNameToUpperCase(clusterName);
        return provider.get(cluster);
    }

    /**
     * YT-клиент для хранения небольших атрибутов и локов.
     * На кластере locke нет grpc-прокси, он поддерживает только HTTP-интерфейс.
     */
    @Bean(name = LOCKING_YT_CLIENT)
    public Yt lockingYtClient(@Qualifier(LOCKING_YT_CONFIG) YtClusterConfig ytClusterConfig) {
        return createYtClient(ytClusterConfig);
    }

    private static Yt createYtClient(YtClusterConfig ytClusterConfig) {
        return YtUtils.http(YtUtils
                .getDefaultConfigurationBuilder(ytClusterConfig.getProxy(), ytClusterConfig.getToken())
                // Особенность текущей реализации yt-клиента:
                // Первые два повтора выполняются без паузы. Начиная с третьего повтора между повторами появляется
                // монотонно возрастающая пауза.
                .withHeavyCommandsRetries(4)
                .withSimpleCommandsRetries(4));
    }

    @Bean
    public ReplicaStateRepository backupYtClient(
            DirectConfig directConfig,
            EnvironmentType environmentType,
            @Qualifier(WORK_MODE_METRIC_REGISTRY) MetricRegistry metricRegistry,
            @Value("${binlogbroker.yt_lock.cluster_name}") String clusterName,
            @Value("${binlogbroker.replica_cluster_name}") String replicaClusterName,
            String ytLocksSubpath
    ) {
        Preconditions.checkState(!clusterName.equals(replicaClusterName),
                "binlogbroker.replica_cluster_name should differs from binlogbroker.yt_lock.cluster_name");

        var backupClusterConfig = createYtConfig(directConfig, replicaClusterName);
        var backupYt = createYtClient(backupClusterConfig);
        var backupYtStateRepository = new YtSourceStateRepository(backupClusterConfig, backupYt, ytLocksSubpath,
                environmentType, false);
        return new ReplicaStateRepository(backupYtStateRepository, metricRegistry);
    }

    @Bean
    public LogbrokerWriterConfig logbrokerWriterConfig(
            DirectConfig directConfig,
            DataFormat logbrokerDataFormat,
            @Value("${binlogbroker.logbroker.host}") String logbrokerHost,
            @Value("${binlogbroker.logbroker.timeout}") int logbrokerTimeout,
            @Value("${binlogbroker.logbroker.retries}") int logbrokerRetries
    ) {
        LogbrokerWriterConfig logbrokerWriterConfig = new LogbrokerWriterConfig();
        String path = "binlogbroker.logbroker." + logbrokerDataFormat;

        boolean writeMetadata = directConfig.findBoolean(path + ".write_metadata_for_logshatter").orElse(false);
        CompressionCodec codec = codecOrDefault(directConfig, path + ".codec", CompressionCodec.LZOP);

        logbrokerWriterConfig.setLogbrokerHost(logbrokerHost);
        logbrokerWriterConfig.setLogbrokerTimeout(logbrokerTimeout);
        logbrokerWriterConfig.setLogbrokerRetries(logbrokerRetries);
        logbrokerWriterConfig.setWriteMetadataForLogshatter(writeMetadata);
        logbrokerWriterConfig.setCodec(codec);

        Map<String, String> topicForSourcePrefix = directConfig.getBranch(path + ".topic_for_source_prefix").asMap();
        Preconditions.checkState(!topicForSourcePrefix.keySet().isEmpty(),
                "topic_for_source_prefix must not be empty for format %s", logbrokerDataFormat);

        for (Map.Entry<String, String> entry : topicForSourcePrefix.entrySet()) {
            logbrokerWriterConfig.setTopicForSourcePrefix(entry.getKey(), entry.getValue());
        }
        return logbrokerWriterConfig;
    }

    @Bean
    public LogbrokerWriterConfig logbrokerQueryWriterConfig(
            DirectConfig directConfig,
            DataFormat logbrokerDataFormat,
            @Value("${binlogbroker.logbroker.host}") String logbrokerHost,
            @Value("${binlogbroker.logbroker.timeout}") int logbrokerTimeout,
            @Value("${binlogbroker.logbroker.retries}") int logbrokerRetries
    ) {
        LogbrokerWriterConfig logbrokerQueryWriterConfig = new LogbrokerWriterConfig();
        String path = "binlogbroker.logbroker." + logbrokerDataFormat;
        String qwPath = path + ".query_writer";

        if (!directConfig.hasPath(qwPath)) {
            return logbrokerQueryWriterConfig;
        }

        Preconditions.checkState(logbrokerDataFormat == DataFormat.JSON,
                "Writing queries only supported for data_format = json");

        boolean writeMetadata = directConfig
                .findBoolean(path + ".write_metadata_for_logshatter")
                .orElse(false);
        CompressionCodec codec = codecOrDefault(directConfig, path + ".codec", CompressionCodec.LZOP);
        Map<String, String> topicForSourcePrefix = directConfig.getBranch(path + ".topic_for_source_prefix").asMap();

        int qwTimeout = directConfig.findInt(qwPath + ".timeout").orElse(logbrokerTimeout);
        int qwRetries = directConfig.findInt(qwPath + ".retries").orElse(logbrokerRetries);
        boolean qwWriteMetadata = directConfig.findBoolean(qwPath + ".write_metadata_for_logshatter")
                .orElse(writeMetadata);
        CompressionCodec qwCodec = codecOrDefault(directConfig, qwPath + ".codec", codec);

        if (directConfig.hasPath(qwPath + ".topic_for_source_prefix")) {
            var qwTopicForSourcePrefix = directConfig.getBranch(qwPath + ".topic_for_source_prefix").asMap();

            Set<String> topics = new HashSet<>(topicForSourcePrefix.values());
            Set<String> qwTopics = new HashSet<>(qwTopicForSourcePrefix.values());

            qwTopics.retainAll(topics);
            Preconditions.checkState(qwTopics.isEmpty(), "Topics for binlog events and for queries should not" +
                    " intersect");
            for (Map.Entry<String, String> entry : qwTopicForSourcePrefix.entrySet()) {
                logbrokerQueryWriterConfig.setTopicForSourcePrefix(entry.getKey(), entry.getValue());
            }
        }

        // В принципе, можно добавить возможность писать запросы в другой хост,
        // но нужно добавлять ещё инстанс LogbrokerClientAsyncFactory.
        logbrokerQueryWriterConfig.setLogbrokerHost(logbrokerHost);
        logbrokerQueryWriterConfig.setLogbrokerTimeout(qwTimeout);
        logbrokerQueryWriterConfig.setLogbrokerRetries(qwRetries);
        logbrokerQueryWriterConfig.setWriteMetadataForLogshatter(qwWriteMetadata);
        logbrokerQueryWriterConfig.setCodec(qwCodec);
        return logbrokerQueryWriterConfig;
    }

    @Bean
    public LogbrokerClientAsyncFactory logbrokerClientFactory(
            @Value("${binlogbroker.logbroker.host}") String logbrokerHost
    ) {
        ProxyBalancer proxyBalancer = new ProxyBalancer(logbrokerHost);
        return new LogbrokerClientAsyncFactory(proxyBalancer);
    }

    @Bean
    public String logbrokerSourceIdPrefix() {
        checkParams();
        // В логброкер нельзя писать сообщения с одним sourceId в разные партиции
        // К примеру, если когда-то что-то писалось в партицию 24 с sourceId="mySource",
        // то в следующий раз при попытке записи с sourceId="mySource" в партицию 25 будет ошибка.
        // Поэтому для дебага желательно использовать кастомные префиксы, а в продакшене при изменении
        // соответствия шардов-партиций установить новый префикс для всех записей.
        return parameters.logbrokerSourceIdPrefix.isEmpty() ? "defaultPrefix:" : parameters.logbrokerSourceIdPrefix;
    }

    @Bean("logbrokerDontTruncateFields")
    public List<String> logbrokerDontTruncateFields(DirectConfig directConfig) {
        checkParams();
        List<String> fromConfig = directConfig.getStringList("binlogbroker.dont_truncate_fields");
        List<String> fromCommandLine = ifNotNull(parameters.dontTruncateFields, s -> Arrays.asList(s.split(",")));

        // Значение из командной строки приоритетнее
        return firstNonNull(fromCommandLine, fromConfig);
    }

    @Bean("logbrokerMysqldBinaryPath")
    public List<String> logbrokerMysqldBinaryPath(DirectConfig directConfig) {
        return directConfig.getStringList("binlogbroker.mysqld_binary_path");
    }

    @Bean("logbrokerDataFormat")
    public DataFormat logbrokerDataFormat(WorkMode workMode) {
        return workMode.getDataFormat();
    }

    @Bean("appNameForMonitoring")
    public String appNameForMonitoring(DirectConfig directConfig, DataFormat logbrokerDataFormat) {
        return directConfig.getString("binlogbroker.app_name_for_monitoring." + logbrokerDataFormat);
    }

    @Bean
    public String ytLocksSubpath(DirectConfig directConfig, DataFormat logbrokerDataFormat) {
        return directConfig.getString("binlogbroker.yt_lock.locks_subpath." + logbrokerDataFormat);
    }

    @Bean(WORK_MODE_METRIC_REGISTRY)
    public MetricRegistry workModeMetricRegistry(WorkMode workMode) {
        var subRegistry = SOLOMON_REGISTRY.subRegistry(
                Labels.of("binlogbroker_chunk_id", String.valueOf(workMode.getChunkId()),
                        "binlogbroker_format", workMode.getDataFormat().toString()
                ));

        long workStartTime = System.nanoTime();
        subRegistry.lazyCounter("work_time", () -> TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - workStartTime));
        return subRegistry;
    }

    @Bean
    public WorkModeChooser<WorkMode> workModeChooser(
            @Qualifier(LOCKING_YT_CONFIG) YtClusterConfig lockingYtConfig,
            @Value("${binlogbroker.yt_lock.work_mode_locks_subpath}") String workModeLocksSubpath,
            EnvironmentType environmentType,
            UrgentAppDestroyer appDestroyer,
            // нужны инициализированные бины, чтобы запускать uptime и другие метрики до взятия лока
            JettyLauncher monitoringHttpServer,
            MetricsConfiguration metricsConfiguration
    ) {
        return new WorkModeChooser<>(
                lockingYtConfig,
                workModeLocksSubpath,
                environmentType,
                appDestroyer,
                Arrays.asList(WorkMode.values())
        );
    }


    private CompressionCodec codecOrDefault(DirectConfig config, String path, CompressionCodec defaultCodec) {
        String codecStr = config.findString(path).orElse(defaultCodec.toString());
        // всё же кидает исключение, если в конфиге было что-то не то
        return CompressionCodec.valueOf(codecStr.toUpperCase());
    }
}
