package ru.yandex.direct.binlogbroker.logbrokerwriter.models;

import java.util.Arrays;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonGetter;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * Состояние binlogbroker
 * <p>
 * Текущая схема предусматривает, что каждый источник данных синхронизируется не более чем одним процессом.
 * При этом синхронизация проводится в логброкер, а логброкер поддерживает разные партиции, разные source id.
 * <p>
 * Также у каждой партиции логброкера может быть несколько source id, у каждого свой seqNo. Для каждого источника
 * выбирается свой source id, потому каждому источника гарантированно соответствует свой seqNo. Технически можно
 * использовать несколько source id для одного источника. На деле менять source id имеет смысл либо при переполнении
 * seqNo (редкая операция, можно поправить seqNo в стейте вручную), либо для разработки/отладки. Поэтому поддержка
 * стейтов для разных source id у одного источника отсутствует.
 * <p>
 * eventsCount - это количество отправленных в logbroker событий с данными (INSERT, UPDATE, DELETE) последней транзакции.
 * Используется, когда в транзакции оказалось слишком много событий и пришлось отправлять ее по частям.
 * Почти всегда равняется 0, т.е. транзакция была отправлена целиком.
 * <p>
 * Стейт состоит из шести компонент:
 * <ul>
 * <li>seqNo - 8 байт</li>
 * <li>hwmTimestamp - 8 байт</li>
 * <li>gtidSet - небольшая строка</li>
 * <li>eventsCount - 4 байта</li>
 * <li>serializedServerSchema - большая развесистая структура, которая в json занимает сотни килобайт</li>
 * <li>clusterName</li> — строка с именем текущего кластера хранения (и взятия блокировок) стейта
 * </ul>
 */

@Immutable
public class ImmutableSourceState {
    private final long seqId;
    private final long hwmTimestamp;
    private final String gtid;
    private final byte[] serializedServerSchema;
    private final int eventsCount;
    private final String clusterName;

    public ImmutableSourceState() {
        seqId = 0L;
        hwmTimestamp = 0L;
        gtid = null;
        eventsCount = 0;
        serializedServerSchema = null;
        clusterName = null;
    }

    // для тестов
    public ImmutableSourceState(long seqId, long hwmTimestamp, @Nullable String gtid, int eventsCount,
                                @Nullable byte[] serializedServerSchema) {
        this(seqId, hwmTimestamp, gtid, eventsCount, serializedServerSchema, null);
    }

    @JsonCreator
    public ImmutableSourceState(
            @JsonProperty("seq_id") long seqId,
            @JsonProperty("hwm_timestamp") long hwmTimestamp,
            @JsonProperty("gtid") @Nullable String gtid,
            @JsonProperty("events_count") int eventsCount,
            @JsonProperty("serializedServerSchema") @Nullable byte[] serializedServerSchema,
            @JsonProperty("cluster_name") String clusterName
    ) {
        this.seqId = seqId;
        this.hwmTimestamp = hwmTimestamp;
        this.gtid = gtid;
        this.eventsCount = eventsCount;
        this.serializedServerSchema = serializedServerSchema;
        this.clusterName = clusterName;
    }

    /**
     * @return logbroker logbroker sequence id от сообщения, которое было записано последним. Если ничего не было
     * записано, то 0.
     */

    @JsonGetter("seq_id")
    public long getSeqId() {
        return seqId;
    }

    /**
     * @return greatest timestamp of events
     */
    @JsonGetter("hwm_timestamp")
    public long getHwmTimestamp() {
        return hwmTimestamp;
    }

    /**
     * @return gtid_set, который был обработан последним
     */
    @JsonGetter("gtid")
    public String getGtid() {
        return gtid;
    }

    /**
     * @return events_count, это количество отправленных в logbroker событий последней транзакции при отправке по частям
     */
    @JsonGetter("events_count")
    public int getEventsCount() {
        return eventsCount;
    }

    /**
     * @return схема БД на момент последнего обработанного gtid_set
     */
    @JsonGetter("serializedServerSchema")
    public byte[] getSerializedServerSchema() {
        return serializedServerSchema;
    }

    @JsonGetter("cluster_name")
    public String getClusterName() {
        return clusterName;
    }

    public boolean isMysqlStateEmpty() {
        return Objects.isNull(serializedServerSchema) || Objects.isNull(gtid);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ImmutableSourceState that = (ImmutableSourceState) o;
        return seqId == that.seqId &&
                hwmTimestamp == that.hwmTimestamp &&
                Objects.equals(gtid, that.gtid) &&
                Objects.equals(clusterName, that.clusterName) &&
                eventsCount == that.eventsCount &&
                Arrays.equals(serializedServerSchema, that.serializedServerSchema);
    }

    @Override
    public int hashCode() {
        int result = Objects.hash(seqId, hwmTimestamp, gtid, clusterName);
        result = 31 * result + Arrays.hashCode(serializedServerSchema);
        return result;
    }

    @Override
    public String toString() {
        return "ImmutableSourceState{" +
                "seqId=" + seqId +
                ", hwmTimestamp=" + hwmTimestamp +
                ", gtid='" + gtid + '\'' +
                ", serializedServerSchema=" + Arrays.toString(serializedServerSchema) +
                ", clusterName='" + clusterName + '\'' +
                '}';
    }
}
