package ru.yandex.direct.binlogbroker.logbrokerwriter.models;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.beust.jcommander.Parameter;

import ru.yandex.kikimr.persqueue.compression.CompressionCodec;

@ParametersAreNonnullByDefault
public class LogbrokerWriterConfig {
    private String logbrokerHost = null;
    private Map<String, String> topicForSourcePrefix;
    private Map<String, String> identForSourcePrefix;
    private Map<String, String> logTypeForSourcePrefix;
    private int logbrokerTimeout = 30;
    private int logbrokerRetries = 15;
    private boolean writeMetadataForLogshatter = false;
    private String ident;
    private String logType;
    private CompressionCodec codec = CompressionCodec.LZOP;

    public LogbrokerWriterConfig() {
        this.topicForSourcePrefix = new HashMap<>();
        this.identForSourcePrefix = new HashMap<>();
        this.logTypeForSourcePrefix = new HashMap<>();
    }

    @Parameter(
            names = {"--logbroker-host"},
            required = true
    )
    public void setLogbrokerHost(String logbrokerHost) {
        this.logbrokerHost = logbrokerHost;
    }

    public String getLogbrokerHost() {
        return logbrokerHost;
    }

    // Префикс здесь -- ppc для ppc:N и ppcdict для ppcdict. Для более вложенных конфигураций вроде ppc:1:bs поведение не определено
    // Имеет смысл оптимизировать случай "все источники пишутся в один топик", но пока жертвуем удобством ради простоты реализации
    public void setTopicForSourcePrefix(String prefix, String topic) {
        // logType и ident используются, если writeMetadataForLogshatter истинно
        String logType = topic.substring(topic.lastIndexOf('/') + 1);
        int identEndIndex = topic.lastIndexOf('/');
        String ident;
        if (identEndIndex == -1) {
            // топики могут указываться через -- вместо /, например, direct--direct-ppcdata-binlog-log
            // для protobuf-версии, где используется --, ident не отсылается, поэтому делаем тут "чтобы не падало"
            // TODO обрабатывать корректно и такие разделители
            ident = "";
        } else {
            ident = topic.substring(0, identEndIndex);
        }
        topicForSourcePrefix.put(prefix, topic);
        identForSourcePrefix.put(prefix, ident);
        logTypeForSourcePrefix.put(prefix, logType);
    }

    private String getPrefixForSource(String source) {
        String sourceWithoutEnv = source.split(":", 2)[1];
        String[] prefixes = topicForSourcePrefix.keySet().stream().filter(s -> sourceWithoutEnv.split(":")[0].equals(s)).toArray(String[]::new);
        if (prefixes.length > 1) {
            throw new IllegalArgumentException(String.format("More than 1 prefix for source '%s'", sourceWithoutEnv));
        } else if (prefixes.length == 0) {
            throw new IllegalArgumentException(String.format("No prefix for source '%s'", sourceWithoutEnv));
        }
        return prefixes[0];
    }

    public String getIdentForSource(String source) {
        return identForSourcePrefix.get(getPrefixForSource(source));
    }

    public String getLogTypeForSource(String source) {
        return logTypeForSourcePrefix.get(getPrefixForSource(source));
    }

    public String getTopicForSource(String source) {
        return topicForSourcePrefix.get(getPrefixForSource(source));
    }

    public Set<String> getTopics() {
        return new HashSet<>(topicForSourcePrefix.values());
    }

    @Parameter(
            names = {"--logbroker-timeout"},
            description = "Logbroker network timeout in seconds (default 30)"
    )
    public void setLogbrokerTimeout(int logbrokerTimeout) {
        this.logbrokerTimeout = logbrokerTimeout;
    }

    public int getLogbrokerTimeout() {
        return logbrokerTimeout;
    }

    @Parameter(
            names = {"--logbroker-retries"},
            description = "Logbroker network retries (default 15)"
    )
    public void setLogbrokerRetries(int logbrokerRetries) {
        this.logbrokerRetries = logbrokerRetries;
    }

    public int getLogbrokerRetries() {
        return logbrokerRetries;
    }

    @Parameter(
            names = {"--logbroker-write-metadata-for-logshatter"},
            description = "Write metadata for Logshatter to Logbroker (ident and log_type, like push-client, default false)"
    )
    public void setWriteMetadataForLogshatter(boolean writeLogType) {
        this.writeMetadataForLogshatter = writeLogType;
    }

    public boolean getWriteMetadataForLogshatter() {
        return writeMetadataForLogshatter;
    }

    @Parameter(
            names = {"--logbroker-compression-codec"},
            description = "LZOP, GZIP or RAW, default LZOP"
    )
    public void setCodec(CompressionCodec codec) {
        this.codec = codec;
    }

    public CompressionCodec getCodec() { return codec; }
}
