package ru.yandex.direct.binlogbroker.ytbootstrap.components;

import java.time.Duration;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlogbroker.replicatetoyt.DummyStateManager;
import ru.yandex.direct.binlogbroker.replicatetoyt.YtReplicator;
import ru.yandex.direct.dbutil.wrapper.DatabaseWrapper;
import ru.yandex.direct.utils.BlockingChannel;
import ru.yandex.direct.utils.Checked;
import ru.yandex.direct.utils.Completer;
import ru.yandex.direct.utils.Interrupts.InterruptibleConsumer;

import static java.util.stream.Collectors.toSet;

@ParametersAreNonnullByDefault
public class DatabaseYtBootstrap {

    private static final Logger logger = LoggerFactory.getLogger(DatabaseYtBootstrap.class);

    /**
     * Время ожидания насильного завершения всех задач в {@link Completer}, значение взято с потолка
     */
    private static final Duration COMPLETER_CLOSE_TIMEOUT = Duration.ofSeconds(10);

    private final Function<String, DatabaseWrapper> databaseWrapperProvider;
    private final String source;
    private final int maxMySqlReaderThreads;
    private final int batchRows;
    private final int batchEvents;

    private final YtReplicator ytReplicator;
    private final int maxYtReplicatorQueueSize;
    private final int maxYtReplicatorThreads;

    private Set<String> includeTables;
    private final Set<String> excludeTables;
    private final long maxFetchRows;

    private final DatabaseYtBootstrapStateManager stateManager;

    @SuppressWarnings("checkstyle:parameternumber")
    public DatabaseYtBootstrap(
            Function<String, DatabaseWrapper> databaseWrapperProvider, String source, int maxMySqlReaderThreads,
            int batchRows, int batchEvents, YtReplicator ytReplicator, int maxYtReplicatorQueueSize,
            int maxYtReplicatorThreads, Set<String> includeTables, Set<String> excludeTables,
            long maxFetchRows) {
        this.databaseWrapperProvider = databaseWrapperProvider;
        this.source = source;
        this.maxMySqlReaderThreads = maxMySqlReaderThreads;
        this.batchRows = batchRows;
        this.batchEvents = batchEvents;
        this.ytReplicator = ytReplicator;
        this.maxYtReplicatorQueueSize = maxYtReplicatorQueueSize;
        this.maxYtReplicatorThreads = maxYtReplicatorThreads;
        this.includeTables = includeTables;
        this.excludeTables = excludeTables;
        this.stateManager = new DatabaseYtBootstrapStateManager(source, batchRows, batchEvents);
        this.maxFetchRows = maxFetchRows;
    }

    public String getSource() {
        return source;
    }

    private List<TableYtBootstrap> initMySqlReaders(@Nullable DatabaseYtBootstrapState state) {
        // таблицы сортируем по общему размеру данных, чтобы самые большие читались первыми
        final DatabaseWrapper wrapper = databaseWrapperProvider.apply(source);
        final String querySortedTablesBySize = "select TABLE_NAME, DATA_LENGTH "
                + "from information_schema.tables "
                + "where TABLE_SCHEMA = 'ppc' "
                + "order by DATA_LENGTH desc";
        final List<String> tables = wrapper.query(querySortedTablesBySize, (rs, rowNum) -> rs.getString(1));

        List<TableYtBootstrap> result = new ArrayList<>(includeTables.size());
        for (String table : tables) {
            if ((includeTables.isEmpty() || includeTables.contains(table)) && !excludeTables.contains(table)) {
                final TableYtBootstrap tableYtBootstrap = new TableYtBootstrap(
                        databaseWrapperProvider, source, table, batchRows, batchEvents, maxFetchRows);
                if (state != null) {
                    final DatabaseYtBootstrapState.TableState tableState = state.tables.get(table);
                    if (tableState != null) {
                        tableYtBootstrap.setFetchStart(tableState.lastReadPrimaryKey.entrySet());
                        tableYtBootstrap.setFetchFirst(
                                tableState.batchesInProgress.stream()
                                        .map(Map::entrySet)
                                        .collect(toSet()),
                                state.batchRows, state.batchEvents);
                    }
                }
                result.add(tableYtBootstrap);
            }
        }
        return result;
    }

    public void run() throws InterruptedException {
        final List<TableYtBootstrap> mySqlReaders = initMySqlReaders(DatabaseYtBootstrapStateManager.loadState(source));

        createTables(mySqlReaders);

        fetchData(mySqlReaders);
    }

    private void createTables(List<TableYtBootstrap> mySqlReaders) throws InterruptedException {
        for (TableYtBootstrap mySqlReader : mySqlReaders) {
            ytReplicator.acceptDDL(mySqlReader.createTableEvent(), DummyStateManager.SHARD_OFFSET_SAVER);
        }
    }

    /**
     * создает две группы потоков:
     * 1) читает данные из MySQL таблиц (по одному читателю на таблицу)
     * 2) записывает пачки событий в YT
     * Группы соединены между собой блокирующей очередью eventsQueue.
     */
    private void fetchData(List<TableYtBootstrap> mySqlReaders) throws InterruptedException {
        BlockingChannel<TableYtBootstrap> fetchersQueue = new BlockingChannel<>(new ArrayDeque<>(mySqlReaders));
        fetchersQueue.close(); // закрываем сразу, чтобы Workers "помирали", когда не останется задач

        final BlockingChannel<List<BinlogEvent>> eventsQueue = new BlockingChannel<>(maxYtReplicatorQueueSize);

        final WorkerGroup workerGroup = new WorkerGroup(maxMySqlReaderThreads, eventsQueue::close);

        final Completer.Builder builder = new Completer.Builder(COMPLETER_CLOSE_TIMEOUT);
        for (int i = 0; i < maxMySqlReaderThreads; i++) {
            builder.submitVoid("MySQLReader" + i,
                    new Worker<TableYtBootstrap, RuntimeException>(
                            t -> t.fetchData(chain(stateManager::batchQueued, eventsQueue::put)),
                            fetchersQueue,
                            workerGroup
                    )
            );
        }
        for (int i = 0; i < maxYtReplicatorThreads; i++) {
            builder.submitVoid("YtReplicator" + i,
                    new Worker<>(
                            chunk -> {
                                ytReplicator.acceptDML(chunk, DummyStateManager.SHARD_OFFSET_SAVER);
                                stateManager.batchFinished(chunk);
                            },
                            eventsQueue));
        }
        try (Completer completer = builder.build()) {
            completer.waitAll();
        }
    }

    /**
     * вспомогательный метод для последовательного объединения двух {@link InterruptibleConsumer} в один
     */
    private static <T> InterruptibleConsumer<T> chain(InterruptibleConsumer<T> first, InterruptibleConsumer<T> second) {
        return t -> {
            first.accept(t);
            second.accept(t);
        };
    }

    /**
     * Вспомогательный класс для выполнения заданий внутри {@link Completer}.
     * Задания берутся из {@link BlockingChannel}, указанного при создании.
     * Работа прекращается, когда {@link BlockingChannel} пуст и закрыт.
     */
    private static class Worker<T, E extends Exception> implements Checked.CheckedRunnable<E> {

        private final InterruptibleConsumer<T> executor;
        private final BlockingChannel<T> tasks;
        @Nullable
        private final Consumer<Worker> onFinishListener;

        Worker(InterruptibleConsumer<T> executor, BlockingChannel<T> tasks) {
            this.executor = executor;
            this.tasks = tasks;
            this.onFinishListener = null;
        }

        Worker(InterruptibleConsumer<T> executor, BlockingChannel<T> tasks,
               Consumer<Worker> onFinishListener) {
            this.executor = executor;
            this.tasks = tasks;
            this.onFinishListener = onFinishListener;
        }

        @Override
        public void run() throws E {
            try {
                while (!Thread.currentThread().isInterrupted()) {
                    final Optional<T> task = tasks.next();
                    if (task.isPresent()) {
                        executor.accept(task.get());
                    } else {
                        // канал закрыт и пуст, больше ждать нечего
                        if (onFinishListener != null) {
                            onFinishListener.accept(this);
                        }
                        return;
                    }
                }
                logger.debug("Worker " + Thread.currentThread().getName() + " is interrupted by flag");
            } catch (InterruptedException e) {
                // Предполагается, что после этого кода ничего работать не будет.
                logger.debug("Worker " + Thread.currentThread().getName() + " has caught InterruptedException", e);
                Thread.currentThread().interrupt();
            }
        }
    }

    private static class WorkerGroup implements Consumer<Worker> {
        private final AtomicInteger workersCount;
        private final Runnable onFinishListener;

        WorkerGroup(int workersCount, Runnable onFinishListener) {
            this.workersCount = new AtomicInteger(workersCount);
            this.onFinishListener = onFinishListener;
        }

        @Override
        public void accept(Worker worker) {
            if (workersCount.decrementAndGet() == 0) {
                onFinishListener.run();
            }
        }
    }
}
