package ru.yandex.direct.binlogbroker.replicatetoyt.components;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Iterators;
import com.google.common.collect.PeekingIterator;

import ru.yandex.direct.binlog.model.BinlogEvent;
import ru.yandex.direct.binlog.model.Operation;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.BinlogEventWithOffset;
import ru.yandex.direct.binlogbroker.replicatetoyt.StateManager;
import ru.yandex.direct.binlogbroker.replicatetoyt.YtReplicator;
import ru.yandex.direct.utils.Interrupts;

@ParametersAreNonnullByDefault
public class EventConsumer implements Interrupts.InterruptibleConsumer<List<BinlogEventWithOffset>> {
    private final YtReplicator ytReplicator;
    private final StateManager stateManager;

    public EventConsumer(YtReplicator ytReplicator, StateManager stateManager) {
        this.ytReplicator = ytReplicator;
        this.stateManager = stateManager;
    }

    static List<EventChunk> splitEvents(Iterator<BinlogEventWithOffset> eventWithOffsetIter) {
        List<EventChunk> result = new ArrayList<>();
        PeekingIterator<BinlogEventWithOffset> iter = Iterators.peekingIterator(eventWithOffsetIter);
        while (iter.hasNext()) {
            BinlogEventWithOffset eventWithOffset = iter.next();
            BinlogEvent event = eventWithOffset.getEvent();
            long offset = eventWithOffset.getOffset();
            List<BinlogEvent> chunk;
            EventType eventType;
            if (event.getOperation() == Operation.SCHEMA) {
                eventType = EventType.DDL;
                chunk = Collections.singletonList(event);
            } else {
                eventType = EventType.DML;
                chunk = new ArrayList<>();
                chunk.add(event);
                while (iter.hasNext() && iter.peek().getEvent().getOperation() != Operation.SCHEMA) {
                    eventWithOffset = iter.next();
                    chunk.add(eventWithOffset.getEvent());
                    offset = eventWithOffset.getOffset();
                }
            }
            result.add(new EventChunk(eventType, offset, chunk));
        }
        return result;
    }

    private Iterator<BinlogEventWithOffset> filterOldOffsets(Iterator<BinlogEventWithOffset> eventWithOffsetIter) {
        Map<String, Optional<Long>> stateMap = new HashMap<>();
        return Iterators.filter(eventWithOffsetIter, eventWithOffset -> stateMap
                .computeIfAbsent(Objects.requireNonNull(eventWithOffset).getEvent().getSource(),
                        stateManager::getShardOffset)
                .map(storedOffset -> storedOffset < eventWithOffset.getOffset())
                .orElse(true));
    }

    /**
     * Обработка пачки событий. Те события, которые уже применялись, игнорируются.
     */
    @Override
    public void accept(List<BinlogEventWithOffset> eventWithOffsetList) throws InterruptedException {
        for (EventChunk eventChunk : splitEvents(filterOldOffsets(eventWithOffsetList.iterator()))) {
            String source = eventChunk.events.get(0).getSource();
            StateManager.ShardOffsetSaver shardOffsetSaver =
                    stateManager.shardOffsetSaver(source, eventChunk.offsetOfLastEvent);
            switch (eventChunk.eventType) {
                case DDL:
                    ytReplicator.acceptDDL(eventChunk.events.get(0), shardOffsetSaver);
                    break;
                case DML:
                    ytReplicator.acceptDML(eventChunk.events, shardOffsetSaver);
                    break;
                default:
                    throw new IllegalStateException(String.format("can't handle %s", eventChunk.eventType));
            }
        }
    }

    enum EventType {
        DDL,
        DML
    }

    static class EventChunk {
        final EventType eventType;
        final long offsetOfLastEvent;
        final List<BinlogEvent> events;

        EventChunk(@Nonnull EventType eventType, long offsetOfLastEvent, @Nonnull List<BinlogEvent> events) {
            this.eventType = eventType;
            this.offsetOfLastEvent = offsetOfLastEvent;
            this.events = events;
        }
    }
}
