package ru.yandex.direct.binlogbroker.replicatetoyt.components;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.ThreadSafe;

import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.BinlogEventWithOffset;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;

import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;

@Component
@Lazy
@ParametersAreNonnullByDefault
@ThreadSafe
public class YtWriterMetricProvider {
    private static final Logger logger = LoggerFactory.getLogger(YtWriterMetricProvider.class);
    private final Object monitor = new Object();
    private final Map<String, LocalDateTime> lastWrittenEventUtcTimestampForSource = new HashMap<>();
    /**
     * (source, timestamp) -> count
     */
    private final Map<Pair<String, Long>, Long> writtenInSecond = new HashMap<>();

    public Interrupts.InterruptibleConsumer<List<BinlogEventWithOffset>> wrap(
            Interrupts.InterruptibleConsumer<List<BinlogEventWithOffset>> forward) {
        return events -> {
            forward.accept(events);
            if (!events.isEmpty()) {
                Map<String, LocalDateTime> newTimestamps = new HashMap<>();
                Map<Pair<String, Long>, Long> addToWrittenInSecond = new HashMap<>();
                long timestamp = System.currentTimeMillis() / 1_000;
                for (BinlogEventWithOffset event : events) {
                    newTimestamps.merge(event.getEvent().getSource(),
                            event.getEvent().getUtcTimestamp(),
                            (l, r) -> l.isBefore(r) ? r : l);
                    long rows = event.getEvent().getRows().size();
                    addToWrittenInSecond.merge(
                            Pair.of(event.getEvent().getSource(), timestamp),
                            rows,
                            (counter, ignored) -> counter + rows);
                }
                synchronized (monitor) {
                    lastWrittenEventUtcTimestampForSource.putAll(newTimestamps);
                    addToWrittenInSecond.forEach(
                            (k, count) -> {
                                writtenInSecond.merge(k, count, (old, ignored) -> old + count);

                                fillSensors();
                            }
                    );
                }
            }
        };
    }

    private void fillSensors() {
        for (Map.Entry<String, LocalDateTime> entry : lastWrittenEventUtcTimestampForSource.entrySet()) {
            String source = entry.getKey();
            GaugeInt64 delaySensor = SOLOMON_REGISTRY.gaugeInt64(
                    "ytwriter_delay_seconds",
                    Labels.of("source", source)
            );
            delaySensor.set(Duration.between(entry.getValue(), LocalDateTime.now(ZoneId.of("UTC"))).getSeconds());
        }

        for (Map.Entry<Pair<String, Long>, Long> entry : writtenInSecond.entrySet()) {
            String source = entry.getKey().getKey();
            GaugeInt64 writtenRowsSensor = SOLOMON_REGISTRY.gaugeInt64(
                    "ytwriter_written_rows",
                    Labels.of("source", source)
            );
            writtenRowsSensor.set(entry.getValue());
        }
    }
}
