package ru.yandex.direct.binlogbroker.replicatetoyt.configuration;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.binlogbroker.logbroker_utils.models.BinlogEventWithOffset;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.utils.Interrupts;

/**
 * Обработчик событий, который выкидывает события с неинтересными
 * {@link ru.yandex.direct.binlog.model.BinlogEvent#getSource() BinlogEvent.getSource()}
 * и передаёт оставшиеся события другому обработчику.
 */
@ParametersAreNonnullByDefault
public class FilteringEventConsumer implements Interrupts.InterruptibleConsumer<List<BinlogEventWithOffset>> {
    private final Interrupts.InterruptibleConsumer<List<BinlogEventWithOffset>> forward;
    private final ImmutableSet<String> sourceNames;

    public FilteringEventConsumer(Interrupts.InterruptibleConsumer<List<BinlogEventWithOffset>> forward,
                                  ImmutableSet<SourceType> sources) {
        Preconditions.checkArgument(!sources.isEmpty(), "Source set is empty");
        this.forward = forward;
        this.sourceNames = sources.stream()
                .map(SourceType::getSourceName)
                .collect(ImmutableSet.toImmutableSet());
    }

    @Override
    public void accept(List<BinlogEventWithOffset> binlogEventWithOffsets) throws InterruptedException {
        List<BinlogEventWithOffset> filtered = new ArrayList<>();
        for (BinlogEventWithOffset event : binlogEventWithOffsets) {
            if (sourceNames.contains(event.getEvent().getSource())) {
                filtered.add(event);
            }
        }
        // DIRECT-82629
        // Если наткнуться на очень длинный пласт игнорируемых сообщений, то стейт будет долго не записываться
        // Нужно вынести запись стейта из EventConsumer в класс, который на самом деле контролирует поток событий и
        // вызывает EventConsumer - сейчас это BinlogLogbrokerReader.
        if (!filtered.isEmpty()) {
            forward.accept(filtered);
        }
    }
}
