package ru.yandex.direct.binlogbroker.replicatetoyt.configuration;

import java.time.Duration;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Lazy;

import ru.yandex.direct.binlogbroker.logbroker_utils.configuration.LogbrokerCommonConfiguration;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.BinlogEventWithOffset;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceType;
import ru.yandex.direct.binlogbroker.logbroker_utils.models.SourceTypeHelper;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.BinlogLogbrokerReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerBatchReader;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.LogbrokerReaderInitException;
import ru.yandex.direct.binlogbroker.logbroker_utils.reader.RetryingLogbrokerBatchReader;
import ru.yandex.direct.binlogbroker.replicatetoyt.RetryingYtReplicator;
import ru.yandex.direct.binlogbroker.replicatetoyt.StateManager;
import ru.yandex.direct.binlogbroker.replicatetoyt.YtAttrStateManager;
import ru.yandex.direct.binlogbroker.replicatetoyt.YtReplicator;
import ru.yandex.direct.binlogbroker.replicatetoyt.YtReplicatorImpl;
import ru.yandex.direct.binlogbroker.replicatetoyt.components.EventConsumer;
import ru.yandex.direct.binlogbroker.replicatetoyt.components.YtWriterMetricProvider;
import ru.yandex.direct.binlogbroker.replicatetoyt.jcommander.LogbrokerParams;
import ru.yandex.direct.binlogbroker.replicatetoyt.jcommander.YtParams;
import ru.yandex.direct.binlogbroker.replicatetoyt.jcommander.YtWriterParams;
import ru.yandex.direct.binlogbroker.replicatetoyt.models.LogbrokerConfig;
import ru.yandex.direct.common.configuration.MetricsConfiguration;
import ru.yandex.direct.utils.Interrupts;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.kikimr.persqueue.LogbrokerClientFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.consumer.SyncConsumer;
import ru.yandex.kikimr.persqueue.consumer.sync.SyncConsumerConfig;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;
import ru.yandex.yt.ytclient.proxy.YtClient;

@ComponentScan(
        basePackages = "ru.yandex.direct.binlogbroker.replicatetoyt.components",
        excludeFilters = @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION))
@Configuration
@Import({
        LogbrokerCommonConfiguration.class,
        MetricsConfiguration.class,
        YtReplicatorConfiguration.class,
        YtWriterJcommanderConfiguration.class,
})
@Lazy
@ParametersAreNonnullByDefault
@SuppressWarnings({
        "squid:S00107",  // too many arguments
})
public class YtWriterConfiguration {
    @Bean
    public YtCluster ytCluster(YtParams ytParams) {
        return ytParams.ytCluster;
    }

    @Bean
    public Set<SourceType> sources(LogbrokerParams logbrokerParams, SourceTypeHelper sourceTypeHelper) {
        return sourceTypeHelper.getSources(logbrokerParams.sources);
    }

    @Bean
    public YtReplicator ytReplicator(YtClient ytClient, Yt yt, YtParams ytParams, YtWriterParams ytWriterParams) {
        return new RetryingYtReplicator(new YtReplicatorImpl(
                ytClient, yt, ytParams.ytPath, ytParams.tmpYtPath, ytWriterParams.skipErroneousEvents));
    }

    @Bean
    public SyncConsumerConfig syncConsumerConfig(LogbrokerConfig logbrokerConfig,
                                                 Supplier<Credentials> credentialsProvider) {
        return SyncConsumerConfig
                .builder(Collections.singletonList(logbrokerConfig.getTopic()), logbrokerConfig.getClientId())
                .setReadDataTimeout((int) logbrokerConfig.getTimeout().toMillis(), TimeUnit.MILLISECONDS)
                .setInitTimeout((int) logbrokerConfig.getInitTimeout().toMillis(), TimeUnit.MILLISECONDS)
                .configureSession(configure -> configure.setGroups(logbrokerConfig.getGroups()))
                .setCredentialsProvider(credentialsProvider)
                .build();
    }

    @Bean
    LogbrokerConfig logbrokerConfig(
            @Value("${binlogbroker.logbroker.host}") String logbrokerHost,
            @Value("${binlogbroker.logbroker.topic}") String logbrokerTopic,
            @Value("${binlogbroker.logbroker.init_timeout}") int logbrokerInitTimeout,
            @Value("${binlogbroker.logbroker.retries}") int retries,
            ImmutableSet<SourceType> sources,
            LogbrokerParams logbrokerParams) {
        List<Integer> logbrokerGroups = sources.stream()
                .map(SourceType::getLogbrokerGroupId)
                .collect(Collectors.toList());
        return new LogbrokerConfig.Builder()
                .setClientId(logbrokerParams.clientId)
                .setGroups(logbrokerGroups)
                .setHost(logbrokerHost)
                .setTopic(logbrokerTopic)

                // Логброкер присылает маленькие пачки событий, поэтому они буферизуются.
                // Большой тайм-аут вредит буферизации, частый поллинг предпочтительнее.
                .setTimeout(Duration.ofMillis(100))

                .setInitTimeout(Duration.ofSeconds(logbrokerInitTimeout))
                .setRetries(retries)
                .build();
    }

    @Bean
    public LogbrokerBatchReader<BinlogEventWithOffset> binlogEventReader(LogbrokerConfig logbrokerConfig,
                                                                         SyncConsumerConfig syncConsumerConfig,
                                                                         LogbrokerParams logbrokerParams) {
        ProxyBalancer proxyBalancer = new ProxyBalancer(logbrokerConfig.getHost());
        LogbrokerClientFactory clientFactory = new LogbrokerClientFactory(proxyBalancer);
        Supplier<SyncConsumer> syncConsumerSupplier = () -> {
            try {
                final SyncConsumer syncConsumer =
                        clientFactory.syncConsumer(syncConsumerConfig, logbrokerConfig.getInitTimeout().toMillis(),
                                TimeUnit.MILLISECONDS);
                syncConsumer.init();
                return syncConsumer;
            } catch (InterruptedException | TimeoutException ex) {
                throw new LogbrokerReaderInitException(ex);
            }
        };
        return new RetryingLogbrokerBatchReader<>(
                () -> new BinlogLogbrokerReader.Builder()
                        .withLogbrokerConsumerSupplier(syncConsumerSupplier)
                        .withLogbrokerNoCommit(logbrokerParams.logbrokerNoCommit)
                        .build(),
                logbrokerConfig.getRetries());
    }

    @Bean
    public StateManager stateManager(Yt yt, YtParams ytParams) {
        return new YtAttrStateManager(yt, ytParams.ytPath);
    }

    @Bean
    public Interrupts.InterruptibleConsumer<List<BinlogEventWithOffset>> eventConsumer(
            YtReplicator ytReplicator,
            ImmutableSet<SourceType> sources,
            StateManager stateManager,
            YtWriterMetricProvider ytWriterMetricProvider
    ) {
        Interrupts.InterruptibleConsumer<List<BinlogEventWithOffset>> result
                = new EventConsumer(ytReplicator, stateManager);
        if (!sources.isEmpty()) {
            result = new FilteringEventConsumer(result, sources);
        }
        result = ytWriterMetricProvider.wrap(result);
        return result;
    }
}
