package ru.yandex.direct.campkindsimporter;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.time.LocalDate;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

import javax.annotation.Generated;
import javax.lang.model.element.Modifier;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.module.SimpleDeserializers;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.type.MapLikeType;
import com.fasterxml.jackson.databind.type.TypeFactory;
import com.fasterxml.jackson.datatype.guava.GuavaModule;
import com.google.common.collect.ImmutableListMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import com.google.common.collect.ImmutableSet;
import com.squareup.javapoet.AnnotationSpec;
import com.squareup.javapoet.ClassName;
import com.squareup.javapoet.CodeBlock;
import com.squareup.javapoet.FieldSpec;
import com.squareup.javapoet.JavaFile;
import com.squareup.javapoet.MethodSpec;
import com.squareup.javapoet.ParameterizedTypeName;
import com.squareup.javapoet.TypeName;
import com.squareup.javapoet.TypeSpec;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.direct.utils.IgnoreCaseEnumDeserializer;

public class CampKindsImporter<T extends Enum> {
    private final URL remoteJson;
    private final File sourcesRootDestDir;
    private final Class<T> valuesEnum;
    private final Package generatedClassPackage;
    private final String generatedClassName;

    CampKindsImporter(URL remoteJson, File sourcesRootDestDir, Package generatedClassPackage, String generatedClassName,
                      Class<T> valuesEnum) {
        this.remoteJson = remoteJson;
        this.sourcesRootDestDir = sourcesRootDestDir;
        this.valuesEnum = valuesEnum;
        this.generatedClassName = generatedClassName;
        this.generatedClassPackage = generatedClassPackage;
    }

    private static CodeBlock headerJavaDoc() {
        String headerJavaDoc = "This file was generated by $L tool.\n"
                + "Please do not edit this file, edit {@code $L} instead.\n"
                + "<p>\n"
                + "Descriptions are available in perl source\n"
                + "\n"
                + "@see $L\n"
                + "@see <a href=\"/direct/perl/protected/Campaign/Types.pm\">Campaign::Types</a>\n"
                + "@see <a href=\"/direct/perl/protected/Intapi/CampaignTypesList.pm\">Intapi::CampaignTypesList</a>\n";
        Class<?> cls = CampKindsImporter.class;
        return CodeBlock.of(headerJavaDoc, cls.getSimpleName(), cls.getCanonicalName(), cls.getPackage().getName());
    }

    void doImport() throws IOException {
        AnnotationSpec generatedAnnotation = AnnotationSpec.builder(Generated.class)
                .addMember("value", "$S", CampKindsImporter.class.getCanonicalName())
                .addMember("date", "$S", LocalDate.now().toString())
                .build();

        TypeSpec.Builder classBuilder = TypeSpec.classBuilder(generatedClassName)
                .addJavadoc(headerJavaDoc())
                .addAnnotation(generatedAnnotation)
                .addModifiers(Modifier.PUBLIC, Modifier.FINAL)
                .addMethod(MethodSpec.constructorBuilder().addModifiers(Modifier.PRIVATE).build());

        readMappings(remoteJson).asMap().entrySet().stream()
                .sorted(ImmutableMap.Entry.comparingByKey())
                .map(this::convertKindMappingToFieldSpec)
                .forEach(classBuilder::addField);

        JavaFile javaFile = JavaFile
                .builder(generatedClassPackage.getName(), classBuilder.build())
                .indent("    ")
                .build();

        javaFile.writeTo(sourcesRootDestDir);
    }

    private FieldSpec convertKindMappingToFieldSpec(Map.Entry<String, Collection<T>> entry) {
        TypeName fieldType = ParameterizedTypeName.get(ClassName.get(Set.class), TypeName.get(valuesEnum));

        Object[] campTypesLiteralNames = entry.getValue().stream().map(CampKindsImporter::enumLiteral).toArray();

        return FieldSpec.builder(fieldType, entry.getKey().toUpperCase())
                .addModifiers(Modifier.PUBLIC, Modifier.STATIC, Modifier.FINAL)
                .initializer(CodeBlock.builder()
                        .add("$T.of(", ImmutableSet.class)
                        .add(StringUtils.repeat("$L", ", ", campTypesLiteralNames.length), campTypesLiteralNames)
                        .add(")").build())
                .build();
    }

    private static <E extends Enum<?>> String enumLiteral(E elem) {
        return String.format("%s.%s", elem.getDeclaringClass().getSimpleName(), elem.name());
    }

    private ImmutableMultimap<String, T> readMappings(URL json) throws IOException {
        MapLikeType ref = TypeFactory.defaultInstance()
                .constructMapLikeType(ImmutableListMultimap.class, String.class, valuesEnum);

        SimpleModule enumModule = new SimpleModule();
        enumModule.setDeserializers(new SimpleDeserializers(
                Collections.singletonMap(valuesEnum, new IgnoreCaseEnumDeserializer<>(valuesEnum))));

        ObjectMapper mapper = new ObjectMapper();
        mapper.registerModule(enumModule);
        mapper.registerModule(new GuavaModule());

        JsonNode campTypesNode = mapper.readTree(json).get("camp_types");
        return mapper.convertValue(campTypesNode, ref);
    }
}


