package ru.yandex.direct.chassis.app

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Value
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.util.ydb.CreateYdbTablesJob
import ru.yandex.direct.common.jetty.JettyLauncherStartListener
import ru.yandex.direct.hourglass.client.HourglassClient
import ru.yandex.direct.hourglass.storage.Job
import ru.yandex.direct.scheduler.hourglass.implementations.HourglassScheduler
import java.time.Instant
import java.time.OffsetDateTime

@Component
@Lazy(false)
open class ChassisJobLauncher(
    @Value("\${chassis.job_launcher.enabled:false}") private val enabled: Boolean,
    private val scheduler: HourglassScheduler,
    private val client: HourglassClient,
) : JettyLauncherStartListener {
    override fun onStart() {
        if (enabled) {
            logger.info("launch jobs")
            logger.info("save hourglass schedule")
            scheduler.saveSchedule()
            logger.info("start hourglass scheduler")
            scheduler.start()
            bootstrapTasks()
        } else {
            logger.info("job launcher disabled")
        }
    }

    private fun bootstrapTasks() {
        logger.info("bootstrap some tasks")
        client.tasks
            .filter { it.taskId().name().equals(CreateYdbTablesJob::class.java.canonicalName) }
            .forEach { bootstrapTask(it) }
    }

    private fun bootstrapTask(job: Job) {
        logger.info("schedule next run at now for ${job.taskId().name()}")
        client.changeNextRunTime(job, Instant.now(), OffsetDateTime.now().offset)
    }

    private val logger = LoggerFactory.getLogger(ChassisJobLauncher::class.java)
}
