package ru.yandex.direct.chassis.configuration

import com.yandex.ydb.core.auth.AuthProvider
import com.yandex.ydb.core.auth.TokenAuthProvider
import org.asynchttpclient.AsyncHttpClient
import org.springframework.beans.factory.annotation.Value
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.ComponentScan
import org.springframework.context.annotation.Configuration
import org.springframework.context.annotation.FilterType
import org.springframework.context.annotation.Import
import org.springframework.context.annotation.Lazy
import org.springframework.scheduling.TaskScheduler
import org.springframework.scheduling.concurrent.ThreadPoolTaskScheduler
import org.springframework.web.client.RestTemplate
import org.tmatesoft.svn.core.internal.wc.DefaultSVNOptions
import org.tmatesoft.svn.core.wc.SVNClientManager
import org.tmatesoft.svn.core.wc.SVNWCUtil
import ru.yandex.direct.asynchttp.FetcherSettings
import ru.yandex.direct.asynchttp.ParallelFetcherFactory
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.util.YpClient
import ru.yandex.direct.chassis.util.jenkins.JenkinsClient
import ru.yandex.direct.chassis.util.staff.StaffTelegramAccountsProvider
import ru.yandex.direct.chassis.util.svn.SshAgentAuthenticationProvider
import ru.yandex.direct.chassis.util.svn.SvnLogSlf4jAdapter
import ru.yandex.direct.chassis.util.ydb.YdbClient
import ru.yandex.direct.common.configuration.CommonConfiguration
import ru.yandex.direct.config.DirectConfig
import ru.yandex.direct.juggler.AsyncHttpJugglerClient
import ru.yandex.direct.juggler.JugglerClient
import ru.yandex.direct.liveresource.LiveResourceWatcherFactory
import ru.yandex.direct.liveresource.provider.LiveResourceFactoryBean
import ru.yandex.direct.staff.client.StaffClient
import ru.yandex.direct.staff.client.model.StaffConfiguration
import ru.yandex.direct.telegram.client.TelegramClient
import ru.yandex.direct.tvm.TvmIntegration
import ru.yandex.direct.tvm.TvmIntegrationImpl
import ru.yandex.direct.tvm.TvmIntegrationStub
import ru.yandex.direct.utils.io.FileUtils
import ru.yandex.direct.ydb.tvm.YdbTvmAuthProvider
import ru.yandex.direct.ytwrapper.client.YtClusterConfigProvider
import ru.yandex.direct.ytwrapper.client.YtClusterTypesafeConfigProvider
import ru.yandex.direct.ytwrapper.client.YtProvider
import ru.yandex.direct.ytwrapper.dynamic.YtDynamicConfig
import ru.yandex.direct.ytwrapper.dynamic.YtDynamicTypesafeConfig
import ru.yandex.yp.YpInstance
import ru.yandex.yp.YpRawClient
import ru.yandex.yp.YpRawClientBuilder
import ru.yandex.yql.YqlDataSource
import ru.yandex.yql.settings.YqlProperties

@Configuration
@ComponentScan(
    basePackages = ["ru.yandex.direct.chassis"],
    excludeFilters = [ComponentScan.Filter(
        value = [Configuration::class],
        type = FilterType.ANNOTATION
    )]
)
@Import(
    CommonConfiguration::class,
    StartrekConfiguration::class,
    AutotestsConfiguration::class,
    ChassisHourglassConfiguration::class,
)
// ide не осознает kotlin-allopen как часть нашей сборки
@Suppress("ImplicitSubclassInspection")
open class AppConfiguration {

    @Bean
    fun svnClientManager(
        @Value("\${chassis.svn.user}") user: String,
        @Value("\${chassis.svn.key_file}") keyFile: String,
    ): SVNClientManager {
        val authManager = SVNWCUtil.createDefaultAuthenticationManager(
            SVNWCUtil.getDefaultConfigurationDirectory(),
            user,
            null as CharArray?,
            FileUtils.expandHome(keyFile).toFile(),
            null,
            false
        )
        authManager.setAuthenticationProvider(SshAgentAuthenticationProvider(user))

        return SVNClientManager.newInstance(
            DefaultSVNOptions(),
            authManager
        ).also {
            it.debugLog = SvnLogSlf4jAdapter()
        }
    }

    @Bean
    fun ypClientCrossDC(
        @Value("\${yp.chassis.token}") tokenResource: String,
        liveResourceFactoryBean: LiveResourceFactoryBean,
    ): YpRawClient {
        val token = liveResourceFactoryBean[tokenResource].content!!.replace("\n", "")
        return YpRawClientBuilder(YpInstance.CROSS_DC) { token }.build()
    }

    @Bean
    fun directAppsConfProvider(
        @Value("\${chassis.direct_apps_conf}") directAppsResource: String,
        liveResourceFactoryBean: LiveResourceFactoryBean,
        liveResourceWatcherFactory: LiveResourceWatcherFactory,
    ): DirectAppsConfProvider {
        val liveResource = liveResourceFactoryBean.get(directAppsResource)
        val provider = DirectAppsConfProvider(liveResource.content)

        val watcher = liveResourceWatcherFactory.createWatcher(liveResource)
        watcher.addListener(provider)

        return provider
    }

    @Bean
    fun restTemplate(): RestTemplate {
        return RestTemplate()
    }

    @Bean
    fun taskScheduler(): TaskScheduler {
        return ThreadPoolTaskScheduler()
    }

    @Bean
    fun tvmIntegration(directConfig: DirectConfig, taskScheduler: TaskScheduler): TvmIntegration {
        val enabled: Boolean = directConfig.findBoolean("tvm.enabled").orElse(false)
        return if (enabled) {
            TvmIntegrationImpl.create(directConfig, taskScheduler)
        } else {
            TvmIntegrationStub()
        }
    }

    @Bean
    fun jenkinsClient(
        asyncHttpClient: AsyncHttpClient,
        liveResourceFactoryBean: LiveResourceFactoryBean,
        @Value("\${jenkins.user}") jenkinsUser: String,
        @Value("\${jenkins.password}") jenkinsPassword: String,
    ): JenkinsClient {
        val passwordLiveResource = liveResourceFactoryBean.get(jenkinsPassword)
        return JenkinsClient(asyncHttpClient, jenkinsUser, passwordLiveResource)
    }

    @Bean
    fun telegramClient(
        asyncHttpClient: AsyncHttpClient,
        liveResourceFactoryBean: LiveResourceFactoryBean,
        @Value("\${telegram.token}") telegramToken: String,
    ): TelegramClient {
        val telegramTokenResource = liveResourceFactoryBean.get(telegramToken)
        return TelegramClient(asyncHttpClient, telegramTokenResource)
    }

    @Bean
    fun staffClient(
        asyncHttpClient: AsyncHttpClient,
        tvmIntegration: TvmIntegration,
        liveResourceFactoryBean: LiveResourceFactoryBean,
        @Value("\${staff.staff_api_url}") staffApiUrl: String,
        @Value("\${staff.gaps_api_url}") gapsApiUrl: String,
        @Value("\${staff.token}") tokenResource: String,
    ): StaffClient {
        val token = liveResourceFactoryBean[tokenResource].content.trim()
        val staffConfiguration = StaffConfiguration(staffApiUrl, gapsApiUrl, token)
        val parallelFetcherFactory = ParallelFetcherFactory(asyncHttpClient, FetcherSettings())
        return StaffClient(staffConfiguration, parallelFetcherFactory, tvmIntegration, true)
    }

    @Bean
    fun staffTelegramAccountsProvider(
        staffClient: StaffClient,
        taskScheduler: TaskScheduler,
    ) = StaffTelegramAccountsProvider(staffClient, taskScheduler)
        .also { it.start() }

    @Bean
    open fun createJugglerClient(
        asyncHttpClient: AsyncHttpClient,
        directConfig: DirectConfig,
    ): JugglerClient {
        val config = directConfig.getBranch("juggler")
        val client = AsyncHttpJugglerClient(config.getString("eventsGateway"), true, asyncHttpClient)
        client.setApiUrl(config.getString("api"))
        return client
    }

    @Bean
    open fun ydbAuthProvider(directConfig: DirectConfig, tvmIntegration: TvmIntegration): AuthProvider {
        val tokenPath = directConfig
            .getBranch("maintenance-helpers")
            .getBranch("ydb").findString("token_file")
        return if (tokenPath.isPresent) {
            val path = FileUtils.expandHome(tokenPath.get())
            val token = FileUtils.slurp(path).trim { it <= ' ' }
            TokenAuthProvider(token)
        } else {
            YdbTvmAuthProvider(tvmIntegration)
        }
    }

    @Bean
    open fun ydbClient(directConfig: DirectConfig, authProvider: AuthProvider): YdbClient {
        val ydbConfig = directConfig
            .getBranch("maintenance-helpers")
            .getBranch("ydb");
        val endpoint = ydbConfig.getString("endpoint")
        val database = ydbConfig.getString("database")
        return YdbClient(endpoint, database, authProvider)
    }

    @Bean
    fun yqlDataSource(
        @Value("\${maintenance-helpers.yql_token_file}") tokenResource: String,
        liveResourceFactoryBean: LiveResourceFactoryBean
    ): YqlDataSource {
        val token = liveResourceFactoryBean.get(tokenResource).content.trim()
        val props = YqlProperties().also {
            it.syntaxVersion = 1
            it.password = token
        }
        return YqlDataSource("jdbc:yql://yql.yandex.net:443", props)
    }

    // клиент из-под robot-direct-sustain, без (?) пишущих прав
    @Bean
    fun maintenanceHelpersYpClient(
        @Value("\${maintenance-helpers.yp_token_file}") tokenResource: String,
        liveResourceFactoryBean: LiveResourceFactoryBean,
    ): YpClient {
        val token = liveResourceFactoryBean[tokenResource].content.trim()
        return YpClient(token)
    }

    @Bean
    @Lazy
    open fun ytProvider(
        directConfig: DirectConfig, dynConfig: YtDynamicConfig
    ): YtProvider {
        val ytConfig = directConfig.getBranch("yt")
        val ytClusterConfigProvider: YtClusterConfigProvider = YtClusterTypesafeConfigProvider(ytConfig.config)
        return YtProvider(ytClusterConfigProvider, dynConfig, null)
    }

    @Bean
    open fun ytDynamicConfig(directConfig: DirectConfig): YtDynamicConfig {
        val ytConfig = directConfig.getBranch("yt")
        return YtDynamicTypesafeConfig(ytConfig.config)
    }
}
