package ru.yandex.direct.chassis.configuration

import org.springframework.beans.factory.annotation.Value
import org.springframework.context.ApplicationContext
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import org.springframework.context.annotation.Import
import ru.yandex.direct.chassis.util.Utils.logger
import ru.yandex.direct.configuration.DirectHourglassConfiguration
import ru.yandex.direct.configuration.DirectHourglassConfiguration.VERSION_BEAN_NAME
import ru.yandex.direct.configuration.HourglassYdbStorageConfiguration
import ru.yandex.direct.hourglass.SchedulerService
import ru.yandex.direct.hourglass.client.HourglassClient
import ru.yandex.direct.hourglass.client.HourglassClientImpl
import ru.yandex.direct.hourglass.implementations.ThreadsHierarchy
import ru.yandex.direct.hourglass.storage.Storage
import ru.yandex.direct.hourglass.updateschedule.MainScheduleVersionExtractor
import ru.yandex.direct.hourglass.updateschedule.ScheduleUpdateService
import ru.yandex.direct.hourglass.updateschedule.SchedulerInstancesRepository
import ru.yandex.direct.hourglass.ydb.schedulerinstances.YdbScheduleInstancesRepository
import ru.yandex.direct.hourglass.ydb.storage.YdbStorageImpl
import ru.yandex.direct.juggler.JugglerAsyncSender
import ru.yandex.direct.juggler.JugglerClient
import ru.yandex.direct.juggler.JugglerSender
import ru.yandex.direct.scheduler.JobInterceptorsList
import ru.yandex.direct.scheduler.hourglass.ScheduleRecordListProvider
import ru.yandex.direct.scheduler.hourglass.TaskFactory
import ru.yandex.direct.scheduler.hourglass.implementations.HourglassScheduler
import ru.yandex.direct.scheduler.hourglass.implementations.TaskParameterizer

@Import(
    DirectHourglassConfiguration::class,
    HourglassYdbStorageConfiguration::class,
)
@Configuration
open class ChassisHourglassConfiguration {

    @Bean(DirectHourglassConfiguration.STORAGE_BEAN_NAME)
    open fun ydbStorage(storage: YdbStorageImpl): Storage {
        return storage
    }

    @Bean(DirectHourglassConfiguration.INSTANCES_REPOSITORY_BEAN_NAME)
    open fun schedulerInstancesRepository(repository: YdbScheduleInstancesRepository): SchedulerInstancesRepository {
        return repository
    }

    @Bean(VERSION_BEAN_NAME)
    open fun version(mainScheduleVersionExtractor: MainScheduleVersionExtractor): String {
        // с переездом в Deploy (где в контейнере живет только одна версия приложения) стало неактуально
        return mainScheduleVersionExtractor.version
    }

    @Bean
    open fun taskParameterizer(
        context: ApplicationContext,
        @Value("\${db_shards}") numOfPpcShards: Int,
    ): TaskParameterizer {
        val shards = IntRange(1, numOfPpcShards).toList()
        return TaskParameterizer(context, shards)
    }

    @Bean
    open fun jugglerSender(jugglerClient: JugglerClient): JugglerSender {
        return JugglerAsyncSender(jugglerClient)
    }

    @Bean
    open fun jobInterceptorsList(
        tracing: TracingInterceptor,
        logging: LoggingInterceptor,
        juggler: JugglerInterceptor,
    ): JobInterceptorsList {
        return JobInterceptorsList(
            logging,
            tracing,
            juggler,
        )
    }

    @Bean
    open fun hourglassScheduler(
        scheduleRecordListProvider: ScheduleRecordListProvider,
        taskFactory: TaskFactory,
        schedulerService: SchedulerService,
        threadsHierarchy: ThreadsHierarchy,
        scheduleUpdateService: ScheduleUpdateService
    ): HourglassScheduler {
        val hourglassScheduler = HourglassScheduler(
            scheduleRecordListProvider, taskFactory, schedulerService,
            scheduleUpdateService
        )

        threadsHierarchy.addUncaughtExceptionHandler { t: Thread, e: Throwable ->
            logger.error("uncaught exception in thread ${t.name}, stop scheduler", e)
            hourglassScheduler.stop()
        }
        return hourglassScheduler
    }

    @Bean
    open fun ydbHourglassClient(
        storage: YdbStorageImpl,
        schedulerInstancesRepository: YdbScheduleInstancesRepository
    ): HourglassClient {
        return HourglassClientImpl(storage, schedulerInstancesRepository)
    }
}
