package ru.yandex.direct.chassis.configuration

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.app.ChassisApp
import ru.yandex.direct.juggler.JugglerEvent
import ru.yandex.direct.juggler.JugglerSender
import ru.yandex.direct.scheduler.JobInterceptor
import ru.yandex.direct.scheduler.support.BaseDirectJob
import ru.yandex.direct.scheduler.support.WrappedJob
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.tracing.TraceHelper
import ru.yandex.direct.tracing.formatTags
import java.time.Duration
import kotlin.reflect.KClass

@Component
class TracingInterceptor(
    private val traceHelper: TraceHelper,
) : JobInterceptor {

    override fun wrap(job: WrappedJob, originalJob: BaseDirectJob): WrappedJob {
        val method = formatMethodName(originalJob::class)
        val tags = formatTags(originalJob)
        return WrappedJob { traceHelper.guard(method, tags).use { job.run() } }
    }
}

private fun <T : BaseDirectJob> formatMethodName(clazz: KClass<T>): String {
    return clazz.simpleName!!
}

@Component
class LoggingInterceptor : JobInterceptor {
    private val logger = LoggerFactory.getLogger(LoggingInterceptor::class.java)

    override fun wrap(code: WrappedJob, originalJob: BaseDirectJob): WrappedJob {
        return WrappedJob {
            val tags = Trace.current().tags
            val msgSuffix = if (tags.isNullOrEmpty()) "" else " $tags"
            logger.info("START$msgSuffix")

            var jobSuccess = false
            try {
                code.run()
                jobSuccess = true
            } catch (e: Exception) {
                logger.error("Task ${originalJob.javaClass.simpleName} $msgSuffix threw Exception", e)
                throw e
            } catch (e: Throwable) {
                logger.error("Task ${originalJob.javaClass.simpleName} $msgSuffix threw Throwable", e)
                throw e
            } finally {
                logger.info("${if (jobSuccess) "FINISH" else "FAILED"} $msgSuffix")
            }
        }
    }
}

@Component
class JugglerInterceptor(
    private val jugglerSender: JugglerSender
) : JobInterceptor {
    private val eventsTimeout = Duration.ofMinutes(5)

    override fun wrap(code: WrappedJob, originalJob: BaseDirectJob): WrappedJob {
        // пока не поддерживаем шардированные/параметризованные джобы
        // когда понадобится — нужно будет пошарить код из jobsAppJugglerChecksProvider
        val taskName = formatMethodName(originalJob::class)
        return WrappedJob {
            code.run()
            val event = JugglerEvent(
                "${ChassisApp.SERVICE}.$taskName",
                originalJob.jugglerStatus,
                originalJob.jugglerDescription,
            )
            jugglerSender.sendEvent(event, eventsTimeout)
        }
    }
}
