package ru.yandex.direct.chassis.entity.aquatests

import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.aqua.AquaClient
import ru.yandex.direct.chassis.util.aqua.GetLaunchesResponse
import ru.yandex.direct.chassis.util.aqua.Launch
import ru.yandex.direct.chassis.util.aqua.LaunchStatus
import ru.yandex.direct.chassis.util.nonCf
import ru.yandex.direct.chassis.util.versionica.PropertyLogEntry
import ru.yandex.direct.chassis.util.versionica.VersionicaClient
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.utils.DateTimeUtils
import ru.yandex.startrek.client.model.Comment
import ru.yandex.startrek.client.model.Issue
import java.time.Instant
import java.time.LocalDateTime
import java.time.ZoneId
import java.time.format.DateTimeFormatter

/**
 * Джоба по добавлению @see
 * <a href="https://aqua.yandex-team.ru/#/launches?skip=0&limit=20&packId=53427b42e4b08984236588ab">
 *     результатов тестов из AQUA</a> в релизный тикет.
 * Подцепляет два результата выполнения тестов, прошедших перед началом деплоя версии из тикета,
 * и два прошедших после окончания деплоя версии тикета.
 */
@Hourglass(periodInSeconds = 3 * 60, needSchedule = ProductionOnly::class)
class AquaTestsAutocommentReleasesJob(
    private val startrek: StartrekHelper,
    private val aquaClient: AquaClient,
    private val versionicaClient: VersionicaClient,
) : DirectJob() {

    companion object {
        private val logger = LoggerFactory.getLogger(AquaTestsAutocommentReleasesJob::class.java)
        private const val TAKE_LAUNCHES_IN_PACK_BY_QUERY_LIMIT = 100
        private val VERSION_REGEX = """(\d+(\.\d+)+-\d+)""".toRegex()
        private const val AQUA_TESTS_COMMENT_PREFIX = "Светофор для версии "
        private const val MAX_SUITES_COUNT = 2
        private const val AQUA_TEST_PACK_ID = "53427b42e4b08984236588ab"
        private const val BEGIN_DEPLOY_LAG_IN_SECONDS = 300
        private const val MIN_DEPLOYS_COUNT_PER_VERSION = 2
        private const val AFTER_DEPLOY_TABLE_HEADER = "После выкладки на ТС:"
        private const val BEFORE_DEPLOY_TABLE_HEADER = "До выкладки на ТС:"
        private const val AQUA_ALL_TESTS_LINK =
            "https://aqua.yandex-team.ru/#/launches?skip=0&limit=20&packId=$AQUA_TEST_PACK_ID"
        private const val AQUA_ALL_TESTS_LINK_TEXT = "все запуски светофора на ТС"
        private const val AQUA_TESTS_NOT_FOUND_TEXT = "Пока никаких тестов не обнаружено"
        private const val ZERO_TESTS_TEXT = "!!(сер)0!!"
        private val DATE_TIME_FORMATTER = DateTimeFormatter.ofPattern("dd.MM.yyyy HH:mm:ss")
    }

    override fun execute() {
        // Тикет для дебага:
        // val releases = startrek.session.issues().find("key: DIRECT-155888").toList().nonCf()

        // Получаем все незакрытые тикеты из очереди DIRECT с компонентом Releases: Direct
        logger.info("Trying to receive not closed tickets from DIRECT queue with 'release' Type " +
            "and 'Releases: Direct' component")
        val releases = startrek.session.issues().find(
            """
            | Queue: DIRECT
            | Type: release
            | Components: "Releases: Direct"
            | Status: !Closed
            | "Sort by": key desc""".trimMargin()
        ).toList().nonCf()

        var requireUpdateComments = 0
        var newComments = 0
        var completedComments = 0
        val affectedReleasesList = mutableListOf<AquaTestIssueInfo>()
        logger.info("Trying to find comments for release versions")
        for (release: Issue in releases) {
            var commentForReleaseVersionIsCompleted = false
            logger.info("Processing ticket ${release.key}")
            val releaseVersion: String? = extractVersion(release.summary)
            if (releaseVersion == null) {
                logger.error(
                    "Unable to found version in release ticket ${release.key}, " +
                        "summary: ${release.summary}, skipping ticket"
                )
                continue
            }
            val commentsByVersionMap = mutableMapOf<String, AquaTestCommentInfo>()
            for (comment: Comment in release.comments) {
                val version: String = checkIsTrafficLightCommentAndGetVersion(comment) ?: continue
                val parsedCommentData: AquaTestCommentData = parseCommentData(comment.text.get())
                if (isCommentDataComplete(parsedCommentData)) {
                    if (version == releaseVersion) {
                        // Если комментарий для версии из заголовка тикета завершен, то ставим флаг, чтобы не
                        // добавлять его в карту требующих изменений комментариев после цикла по комментариям.
                        // Флаг нужен для потому что в карте комментариев этой версии не будет, и простой проверки
                        // на наличие версии в карте будет недостаточно.
                        commentForReleaseVersionIsCompleted = true
                    }
                    logger.info(
                        "Found completed comment with version $version in ticket ${release.key}" +
                            ", skipping"
                    )
                    completedComments++
                } else {
                    logger.info(
                        "Found uncompleted comment with version $version in ticket ${release.key}" +
                            ", adding to the list for changes"
                    )
                    commentsByVersionMap[version] = AquaTestCommentInfo(version, comment, parsedCommentData)
                    requireUpdateComments++
                }
            }

            if (!commentForReleaseVersionIsCompleted && releaseVersion !in commentsByVersionMap) {
                // Если у тикета есть версия в заголовке, а комментария для нее еще нет,
                // добавим для этой версии в карту комментариев новый пустой комментарий
                logger.info("Found new version $releaseVersion in release ticket ${release.key} summary")
                commentsByVersionMap[releaseVersion] = AquaTestCommentInfo(releaseVersion)
                newComments++
            }

            if (commentsByVersionMap.isNotEmpty()) {
                // Если есть что обновлять в тикете, то сохраняем его
                affectedReleasesList.add(AquaTestIssueInfo(release, commentsByVersionMap.values.toList()))
            }
        }

        if (affectedReleasesList.isEmpty()) {
            logger.info("None of the version comments require updates (completed: $completedComments), exit")
            return
        }

        logger.info("Found comments requiring updates: $requireUpdateComments, new: $newComments, " +
            "completed: $completedComments in ${affectedReleasesList.size} tickets")

        // Получаем логи деплоя из версионики
        val versionDeployTimesMap: Map<String, DeployTimesInfo> = getVersionDeployTimesMapFromVersionica()

        // Для версии каждого найденного комментария ищем минимальное время деплоя
        val minTimes: List<Instant> = affectedReleasesList
            .flatMap { it.comments }
            .mapNotNull { versionDeployTimesMap[it.version] }
            .map { it.minDeployTime }
            .toList()

        // Если ни для кого ничего не нашли, значит остается только выйти. Такое может быть, если в комментарии
        // указаны несуществующие версии, либо ТС еще не обновилось, либо версии настолько старые,
        // что по ним не осталось логов
        if (minTimes.isEmpty()) {
            logger.warn("Unable to find min deploy time for any version from release tickets comments. exit")
            return
        }
        val minTime = minTimes.minOf { it }.minusSeconds(BEGIN_DEPLOY_LAG_IN_SECONDS.toLong())

        // Получаем из аквы все запуски, так до тех пор, пока не получим MAX_SUITES_COUNT запусков с датой
        // окончания менее minTime, или все запуски не кончатся
        logger.info("Trying to receive AQUA test launches for min deploy time '${DATE_TIME_FORMATTER.format(
            LocalDateTime.ofInstant(minTime, ZoneId.of("Europe/Moscow")))}'")
        val aquaLaunches: List<Launch> = getAquaLaunches(minTime)
        val changedReleasesList = mutableListOf<AquaTestIssueInfo>()
        var commentsForUpdate = 0
        // Теперь смотрим, можем ли мы что-то добавить к найденным комментариям
        for (affectedRelease in affectedReleasesList) {
            val updatedComments = mutableListOf<AquaTestCommentInfo>()
            for (aquaTestComment: AquaTestCommentInfo in affectedRelease.comments) {
                logger.info("Trying to update comment for version ${aquaTestComment.version}")
                val versionDeployTimes: DeployTimesInfo? = versionDeployTimesMap[aquaTestComment.version]
                if (versionDeployTimes == null) {
                    logger.warn("Unable to found deploy times in versionica for version ${aquaTestComment.version}" +
                        ", skipping")
                    continue
                }
                // Ищем ближайшие MIN_DEPLOYS_COUNT_PER_VERSION запуска теста из аквы до и после деплоя
                val aquaDeployLaunches: AquaDeployLaunches = findLaunches(aquaLaunches, versionDeployTimes)
                // Конвертируем их в информацию о запусках для комментариев
                val aquaDeployCommentData: AquaTestCommentData = convertAquaDeployLaunches(aquaDeployLaunches)
                // Обновляем текущий комментарий новой информацией
                val updatedCommentData: AquaTestCommentData = updateCommentData(
                    aquaTestComment.commentData, aquaDeployCommentData, versionDeployTimes)
                if (updatedCommentData != aquaTestComment.commentData) {
                    // Если содержание комментария изменилось, то добавляем его в список для обновления
                    logger.info("New information found for comment for version ${aquaTestComment.version}" +
                        ", adding to update list")
                    updatedComments.add(
                        AquaTestCommentInfo(aquaTestComment.version, aquaTestComment.comment, updatedCommentData))
                    commentsForUpdate++
                } else {
                    logger.info("Comment for version ${aquaTestComment.version} has no changes, skipping")
                }
            }
            if (updatedComments.isNotEmpty()) {
                changedReleasesList.add(AquaTestIssueInfo(affectedRelease.ticket, updatedComments))
            }
        }
        if (changedReleasesList.isEmpty()) {
            logger.info("All comments has no changes yet, exit")
            return
        }

        logger.info("Found $commentsForUpdate comment(s) in ${changedReleasesList.size} tickets " +
            "for insert or update")
        // Обновляем старые или добавляем новые комментарии
        for (changedRelease in changedReleasesList) {
            for (aquaTestComment: AquaTestCommentInfo in changedRelease.comments) {
                val newCommentText: String = renderCommentInfo(aquaTestComment)
                if (aquaTestComment.comment == null) {
                    logger.info("Trying to add new comment for version ${aquaTestComment.version}")
                    startrek.addComment(changedRelease.ticket, newCommentText, false, footer = false)
                    logger.info("Comment for version ${aquaTestComment.version} successfully added")
                } else {
                    logger.info("Trying to update comment for version ${aquaTestComment.version}")
                    startrek.updateCommentText(
                        changedRelease.ticket, aquaTestComment.comment, newCommentText)
                    logger.info("Comment for version ${aquaTestComment.version} successfully updated")
                }
            }
        }
    }

    /**
     * Получает логи деплоя из версионики
     */
    private fun getVersionDeployTimesMapFromVersionica(): Map<String, DeployTimesInfo> {
        val property = "yandex-direct\$"
        val hostGroup = listOf(
            "c:direct_perl_test_perl_web_rtc",
            "c:direct_perl_test_perl_intapi_rtc",
            "c:direct_perl_test_perl_api_rtc"
        )
        logger.info("Trying to get deploy properties logs from versionica")
        val propertiesLogs: List<PropertyLogEntry> = versionicaClient.getPropertyLogs(property, hostGroup)

        // Группируем логи деплоя по версии, и для каждой находим минимальное и максимальное время деплоя
        logger.info("${propertiesLogs.size} deploy property log entries received from versionica")
        val versionDeployTimesMap: Map<String, DeployTimesInfo> = createVersionDeployTimesMap(propertiesLogs)
        logger.info("${versionDeployTimesMap.size} versions found in property log entries received from versionica")
        return versionDeployTimesMap
    }

    /**
     * Рендерит комментарий с тестами от AQUA
     */
    private fun renderCommentInfo(commentInfo: AquaTestCommentInfo): String {
        val sbld: StringBuilder = StringBuilder()
        sbld.appendLine("$AQUA_TESTS_COMMENT_PREFIX**${commentInfo.version}**")
        sbld.appendLine()
        sbld.appendLine(AFTER_DEPLOY_TABLE_HEADER)
        sbld.appendLine()
        renderAquaTestLaunches(sbld, commentInfo.commentData.afterDeployLaunches)
        sbld.appendLine()
        sbld.appendLine(BEFORE_DEPLOY_TABLE_HEADER)
        sbld.appendLine()
        renderAquaTestLaunches(sbld, commentInfo.commentData.beforeDeployLaunches)
        sbld.appendLine()
        sbld.append("(($AQUA_ALL_TESTS_LINK $AQUA_ALL_TESTS_LINK_TEXT))")
        return sbld.toString()
    }

    /**
     * Рендерит список AQUA тестов и добавляет их в StringBuilder
     */
    private fun renderAquaTestLaunches(sbld: StringBuilder, launches: List<AquaTestLaunch>) {
        if (launches.isEmpty()) {
            sbld.appendLine(AQUA_TESTS_NOT_FOUND_TEXT)
            return
        }
        for (launch: AquaTestLaunch in launches) {
            renderAquaTestLaunch(sbld, launch)
        }
    }

    /**
     * Рендрит раскрывающийся блок с таблицей, в которой содержится информация о запуске тестов в AQUA
     */
    private fun renderAquaTestLaunch(sbld: StringBuilder, launch: AquaTestLaunch) {
        sbld.appendLine("<{${launch.totalTests}(!!(зел)**${launch.passedTests}**!!/" +
            "!!(крас)**${launch.failedTests + launch.revokedTests}**!!):")
        sbld.appendLine("#|")
        sbld.appendLine("||Время запуска|Всего тестов|Успешно|Выполняется|Отменено|С ошибкой|" +
            "&nbsp;&nbsp;Время завершения&nbsp;|Статус выполнения||")
        sbld.append("||")
        sbld.append(DATE_TIME_FORMATTER.format(LocalDateTime.ofInstant(
            launch.startTime, ZoneId.of("Europe/Moscow"))))
        var color = "зел"
        if (launch.failedTests + launch.revokedTests  > 0) {
            color = "крас"
        } else if (launch.runningTests > 0) {
            color = "син"
        }
        sbld.append("|%%(wacko wrapper=text align=center)!!($color)**${launch.totalTests}**!!%%")
        sbld.append("|%%(wacko wrapper=text align=center)!!(зел)${launch.passedTests}!!%%")
        appendTests(sbld, launch.runningTests, "син")
        appendTests(sbld, launch.revokedTests, "крас")
        appendTests(sbld, launch.failedTests, "крас")

        val stopTimeAsString = DATE_TIME_FORMATTER.format(LocalDateTime.ofInstant(
            launch.stopTime, ZoneId.of("Europe/Moscow")))

        if (launch.stopTimeIsEstimated) {
            sbld.append("|%%(wacko wrapper=text align=center)!!(зел)~~ ${stopTimeAsString}!!%%")
        } else {
            sbld.append("|%%(wacko wrapper=text align=center)${stopTimeAsString}%%")
        }
        sbld.append("|((${launch.launchUrl} ${launch.status.description}))")
        sbld.appendLine("||")
        sbld.appendLine("|#")
        sbld.appendLine("}>")
    }

    /**
     * Добавляет колонку с количеством тестов. Серую, если тестов 0, и жирную цвета color, если не 0
     */
    private fun appendTests(
        sbld: StringBuilder,
        testsCount: Int,
        color: String
    ) {
        var testText = ZERO_TESTS_TEXT
        if (testsCount > 0) {
            testText = "!!($color)**${testsCount}**!!"
        }
        sbld.append("|%%(wacko wrapper=text align=center)$testText%%")
    }

    /**
     * Парсит информацию о тестах в имеющемся комментарии
     */
    private fun parseCommentData(commentText: String): AquaTestCommentData {
        try {
            var indexBlockAfter = commentText.indexOf(AFTER_DEPLOY_TABLE_HEADER)
            indexBlockAfter += AFTER_DEPLOY_TABLE_HEADER.length

            var indexBlockBefore = commentText.indexOf(BEFORE_DEPLOY_TABLE_HEADER, indexBlockAfter)
            val indexEndOfCommentTables = commentText.lastIndexOf("|#")

            val afterDeployLaunches: List<AquaTestLaunch> =
                parseCommentLaunchTables(indexBlockAfter, indexBlockBefore, commentText)

            indexBlockBefore += BEFORE_DEPLOY_TABLE_HEADER.length

            val beforeDeployLaunches: List<AquaTestLaunch> =
                parseCommentLaunchTables(indexBlockBefore, indexEndOfCommentTables, commentText)

            return AquaTestCommentData(afterDeployLaunches, beforeDeployLaunches)
        } catch (e: Exception) {
            logger.error("Unable to parse comment from text:\n$commentText", e)
            // Если не удалось распарсить, то просто отдаем пустые данные комментария
            return AquaTestCommentData()
        }
    }

    /**
     * Парсит таблицы комментария с информацией о запусках тестов
     */
    private fun parseCommentLaunchTables(from: Int, to: Int, commentText: String): List<AquaTestLaunch> {
        val launches: MutableList<AquaTestLaunch> = mutableListOf()
        var pos = from
        while (pos < to) {
            pos = commentText.indexOf("||", pos)
            if (pos < 0 || pos >= to)
                break
            pos += 2
            if (pos < commentText.length && commentText[pos].isDigit()) {
                val tableEndIndex = commentText.indexOf("||", pos)
                if (tableEndIndex < 0 || tableEndIndex >= to)
                    break
                val launch: AquaTestLaunch = parseAquaTestLaunch(commentText.substring(pos, tableEndIndex))
                launches.add(launch)
                pos = tableEndIndex + 2
            }
        }
        return launches
    }

    /**
     * Парсит один запуск тестов из комментария
     */
    private fun parseAquaTestLaunch(str: String): AquaTestLaunch {
        val columns: List<String> = str.split('|')

        val startTime: Instant = findTimeInColumn(columns[0]).first
        val (stopTime: Instant, stopTimeIsEstimated: Boolean) = findTimeInColumn(columns[6])
        val totalTests = findIntInColumn(columns[1])
        val passedTests = findIntInColumn(columns[2])
        val runningTests = findIntInColumn(columns[3])
        val revokedTests = findIntInColumn(columns[4])
        val failedTests = findIntInColumn(columns[5])
        val (status: AquaTestLaunchStatus, launchUrl: String) = findStatusAndLaunchUrlInColumn(columns[7])
        return AquaTestLaunch(
            startTime,
            stopTime,
            stopTimeIsEstimated,
            totalTests,
            passedTests,
            runningTests,
            revokedTests,
            failedTests,
            status,
            launchUrl)
    }

    /**
     * Возвращает статус запуска тестов и ссылку на запуск из ячейки таблицы с запуском тестов из комментария
     */
    private fun findStatusAndLaunchUrlInColumn(column: String): Pair<AquaTestLaunchStatus, String> {
        val (statusAsString: String, url: String) = findLinkInColumn(column)
        val status: AquaTestLaunchStatus = convertToLaunchStatus(statusAsString)
        return Pair(status, url)
    }

    /**
     * Конвертирует строку с описанием статуса в статус запуска
     */
    private fun convertToLaunchStatus(statusAsString: String): AquaTestLaunchStatus =
        when (statusAsString) {
            AquaTestLaunchStatus.RUNNING.description -> AquaTestLaunchStatus.RUNNING
            AquaTestLaunchStatus.FINISHED.description -> AquaTestLaunchStatus.FINISHED
            AquaTestLaunchStatus.FAILED.description -> AquaTestLaunchStatus.FAILED
            AquaTestLaunchStatus.REVOKED.description -> AquaTestLaunchStatus.REVOKED
            else -> AquaTestLaunchStatus.UNKNOWN
        }

    /**
     * Возвращает текст и ссылку из ячейки таблицы с запуском тестов из комментария
     */
    private fun findLinkInColumn(column: String): Pair<String, String> {
        var index: Int = column.indexOf("((") + 2
        var endIndex: Int = column.indexOf(' ', index)
        val url: String = column.substring(index, endIndex)
        index = endIndex + 1
        endIndex = column.indexOf("))", index)
        val text: String = column.substring(index, endIndex).trim()
        return Pair(text, url)
    }

    /**
     * Возвращает целое число из ячейки таблицы с запуском тестов из комментария
     */
    private fun findIntInColumn(column: String): Int {
        val (startIndex: Int, endIndex: Int) = findFieldIndexes(column)
        return column.substring(startIndex, endIndex).toInt()
    }

    /**
     * Возвращает время и флаг приблизительности времени из ячейки таблицы с запуском тестов из комментария
     */
    private fun findTimeInColumn(column: String): Pair<Instant, Boolean> {
        val (startIndex: Int, endIndex: Int) = findFieldIndexes(column)
        val time: Instant = DateTimeUtils.moscowDateTimeToInstant(
            LocalDateTime.parse(column.substring(startIndex, endIndex), DATE_TIME_FORMATTER))
        // Символ тильды перед началом времени говорит о том, что время приблизительное.
        // Он может отстоять на пробел, поэтому проверяем два символа
        val isEstimated: Boolean = (startIndex > 1 && column[endIndex - 2] == '~')
            || (startIndex > 0 && column[startIndex - 1] == '~')
        return Pair(time, isEstimated)
    }

    /**
     * Ищет индексы начала и конца значения поля из ячейки таблицы с запуском тестов из комментария.
     * По факту ищет индекс первой цифры в колонке, а за ней индекс первого символа из следующих:
     * окончание выделения болдом '*', окончание подсветки '!', окончание блока '%', окончание колонки '|'.
     */
    private fun findFieldIndexes(str: String): Pair<Int, Int> {
        var begin = 0
        while (begin < str.length && !str[begin].isDigit()) {
            begin++
        }
        var end = begin + 1
        while (end < str.length && str[end] != '*' && str[end] != '!' && str[end] != '%' && str[end] != '|') {
            end++
        }
        return Pair(begin, end)
    }

    /**
     * Проверяет что количество запусков тестов в комментарии ровно [MAX_SUITES_COUNT], и среди запусков нет тех,
     * которые еще не завершены
     */
    private fun isCommentDataComplete(commentData: AquaTestCommentData): Boolean {
        return commentData.beforeDeployLaunches.size == MAX_SUITES_COUNT
            && commentData.afterDeployLaunches.size == MAX_SUITES_COUNT
            && commentData.beforeDeployLaunches.all { it.status != AquaTestLaunchStatus.RUNNING }
            && commentData.afterDeployLaunches.all { it.status != AquaTestLaunchStatus.RUNNING }
    }

    /**
     * Мержит данные из комментария с данными из запусков и возвращает обновленные данные комментария
     */
    private fun updateCommentData(
        commentData: AquaTestCommentData, aquaDeployCommentData: AquaTestCommentData, deployTimes: DeployTimesInfo
    ): AquaTestCommentData {
        val afterDeployLaunches: List<AquaTestLaunch> =
            mergeLaunches(aquaDeployCommentData.afterDeployLaunches, commentData.afterDeployLaunches)
            { it.startTime < deployTimes.maxDeployTime }.takeLast(MAX_SUITES_COUNT)
        val beforeDeployLaunches: List<AquaTestLaunch> =
            mergeLaunches(aquaDeployCommentData.beforeDeployLaunches, commentData.beforeDeployLaunches)
            { it.stopTime > deployTimes.minDeployTime }.take(MAX_SUITES_COUNT)
        return AquaTestCommentData(afterDeployLaunches, beforeDeployLaunches)
    }

    /**
     * Мержит запуски тестов из комментария и из AQUA
     */
    private fun mergeLaunches(
        aquaDeployLaunches: List<AquaTestLaunch>,
        commentLaunches: List<AquaTestLaunch>,
        commentLaunchesFilter: (AquaTestLaunch) -> Boolean
    ): List<AquaTestLaunch> {
        val mergedCommentLaunches: List<AquaTestLaunch>
        if (aquaDeployLaunches.size < MAX_SUITES_COUNT) {
            // Если из AQUA приехало недостаточное количество запусков тестов, то делаем карту запусков
            // по времени начала запуска. Записываем туда сначала отфильтрованные запуски тестов из комментария,
            // затем добавляем туда запуски тестов из AQUA, сортируем по времени в обратном порядке
            // и берем не более MAX_SUITES_COUNT штук. Фильтр для запусков тестов из комментария нужен потому,
            // что нужные нам запуски тестов зависят от времени деплоя, а оно могло поменяться со времени
            // создания комментария. Например, когда комментарий создавался, релиз мог быть выложен на две машины,
            // а сейчас уже на три, и время окончания деплоя сдвинулось, поэтому надо удалить из комментария те тесты,
            // которые начались до нового времени окончания деплоя. А запуски тестов из AQUA фильтровать не нужно,
            // они и так получены с учетом новых времен начала и окончания деплоя
            val launchesByTimeMap = commentLaunches.filter(commentLaunchesFilter)
                .associateBy { it.startTime }.toMutableMap()
            aquaDeployLaunches.forEach { launchesByTimeMap[it.startTime] = it }
            mergedCommentLaunches = launchesByTimeMap.values.toList()
        } else {
            // Если из AQUA приехало нужное количество запусков тестов, то просто заменяем тесты из комментария
            // на тесты из AQUA.
            mergedCommentLaunches = aquaDeployLaunches
        }
        return mergedCommentLaunches.sortedByDescending { it.startTime }
    }

    /**
     * Конвертирует запуски тестов из AQUA в данные для комментария
     */
    private fun convertAquaDeployLaunches(launches: AquaDeployLaunches): AquaTestCommentData {
        val afterDeployLaunches: List<AquaTestLaunch> = launches.afterDeployLaunches.map { convertLaunch(it) }.toList()
        val beforeDeployLaunches: List<AquaTestLaunch> =
            launches.beforeDeployLaunches.map { convertLaunch(it) }.toList()
        return AquaTestCommentData(afterDeployLaunches, beforeDeployLaunches)
    }

    /**
     * Конвертирует запуск тестов из AQUA в информацию о запуске для комментария
     */
    private fun convertLaunch(launch: Launch): AquaTestLaunch {
        val stopTimeIsEstimated: Boolean = launch.launchStatus in setOf(LaunchStatus.RUNNABLE, LaunchStatus.RUNNING)
        var stopTime: Instant = launch.stopTime
        if (stopTimeIsEstimated) {
            // Если тесты еще выполняются, то берем в качестве времени окончания ожидаемое время окончания
            stopTime = launch.estimateTime
            if (stopTime < launch.startTime) {
                // А если оно меньше времени начала (если в json вернулся 0), то установим его
                // как 5 минут после времени начала
                stopTime = launch.startTime.plusSeconds(300)
            }
        }
        return AquaTestLaunch(
            launch.startTime.trimToSeconds(),
            stopTime.trimToSeconds(),
            stopTimeIsEstimated,
            launch.totalSuites,
            launch.passedSuites,
            launch.runningSuites,
            launch.revokedSuites,
            launch.failedSuites,
            convertLaunchStatus(launch.launchStatus),
            launch.launchUrl)
    }

    /**
     * Переводит статус выполнения тестов из LaunchStatus в AquaTestLaunchStatus. В последнем все статусы
     * построения отчета из [LaunchStatus] учитываются как [AquaTestLaunchStatus.FINISHED], а [LaunchStatus.RUNNABLE]
     * как и [LaunchStatus.RUNNING] мапируется на [AquaTestLaunchStatus.RUNNING]
     */
    private fun convertLaunchStatus(launchStatus: LaunchStatus): AquaTestLaunchStatus =
        when (launchStatus) {
            LaunchStatus.RUNNABLE, LaunchStatus.RUNNING -> AquaTestLaunchStatus.RUNNING
            LaunchStatus.FINISHED, LaunchStatus.REPORT_REQUESTED, LaunchStatus.REPORT_STARTED,
            LaunchStatus.REPORT_FAILED, LaunchStatus.REPORT_READY ->
                AquaTestLaunchStatus.FINISHED
            LaunchStatus.FAILED -> AquaTestLaunchStatus.FAILED
            LaunchStatus.REVOKED -> AquaTestLaunchStatus.REVOKED
            else -> AquaTestLaunchStatus.FAILED
        }

    private fun Instant.trimToSeconds(): Instant {
        return Instant.ofEpochSecond(this.epochSecond)
    }

    /**
     * Находит максимум [MAX_SUITES_COUNT] запусков, созданных не ранее versionDeployTimes.maxDeployTime,
     * и максимум [MAX_SUITES_COUNT] запусков, завершившихся до
     * versionDeployTimes.minDeployTime - [BEGIN_DEPLOY_LAG_IN_SECONDS]
     */
    private fun findLaunches(launches: List<Launch>, deployTimes: DeployTimesInfo): AquaDeployLaunches {
        var index = launches.binarySearchBy(deployTimes.maxDeployTime) { launch -> launch.startTime }
        if (index < 0) {
            index = -index - 1
        }
        var beforeIndex = index - 1
        val afterDeployLaunches = mutableListOf<Launch>()
        var found = 0
        while (index < launches.size && found < MAX_SUITES_COUNT) {
            if (launches[index].startTime >= deployTimes.maxDeployTime) {
                afterDeployLaunches.add(launches[index])
                found++
            }
            index++
        }
        val beforeDeployLaunches = mutableListOf<Launch>()
        found = 0
        val minDeployTime = deployTimes.minDeployTime.minusSeconds(BEGIN_DEPLOY_LAG_IN_SECONDS.toLong())
        while (beforeIndex >= 0 && found < MAX_SUITES_COUNT) {
            if (launches[beforeIndex].stopTime < minDeployTime) {
                beforeDeployLaunches.add(launches[beforeIndex])
                found++
            }
            beforeIndex--
        }
        return AquaDeployLaunches(afterDeployLaunches, beforeDeployLaunches)
    }

    /**
     * Возвращает все запуски тестов в AQUA для пака с идентификатором [AQUA_TEST_PACK_ID] от самого нового, до тех пор,
     * пока не получится добавить [MAX_SUITES_COUNT] запусков с датой окончания менее [minTime],
     * или все запуски не кончатся
     */
    private fun getAquaLaunches(minTime: Instant): List<Launch> {
        var skip = 0
        var total = Int.MAX_VALUE
        var addedSuitsEndedBeforeMinTime = 0
        // так как между запросами очередной порции запусков их состав может измениться, нужно подстраховаться,
        // чтобы не добавлять повторно полученные запуски
        val addedLaunchesSet = mutableSetOf<Instant>()
        val launches = mutableListOf<Launch>()
        while (skip < total && addedSuitsEndedBeforeMinTime < MAX_SUITES_COUNT) {
            // Получаем очередную порцию тестов, отсортированных по времени в обратном порядке
            val launchesResponse: GetLaunchesResponse =
                aquaClient.getPackLaunches(skip, TAKE_LAUNCHES_IN_PACK_BY_QUERY_LIMIT, AQUA_TEST_PACK_ID)
            if (launchesResponse.launches.isEmpty()) {
                // Если ничего не вернулось, перестаем получать тесты, чтобы не застрять в цикле навечно
                break
            }
            skip += launchesResponse.launches.size
            total = launchesResponse.count
            for (launch: Launch in launchesResponse.launches) {
                if (addedLaunchesSet.contains(launch.createdTime)) {
                    // Пропускаем уже добавленные запуски тестов
                    continue
                }
                addedLaunchesSet.add(launch.createdTime)
                launches.add(launch)
                // У не завершившихся тестов stopTime = 0, поэтому для увеличения счетчика найденных
                // завершившихся тестов они не годятся, нужно отсеять
                if (launch.stopTime > launch.startTime &&
                    launch.stopTime < minTime) {
                    addedSuitsEndedBeforeMinTime++
                }
                if (addedSuitsEndedBeforeMinTime >= MAX_SUITES_COUNT) {
                    // Если набрали нужные тесты, можно закругляться
                    break
                }
            }
        }
        // Сортируем список запусков по времени начала запуска, чтобы по нему нормально работал
        // встроенный бинарный поиск.
        return launches.sortedBy { it.startTime }
    }

    /**
     * Проходит по списку логов деплоя, для каждой версии вычисляет минимальное и максимальное время деплоя, а так же
     * количество деплоев и возвращает карту времен по версии деплоя, для версий где деплоев не менее
     * чем [MIN_DEPLOYS_COUNT_PER_VERSION]
     */
    private fun createVersionDeployTimesMap(
        propertiesLogs: List<PropertyLogEntry>
    ): Map<String, DeployTimesInfo> {
        val versionDeployTimesMap = mutableMapOf<String, DeployTimesInfo>()
        for (propertyLog: PropertyLogEntry in propertiesLogs) {
            if (propertyLog.action != "update") {
                continue
            }
            val versionDeployTimes: DeployTimesInfo =
                versionDeployTimesMap.computeIfAbsent(propertyLog.value) { DeployTimesInfo() }
            versionDeployTimes.minDeployTime = minOf(versionDeployTimes.minDeployTime, propertyLog.logtime)
            versionDeployTimes.maxDeployTime = maxOf(versionDeployTimes.maxDeployTime, propertyLog.logtime)
            versionDeployTimes.deploysCount++
            versionDeployTimes.logs.add(propertyLog)
        }
        return versionDeployTimesMap.filter { it.value.deploysCount >= MIN_DEPLOYS_COUNT_PER_VERSION }
    }

    /**
     * Берет первую строку из комментария, пытается по ней понять, не является ли этот комментарий - комментарием
     * с тестами AQUA для определенной версии, и если является, то извлекает из строки версию и возвращает ее.
     * А если нет, то возвращает null
     */
    private fun checkIsTrafficLightCommentAndGetVersion(comment: Comment): String? {
        if (comment.text.isEmpty())
            return null
        val commentText: String = comment.text.orElse("")
        var version: String? = null
        if (commentText.startsWith(AQUA_TESTS_COMMENT_PREFIX)) {
            version = extractVersion(commentText.lineSequence().first())
        }
        return version
    }

    /**
     * Пытается найти в строке последнюю версию деплоя. И если находит, то возвращает ее.
     * А если не находит, то возвращает null
     */
    private fun extractVersion(str: String?): String? {
        if (str == null) {
            return null
        }
        return VERSION_REGEX.findAll(str).map { match -> match.value }.lastOrNull()
    }

    private class DeployTimesInfo(val logs: MutableList<PropertyLogEntry> = mutableListOf()) {
        var minDeployTime: Instant = Instant.MAX
        var maxDeployTime: Instant = Instant.MIN
        var deploysCount: Int = 0
    }

    private data class AquaDeployLaunches(
        val afterDeployLaunches: List<Launch>,
        val beforeDeployLaunches: List<Launch>,
    )
}
