package ru.yandex.direct.chassis.entity.buildbot

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.buildbot.BuildBotConstants.PERL_UNIT_TEST_FAILURES_TAG
import ru.yandex.direct.chassis.entity.buildbot.BuildBotConstants.buildbotBuild
import ru.yandex.direct.chassis.entity.buildbot.BuildBotConstants.commitUrl
import ru.yandex.direct.chassis.entity.buildbot.BuildBotConstants.svnlogUrl
import ru.yandex.direct.chassis.entity.startrek.commentSummonee
import ru.yandex.direct.chassis.properties.PropertiesLightSupport
import ru.yandex.direct.chassis.repository.BuildbotBuildRepository
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.Utils
import ru.yandex.direct.chassis.util.buildbot.BuildbotClient
import ru.yandex.direct.dbschema.chassis.enums.BuildbotBuildsStatus
import ru.yandex.startrek.client.model.Issue
import java.time.Duration
import java.time.Instant
import java.time.LocalDateTime
import java.time.ZoneId

object BuildBotConstants {
    /**
     * Текущее количество различных паков в buildbot
     */
    const val TARGET_BUILD_COUNT = 14

    const val PERL_UNIT_TEST_FAILURES_TAG = "perl_unit_test_failures"

    fun commitUrl(revision: Long) = "https://a.yandex-team.ru/arc/commit/$revision"

    fun buildbotBuild(build: BuildInfo) =
        "https://direct.ppcbuild.ppc.yandex.ru/builders/${build.builder}/builds/${build.buildNumber}"

    fun svnlogUrl() = "https://javadirect-dev.yandex-team.ru/svnlog/?app_id=14&release_id=-1&days=7&since=&till=&" +
            "branch=%2Ftrunk&file_regexp=trunk%2Farcadia%2Fdirect%2Fperl&author=&startrek_query="
}

@Component
class BuildbotService(
    private val startrek: StartrekHelper,
    private val buildbotClient: BuildbotClient,
    private val buildbotBuildRepository: BuildbotBuildRepository,
    private val ppcPropertiesLightSupport: PropertiesLightSupport,
) {
    private val logger = LoggerFactory.getLogger(BuildbotService::class.java)

    private val buildbotCommentsEnabled
        get() = ppcPropertiesLightSupport["buildbot_comments_enabled"] in listOf("true", "1")

    fun process(packets: List<BuildbotPacket>) {
        logger.info("Buildbot packets: {}", packets)

        val builds = saveBuilds(packets)
        notify(builds)
    }

    private fun saveBuilds(packets: List<BuildbotPacket>): List<BuildInfo> {
        val buildPackets = packets
            .filter { it.event == "buildFinished" }

        val builds: List<BuildInfo> = buildPackets
            .mapNotNull { packet -> packet.payload?.build?.let { build -> toBuildInfo(build) } }
            .also { logger.info("All builds: {}", it) }
            .filter { build -> build.branch == "trunk" }

        buildbotBuildRepository.insertBuildInfo(builds)

        return builds
    }

    private fun toBuildInfo(build: BuildbotBuild): BuildInfo? {
        val buildNumber = build.buildnumber?.toLongOrNull() ?: return null

        val builder = build.builderName ?: return null

        val branch = build.branch
            .let { if (it.isNullOrEmpty()) "trunk" else it }

        val revision = build.revision?.toLongOrNull() ?: return null

        val statusText = build.text?.joinToString(" ") ?: "unknown"

        val status = when {
            statusText == "build successful" -> BuildbotBuildsStatus.success
            statusText.contains("failed") -> BuildbotBuildsStatus.failed
            else -> BuildbotBuildsStatus.unknown
        }

        val commitMessage = extractCommitMessage(build)

        val (startTime, endTime) = build.times.map {
            LocalDateTime.ofInstant(
                Instant.ofEpochSecond(it.toLong()),
                ZoneId.systemDefault()
            )
        }

        return BuildInfo(
            buildNumber = buildNumber,
            builder = builder,
            branch = branch,
            revision = revision,
            status = status,
            statusText = statusText,
            commitMessage = commitMessage,
            startTime = startTime,
            endTime = endTime,
            restarted = false,
        )
    }

    private fun extractCommitMessage(build: BuildbotBuild): String? {
        return build.sourceStamps?.get(0)?.changes?.get(0)?.comments
    }

    private fun notify(builds: List<BuildInfo>) {
        val revisions = builds.map { it.revision }.toSet()
        val allBuilds = buildbotBuildRepository.getBuildsForRevision(revisions)
            .groupBy { it.revision }
        allBuilds.forEach { (revision, builds) -> notifyRevision(revision, builds) }
    }

    private fun notifyRevision(revision: Long, builds: List<BuildInfo>) {
        val buildsByBuilder: Map<String, List<BuildInfo>> = builds
            .groupBy { it.builder }

        val lastBuilds: List<BuildInfo> = buildsByBuilder
            .map { (_, builds) -> builds.maxByOrNull { it.startTime }!! }

        // Еще не все билды завершились
        if (lastBuilds.size != BuildBotConstants.TARGET_BUILD_COUNT) {
            logger.info("Not all builds are finished for $revision")
            return
        }

        // Некоторые билды были перезапущены, и еще не добежали
        if (lastBuilds.any { it.restarted }) {
            logger.info("Some restarted builds are still running for revision $revision")
            return
        }

        logger.info("All builds are finished for revision $revision")

        // Рестартим билд, если это первый упавший запуск
        val buildsToRestart = buildsByBuilder.values
            .filter { it.size == 1 }
            .map { it.first() }
            .filter { it.status == BuildbotBuildsStatus.failed }

        if (buildsToRestart.isNotEmpty()) {
            restartBuilds(revision, buildsToRestart)
            return
        }

        // Если же билд со второго раза не прошел, то оповещаем
        val failedBuilds = lastBuilds
            .filter { it.status == BuildbotBuildsStatus.failed }

        if (failedBuilds.isEmpty()) {
            logger.info("No failed builds for revision $revision")
            return
        }

        logger.info("Notifying on failed builds for revision $revision")
        notifyOnBuildFailure(revision, failedBuilds)
    }

    private fun restartBuilds(revision: Long, builds: List<BuildInfo>) {
        logger.info(
            "Restarting builders {} for revision {}",
            builds.joinToString(", ") { it.builder },
            revision,
        )

        for (build in builds) {
            Utils.retry(3, Duration.ofSeconds(3)) {
                buildbotClient.restartBuild(build.builder, build.buildNumber)
            }
        }

        val buildIds = builds.map { it.id }
        buildbotBuildRepository.setRestarted(buildIds)
    }

    private fun notifyOnBuildFailure(revision: Long, failedBuilds: List<BuildInfo>) {
        val commitMessage = failedBuilds
            .map { it.commitMessage }
            .firstOrNull()
        if (commitMessage == null) {
            logger.info("No commit message for revision $revision")
            return
        }

        val issueKey = Utils.getTicketByCommitMessage(commitMessage)
        if (issueKey == null) {
            logger.info("Could not find ticket in commit message, skipping build")
            return
        }

        val issue = startrek.session.issues().get(issueKey)
        if (issue == null) {
            logger.info("Can't find issue $issueKey")
            return
        }

        notifyIssue(issue, revision, failedBuilds)
    }

    private fun notifyIssue(
        issue: Issue,
        revision: Long,
        failedBuilds: List<BuildInfo>,
    ) {
        val status = failedBuilds
            .joinToString("\n") { build -> "!!**${build.builder}**!!: ${buildbotBuild(build)}" }

        val message = """
            |В коммите ((${commitUrl(revision)} r$revision)) под этим тикетом падают юнит-тесты в транке! Если падение вызвано правками в этом коммите, необходимо как можно быстрее это исправить.
            |Падения юнит-тестов в транке блокируют сборку релизов!
            |
            |$status
            |
            |<{Что делать с этим комментарием:
            | 1. Открой ((${svnlogUrl()} svnlog)) и посмотри когда начали падать unit-тесты (при успешном прохождении тестов должно быть 14 зеленых смайликов).
            | 2. Открой логи падений по ссылкам из этого комментария, из них обычно понятна причина падения тестов.
            | 3. Если падение тестов вызвано этим коммитом, как можно быстрее это исправь!
            | 4. Если же тесты падали до этого коммита и/или в логе нет падений связанных с этим коммитом, то этот комментарий можно игнорировать.
            |}>
        """.trimMargin()

        logger.info("Issue: ${issue.key}, message: $message")

        if (!buildbotCommentsEnabled) {
            return
        }

        Utils.retry(tries = 3, pause = Duration.ofSeconds(3)) {
            startrek.addComment(issue, message, summonee = issue.commentSummonee, ifNoTag = PERL_UNIT_TEST_FAILURES_TAG)
        }
    }
}
