package ru.yandex.direct.chassis.entity.deploy

import com.google.protobuf.Timestamp
import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.entity.deploy.model.DeployInfo
import ru.yandex.direct.chassis.entity.deploy.model.DeployNotification
import ru.yandex.direct.chassis.entity.deploy.repository.DeployNotificationRepository
import ru.yandex.direct.chassis.entity.release.model.ReleaseInfo
import ru.yandex.direct.chassis.entity.release.service.ReleaseService
import ru.yandex.direct.chassis.entity.telegram.TelegramChats
import ru.yandex.direct.chassis.entity.telegram.TelegramUtils.htmlLink
import ru.yandex.direct.chassis.entity.telegram.TelegramUtils.issueLink
import ru.yandex.direct.chassis.util.DeployStage
import ru.yandex.direct.chassis.util.DirectAppsConfEntry
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.util.ReleaseVersion
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.telegram.client.TelegramClient
import ru.yandex.direct.telegram.client.api.EditMessageTextRequest
import ru.yandex.direct.telegram.client.api.ParseMode
import ru.yandex.direct.telegram.client.api.executeTelegram
import ru.yandex.direct.utils.DateTimeUtils.instantToMoscowDateTime
import java.time.Instant
import java.time.format.DateTimeFormatter
import java.time.temporal.ChronoUnit

@Hourglass(periodInSeconds = 60)
class DeployNotificationJob(
    private val releaseService: ReleaseService,
    private val deployService: YDeployService,
    private val telegram: TelegramClient,
    private val deployNotificationRepository: DeployNotificationRepository,
    private val directAppsConfProvider: DirectAppsConfProvider,
) : DirectJob() {

    private val logger = LoggerFactory.getLogger(DeployNotificationJob::class.java)

    private val chatId = TelegramChats.DIRECT_ADMIN_CHAT

    override fun execute() {
        val releases: Map<String, List<ReleaseInfo>> = releaseService.getReadyToDeployReleases()
            .groupBy { it.app.name }

        val notifications: Map<String, List<DeployNotification>> =
            deployNotificationRepository.getUnfinishedNotifications()
                .groupBy { it.app }

        val apps = notifications.keys + releases.keys
        apps.forEach { appName ->
            val app: DirectAppsConfEntry = directAppsConfProvider.getDirectAppsConf()
                .first { it.name == appName }
            processApp(app, releases[appName].orEmpty(), notifications[appName].orEmpty())
        }
    }

    private fun processApp(
        app: DirectAppsConfEntry,
        releases: List<ReleaseInfo>,
        notifications: List<DeployNotification>,
    ) {
        logger.info("Processing releases for app ${app.name}")

        val stage = getStage(app) ?: run {
            logger.warn("Non-supported app: ${app.name}, missing stage or resource type")
            return
        }

        val deployInfo = deployService.getDeployInfo(stage, app.yaDeployResourceType!!) ?: run {
            logger.warn("Failed to get deploy info for app ${app.name}")
            return
        }
        logger.info("Deploy info: $deployInfo")

        notifications.forEach { notification ->
            updateNotification(deployInfo, notification)
        }

        val releaseInfo = releases
            .find { it.version.toString() == deployInfo.version }
            ?: run {
                logger.warn("No release issue with matching version. Releases: $releases, deploy info: $deployInfo")
                return
            }

        if (deployInfo.isNew()) {
            logger.info("Deploy ticket progress is not started, skipping")
            return
        }

        val existingNotification = deployNotificationRepository.getNotification(app.name, releaseInfo.issue.key)
        if (existingNotification != null) {
            logger.info("Notification for release ${releaseInfo.issue.key}")
            return
        }

        val notification = createNotification(releaseInfo, deployInfo)
        deployNotificationRepository.insertNotification(notification)
    }

    private fun getStage(app: DirectAppsConfEntry): String? {
        val stages: List<String> = app.stages[DeployStage.PRODUCTION].orEmpty()
        if (stages.size != 1) {
            logger.warn("Stages for app ${app.name} are not supported: $stages")
            return null
        }
        return stages.single()
    }

    private fun updateNotification(currentDeployInfo: DeployInfo, notification: DeployNotification) {
        val releaseInfo = releaseService.getReleaseInfo(notification.issueKey) ?: run {
            logger.error("Failed to find release for existing notification: $notification")
            return
        }

        val deployInfo = deployService.getDeployInfoForRelease(
            currentDeployInfo,
            releaseInfo.app.yaDeployResourceType!!,
            notification.releaseId
        ) ?: run {
            logger.error("Failed to find deploy info for existing notification: $notification")
            return
        }

        val text = formatNotification(releaseInfo, deployInfo)
        logger.info("Editing notification: $text")

        val response = telegram.api.editMessageText(
            EditMessageTextRequest(
                chatId = chatId,
                messageId = notification.messageId,
                text = text,
                parseMode = ParseMode.HTML,
            )
        ).executeTelegram()

        if (!response.ok) {
            logger.error("Error response on edit: $response")
        }

        if (deployInfo.isFinished()) {
            logger.info("Deploy ticket is finished, updating notification")
            deployNotificationRepository.updateNotification(notification.copy(finished = true))
        }
    }

    private fun createNotification(releaseInfo: ReleaseInfo, deployInfo: DeployInfo): DeployNotification {
        val text = formatNotification(releaseInfo, deployInfo)

        logger.info("Sending notification: $text")
        val message = telegram.sendMessage(chatId, text, parseMode = ParseMode.HTML)

        return DeployNotification(
            app = releaseInfo.app.name,
            releaseId = deployInfo.release.meta.id,
            issueKey = releaseInfo.issue.key,
            messageId = message.messageId,
            finished = deployInfo.isFinished(),
        )
    }

    private fun formatNotification(releaseInfo: ReleaseInfo, deployInfo: DeployInfo): String {
        val login = extractLogin(deployInfo.release.spec.title, deployInfo.release.spec.sandbox.releaseAuthor)

        val issueLink = issueLink(releaseInfo.issue)
        val ciLink = htmlLink("CI", formatCiLink(releaseInfo.app, releaseInfo.version))
        val deployLink = htmlLink("deploy", "https://deploy.yandex-team.ru/stages/${deployInfo.stage.meta.id}")

        val dateTimeFormat = DateTimeFormatter.ofPattern("dd.MM.yyyy HH:mm:ss")

        val startTime: Long = deployInfo.deployTicket.meta.creationTime
        val startDateTime =
            instantToMoscowDateTime(Instant.EPOCH.plus(startTime, ChronoUnit.MICROS)).truncatedTo(ChronoUnit.SECONDS)

        val endTime: Timestamp? = deployInfo.deployTicket.status.progress.endTime
        val endDateTime = instantToMoscowDateTime(Instant.ofEpochSecond(endTime!!.seconds))

        val podsReady = deployInfo.deployUnitStatus.progress.podsReady
        val podsTotal = deployInfo.deployUnitStatus.progress.podsTotal
        val waitingForApprove = if (deployInfo.isWaitingForApprove()) ", ожидает подтверждения" else ""

        return when {
            !deployInfo.isFinished() -> """
                ⏳ Идет выкладка <b>${releaseInfo.app.name}</b> ${releaseInfo.version}: $login@
                Начало: ${startDateTime.format(dateTimeFormat)}
                Прогресс: $podsReady/$podsTotal$waitingForApprove
                $issueLink, $ciLink, $deployLink
            """.trimIndent()

            deployInfo.isSuccessful() -> """
                ✅ Выложен релиз <b>${releaseInfo.app.name}</b> ${releaseInfo.version}: $login@
                Начало: ${startDateTime.format(dateTimeFormat)}
                Конец: ${endDateTime.format(dateTimeFormat)}
                $issueLink, $ciLink, $deployLink
            """.trimIndent()

            else -> """
                ❌ Не завершена выкладка <b>${releaseInfo.app.name}</b> ${releaseInfo.version}: $login@
                Начало: ${startDateTime.format(dateTimeFormat)}
                Конец: ${endDateTime.format(dateTimeFormat)}
                $issueLink, $ciLink, $deployLink
            """.trimIndent()
        }
    }


    private fun extractLogin(title: String, author: String): String {
        val titleRegex = """^([a-zA-Z\d_-]+)@.*""".toRegex()
        val match = titleRegex.matchEntire(title)
            ?: return author
        val (_, login) = match.groupValues
        return login
    }


    private fun formatCiLink(app: DirectAppsConfEntry, version: ReleaseVersion): String {
        val ciPathRegex = "dir=(.*)&".toRegex()
        val dir: String? = app.newCiGraph
            ?.let { ciPathRegex.find(app.newCiGraph) }
            ?.groupValues?.get(1)

        return if (dir != null && version.isCi) {
            "https://a.yandex-team.ru/projects/direct/ci/releases/flow?dir=$dir&id=deploy-release&version=$version"
        } else {
            "https://a.yandex-team.ru/projects/direct/ci/releases"
        }
    }
}
