package ru.yandex.direct.chassis.entity.deploy

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.util.DirectAppsConfEntry
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.util.DirectReleaseUtils
import ru.yandex.direct.chassis.util.InvalidAppConfig
import ru.yandex.direct.chassis.util.startrek.StartrekQueue.DIRECT
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey.READY_TO_DEPLOY
import ru.yandex.startrek.client.Session
import ru.yandex.startrek.client.model.CommentCreate
import ru.yandex.startrek.client.model.Issue
import ru.yandex.startrek.client.model.IssueUpdate

class DirectAppRelease(val issue: Issue, val app: DirectAppsConfEntry, val version: String) {
    private val trackerDeployedTag = app.trackerDeployedTag
        ?: throw InvalidAppConfig("missing deployed tag for tracker")

    fun closeIssue(comment: String) {
        val issueUpdateBuilder =
            IssueUpdate.builder().comment(CommentCreate.builder().comment(comment).build()).resolution("fixed")
        if (trackerDeployedTag !in issue.tags.toSet()) {
            val tags: Array<String> = issue.tags.toTypedArray()
            issueUpdateBuilder.tags(*tags, trackerDeployedTag)
        }
        val issueUpdate = issueUpdateBuilder.build()
        issue.executeTransition("deploy", issueUpdate)
    }
}

class InvalidDirectReleaseError(message: String) : Exception(message)

@Component
class DirectReleasesService(
    private val startrekSession: Session,
    private val directAppsConfProvider: DirectAppsConfProvider,
) {
    companion object {
        private val logger = LoggerFactory.getLogger(DirectReleasesService::class.java)
    }

    private val appsByTrackerComponent: Map<String, DirectAppsConfEntry>
        get() = directAppsConfProvider.getDirectAppsConf()
            .associateBy { it.trackerComponent }

    private fun getDirectAppReleaseObjectFromIssue(issue: Issue): DirectAppRelease {
        val directAppComponents =
            issue.components.orEmpty().mapNotNull { it.display }.filter { it in appsByTrackerComponent }
        if (directAppComponents.size > 1) throw InvalidDirectReleaseError(
            "more than 1 app component: " + directAppComponents.joinToString(", ")
        )
        val appComponent = directAppComponents.getOrNull(0) ?: throw InvalidDirectReleaseError("missing app component")
        val app = appsByTrackerComponent[appComponent] ?: error("no app for component $appComponent")

        val releaseVersion = DirectReleaseUtils.getReleaseVersionString(issue.summary)
            ?: throw InvalidDirectReleaseError("could not parse version from summary: ${issue.summary}")

        return DirectAppRelease(issue = issue, app = app, version = releaseVersion)
    }

    fun getReadyReleasesByApp(): Map<DirectAppsConfEntry, List<DirectAppRelease>> {
        val query =
            """Queue: $DIRECT Type: Release Status: $READY_TO_DEPLOY Components: "Releases: Direct", "Releases: JavaDirect", "App: uac""""
        logger.debug("getting new releases from Startrek with query: $query")

        val readyReleasesByApp: Map<DirectAppsConfEntry, List<DirectAppRelease>> =
            startrekSession.issues().find(query).toList().mapNotNull { issue ->
                try {
                    getDirectAppReleaseObjectFromIssue(issue)
                } catch (e: InvalidDirectReleaseError) {
                    logger.warn("skipping invalid release ${issue.key}: ${e.message}")
                    null
                }
            }.filter { it.app.deployType == "yadeploy" }.groupBy { it.app }
        return readyReleasesByApp
    }
}
