package ru.yandex.direct.chassis.entity.deploy

import org.springframework.stereotype.Component
import ru.yandex.direct.dbschema.chassis.enums.ReleasesDeployQueueStatus
import ru.yandex.direct.dbschema.chassis.tables.ReleasesDeployQueue.RELEASES_DEPLOY_QUEUE
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.utils.JsonUtils
import java.time.LocalDateTime

data class QueueItem(
    var status: ReleasesDeployQueueStatus,
    var state: State?,
    var createTime: LocalDateTime,
    var login: String
)

data class State(
    var started: Boolean = false,
    var released: Boolean = false,
    var commited: Boolean = false,
    var progress: Int = 0,
    var deployed: Boolean = false,
    var errors: String? = null,
)


@Component
class ReleasesDeployQueueRepository(private val dslContextProvider: DslContextProvider) {

    fun getByTicket(releaseTicket: String): QueueItem? {
        return dslContextProvider.chassis()
            .select(
                RELEASES_DEPLOY_QUEUE.STATUS,
                RELEASES_DEPLOY_QUEUE.STATE,
                RELEASES_DEPLOY_QUEUE.CREATE_TIME,
                RELEASES_DEPLOY_QUEUE.LOGIN
            ).from(RELEASES_DEPLOY_QUEUE)
            .where(RELEASES_DEPLOY_QUEUE.RELEASE_TICKET.eq(releaseTicket))
            .fetchOne {
                val result = QueueItem(
                    it.getValue(RELEASES_DEPLOY_QUEUE.STATUS),
                    it.getValue(RELEASES_DEPLOY_QUEUE.STATE)?.let { s -> JsonUtils.fromJson(s, State::class.java) },
                    it.getValue(RELEASES_DEPLOY_QUEUE.CREATE_TIME),
                    it.getValue(RELEASES_DEPLOY_QUEUE.LOGIN)
                )
                result
            }
    }

    fun getReleasesForDeploy(): Map<String, State> {
        return dslContextProvider.chassis()
            .select(
                RELEASES_DEPLOY_QUEUE.RELEASE_TICKET,
                RELEASES_DEPLOY_QUEUE.STATE
            ).from(RELEASES_DEPLOY_QUEUE)
            .where(
                RELEASES_DEPLOY_QUEUE.STATUS.`in`(
                    ReleasesDeployQueueStatus.Ready,
                    ReleasesDeployQueueStatus.InProgress
                )
            )
            .orderBy(RELEASES_DEPLOY_QUEUE.CREATE_TIME)
            .fetchMap(RELEASES_DEPLOY_QUEUE.RELEASE_TICKET) {
                val state = it.getValue(RELEASES_DEPLOY_QUEUE.STATE)
                state?.let { s -> JsonUtils.fromJson(s, State::class.java) }
                    ?: State()
            }
    }

    fun addToQueue(releaseTicket: String, login: String) {
        dslContextProvider.chassis()
            .insertInto(RELEASES_DEPLOY_QUEUE)
            .columns(
                RELEASES_DEPLOY_QUEUE.STATUS,
                RELEASES_DEPLOY_QUEUE.RELEASE_TICKET,
                RELEASES_DEPLOY_QUEUE.LOGIN,
                RELEASES_DEPLOY_QUEUE.STATE
            )
            .values(
                ReleasesDeployQueueStatus.Ready,
                releaseTicket,
                login,
                JsonUtils.toJson(State()),
            )
            .execute()
    }

    fun updateStatus(releaseTicket: String, status: ReleasesDeployQueueStatus) {
        dslContextProvider.chassis()
            .update(RELEASES_DEPLOY_QUEUE)
            .set(RELEASES_DEPLOY_QUEUE.STATUS, status)
            .where(RELEASES_DEPLOY_QUEUE.RELEASE_TICKET.eq(releaseTicket))
            .execute()
    }

    fun saveState(releaseTicket: String, status: ReleasesDeployQueueStatus, state: State) {
        dslContextProvider.chassis()
            .update(RELEASES_DEPLOY_QUEUE)
            .set(RELEASES_DEPLOY_QUEUE.STATE, JsonUtils.toJson(state))
            .set(RELEASES_DEPLOY_QUEUE.STATUS, status)
            .where(RELEASES_DEPLOY_QUEUE.RELEASE_TICKET.eq(releaseTicket))
            .execute()
    }
}
