package ru.yandex.direct.chassis.entity.deploy

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.deploy.model.DeployInfo
import ru.yandex.yp.YpRawClient
import ru.yandex.yp.client.api.Autogen.TDeployTicket
import ru.yandex.yp.client.api.Autogen.TRelease
import ru.yandex.yp.client.api.Autogen.TStage
import ru.yandex.yp.client.api.TDeployUnitStatus
import ru.yandex.yp.client.api.TPodTemplateSpec
import ru.yandex.yp.client.pods.TLayer
import ru.yandex.yp.model.YpGetManyStatement
import ru.yandex.yp.model.YpGetStatement
import ru.yandex.yp.model.YpObjectType
import ru.yandex.yp.model.YpPayload
import ru.yandex.yp.model.YpSelectStatement
import ru.yandex.yp.model.YpTypedId
import java.time.Duration
import java.time.Instant
import java.util.concurrent.CompletableFuture
import java.util.concurrent.TimeUnit

@Component
class YDeployService(
    private val ypClient: YpRawClient,
) {

    companion object {
        private val logger = LoggerFactory.getLogger(YDeployService::class.java)

        private const val RESOURCE_VERSION_ATTRIBUTE = "resource_version"
    }

    fun getDeployInfoForRelease(deployInfo: DeployInfo, resourceType: String, releaseId: String): DeployInfo? {
        val release: TRelease? = getReleasesById(listOf(releaseId)).await().firstOrNull()
        if (release == null) {
            logger.info("Failed to find release with release id $releaseId, deploy info: $deployInfo")
            return null
        }

        val releaseVersion: String? = findResourceVersion(release, resourceType)
        if (releaseVersion == null) {
            logger.info("Failed to find resource version for release $release")
            return null
        }

        val deployTicket: TDeployTicket = getDeployTicketsForRelease(deployInfo.stage, release).await()
            .minByOrNull { it.meta.creationTime }!!

        return deployInfo.copy(
            version = releaseVersion,
            release = release,
            deployTicket = deployTicket,
        )
    }

    fun getDeployInfo(stageName: String, resourceType: String): DeployInfo? {
        val stage = getStage(stageName).await()
            ?: return null

        val deployUnitStatus: TDeployUnitStatus = getDeployUnitStatus(stage)
            ?: return null

        val deployUnitVersion = findResourceVersion(deployUnitStatus, resourceType)
        if (deployUnitVersion == null) {
            logger.info("Failed to find resource version for deploy unit: $deployUnitStatus")
            return null
        }

        val deployTickets: List<TDeployTicket> = getRecentDeployTickets(stage).await()
        val releaseIds: Set<String> = deployTickets.map { it.spec.releaseId }.toSet()

        val release: TRelease? = getReleasesById(releaseIds).await()
            .filter { release -> findResourceVersion(release, resourceType) == deployUnitVersion }
            .minByOrNull { it.meta.creationTime }
        if (release == null) {
            logger.info("Failed to find release for version $deployTickets, deploy tickets: $deployTickets")
            return null
        }

        val deployTicket: TDeployTicket = deployTickets
            .first { it.spec.releaseId == release.meta.id }

        return DeployInfo(
            stage = stage,
            deployUnitStatus = deployUnitStatus,
            version = deployUnitVersion,
            release = release,
            deployTicket = deployTicket,
        )
    }

    private fun getStage(stageName: String): CompletableFuture<TStage?> {
        return ypClient.objectService().getObject(
            YpGetStatement.protobufBuilder(YpTypedId(stageName, YpObjectType.STAGE))
                .addSelector("")
                .build()
        ) { payload: List<YpPayload> -> TStage.parseFrom(payload.single().protobuf.get()) }
    }

    private fun getDeployUnitStatus(stage: TStage): TDeployUnitStatus? {
        val deployUnits: Map<String, TDeployUnitStatus> = stage.status.deployUnitsMap
        return if (deployUnits.size == 1) {
            deployUnits.values.single()
        } else {
            logger.error("Expected single deploy unit status, but got $deployUnits")
            null
        }
    }

    private fun getRecentDeployTickets(stage: TStage): CompletableFuture<List<TDeployTicket>> {
        val minCreationTime: Instant = Instant.now() - Duration.ofDays(7)
        val minCreationMicroseconds = TimeUnit.SECONDS.toMicros(minCreationTime.epochSecond)

        return ypClient.objectService().selectObjects(
            YpSelectStatement.protobufBuilder(YpObjectType.DEPLOY_TICKET)
                .addSelector("")
                .setFilter("[/meta/stage_id]='${stage.meta.id}' AND [/meta/creation_time] > $minCreationMicroseconds")
                .build()
        ) { payload: List<YpPayload> -> TDeployTicket.parseFrom(payload.single().protobuf.get()) }
            .thenApply { it.results }
    }

    private fun getDeployTicketsForRelease(stage: TStage, release: TRelease): CompletableFuture<List<TDeployTicket>> {
        return ypClient.objectService().selectObjects(
            YpSelectStatement.protobufBuilder(YpObjectType.DEPLOY_TICKET)
                .addSelector("")
                .setFilter("[/meta/stage_id]='${stage.meta.id}' AND [/spec/release_id] = '${release.meta.id}'")
                .build()
        ) { payload: List<YpPayload> -> TDeployTicket.parseFrom(payload.single().protobuf.get()) }
            .thenApply { it.results }
    }

    private fun getReleasesById(ids: Collection<String>): CompletableFuture<List<TRelease>> {
        return ypClient.objectService().getObjects(
            YpGetManyStatement.protobufBuilder(YpObjectType.RELEASE)
                .addSelector("")
                .addIds(ids.toList())
                .build()
        ) { payload: List<YpPayload> -> TRelease.parseFrom(payload.single().protobuf.get()) }
    }

    /**
     * Выбирает из спеки deploy_unit слой с типом ресурса [resourceType],
     * для которого возвращает значение атрибута [RESOURCE_VERSION_ATTRIBUTE]
     */
    private fun findResourceVersion(deployUnitState: TDeployUnitStatus, resourceType: String): String? {
        val podTemplateSpec: TPodTemplateSpec = deployUnitState.podTemplateSpec()
        val layers: List<TLayer> = podTemplateSpec.spec.podAgentPayload.spec.resources.layersList

        val resource = layers
            .mapNotNull { it.meta?.sandboxResource }
            .find { it.resourceType == resourceType }

        return resource?.let { it.attributesMap[RESOURCE_VERSION_ATTRIBUTE] }
    }

    /**
     * Выбирает из спеки release ресурс с типом [resourceType],
     * для которого возвращает значение атрибута [RESOURCE_VERSION_ATTRIBUTE]
     */
    private fun findResourceVersion(release: TRelease, resourceType: String): String? {
        val resource = release.spec.sandbox.resourcesList
            .find { it.type == resourceType }

        return resource?.let { it.attributesMap[RESOURCE_VERSION_ATTRIBUTE] }
    }

    private fun TDeployUnitStatus.podTemplateSpec(): TPodTemplateSpec = if (currentTarget.hasMultiClusterReplicaSet()) {
        currentTarget.multiClusterReplicaSet.replicaSet.podTemplateSpec
    } else {
        currentTarget.replicaSet.replicaSetTemplate.podTemplateSpec
    }

    // Ожидание CompletableFuture с таймаутом по умолчанию
    private fun <T> CompletableFuture<T>.await() = this.get(1, TimeUnit.MINUTES)
}
