package ru.yandex.direct.chassis.entity.deploy

import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import ru.yandex.direct.chassis.util.DeployException
import ru.yandex.direct.chassis.util.DeployService
import ru.yandex.direct.chassis.util.DeployStage
import ru.yandex.direct.env.Environment
import ru.yandex.direct.env.EnvironmentType
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.utils.SystemUtils
import ru.yandex.startrek.client.error.StartrekClientException

@Hourglass(periodInSeconds = 5 * 60, needSchedule = ProductionOnly::class)
class YaDeployAppReleaseTicketCloser @Autowired constructor(
    private val deployService: DeployService,
    private val directReleasesService: DirectReleasesService,
) : DirectJob() {
    private val logger = LoggerFactory.getLogger(YaDeployAppReleaseTicketCloser::class.java)

    override fun execute() {
        val readyReleasesByApp = directReleasesService.getReadyReleasesByApp()

        for ((app, releases) in readyReleasesByApp) {
            logger.info("releases for ${app.name}: " + releases.map { "${it.issue.key}/${it.version}" }
                .joinToString(", "))
            try {
                val stage = DeployStage.PRODUCTION
                val stageNameList = app.yaDeployStages[stage.label]
                val deployStatusSummaryList = stageNameList?.map { deployService.getDeployStatusSummary(app, it, stage.label) }
                val isReleaseDeployed = deployStatusSummaryList?.all { it.isDeployTicketClosed }
                val isDeployWaitingForApproveLocation = deployStatusSummaryList?.any { it.isDeployWaitingForApproveLocation }
                val version = deployStatusSummaryList?.first()?.version

                logger.info("version for ${app.name} in Ya.Deploy: $version")
                if (isReleaseDeployed == true) {
                    for (release in releases.filter { it.version == version }) {
                        try {
                            val comment =
                                "Пакет версии $version установлен на группе продакшеновых машин\n\n\n------\n" +
                                    "chassis-задача ${YaDeployAppReleaseTicketCloser::class.simpleName} с хоста ${SystemUtils.hostname()}"
                            logger.info("closing release ticket ${release.issue.key}")
                            if (Environment.get() == EnvironmentType.PRODUCTION) {
                                release.closeIssue(comment)
                            } else {
                                logger.info("won't close ticket in non-production environment")
                            }
                        } catch (e: StartrekClientException) {
                            logger.error("Startrek error while processing release ticket ${release.issue.key}: $e")
                        }
                    }
                } else {
                    if (isDeployWaitingForApproveLocation == true) {
                        logger.info("Deploy is waiting for approve remaining locations for app ${app.name}")
                    }
                }
            } catch (e: DeployException) {
                logger.warn("skipping app ${app.name} with deploy error: ${e.message}")
            }
        }
    }
}
