package ru.yandex.direct.chassis.entity.deploy

import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.util.DirectAppsConfEntry
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.entity.sandbox.ReleaseRequest
import ru.yandex.direct.chassis.entity.sandbox.ReleaseSubject
import ru.yandex.direct.chassis.entity.sandbox.SandboxClient
import ru.yandex.direct.chassis.util.DeployService
import ru.yandex.direct.chassis.util.DeployStage
import ru.yandex.direct.chassis.util.DirectReleaseUtils.getReleaseVersionString
import ru.yandex.direct.chassis.util.startrek.StartrekField
import ru.yandex.direct.chassis.util.startrek.StartrekHelper.Companion.field
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.direct.dbschema.chassis.enums.ReleasesDeployQueueStatus
import ru.yandex.direct.env.Environment
import ru.yandex.direct.env.EnvironmentType
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.direct.tracing.Trace
import ru.yandex.direct.utils.InterruptedRuntimeException
import ru.yandex.startrek.client.Session
import ru.yandex.yp.YpRawClient
import ru.yandex.yp.client.api.EDeployPatchActionType
import ru.yandex.yp.client.api.EDeployTicketPatchSelectorType
import ru.yandex.yp.client.api.TDeployTicketControl
import ru.yandex.yp.model.YpObjectType
import ru.yandex.yp.model.YpObjectUpdate
import ru.yandex.yp.model.YpPayload
import ru.yandex.yp.model.YpSetUpdate
import ru.yandex.yp.model.YpTypedId

/**
 * Берет готовые релизы из таблицы releases_deploy_queue и выкладывает их в deploy
 * Делает следующие вещи:
 * 1) получает sandbox таску из релиза
 * 2) релизит ее ресурс
 * 3) коммитит соответствующий deploy-ticket
 *
 * Все шаги выполняются без ожидания, при каждом запуске проверятся, какие шаги уже сделаны, и делается следующий
 */
@Hourglass(periodInSeconds = 30, needSchedule = ProductionOnly::class)
class YaDeployAppReleaser(
    private val startrekSession: Session,
    private val sandboxClient: SandboxClient,
    private val ypClientCrossDC: YpRawClient,
    private val deployService: DeployService,
    private val releasesDeployQueueRepository: ReleasesDeployQueueRepository,
    private val directAppsConfProvider: DirectAppsConfProvider,
) : DirectJob() {
    companion object {
        private val logger = LoggerFactory.getLogger(YaDeployAppReleaser::class.java)
    }

    private val deployStage = getDeployStage()
    private val releaseSubject = if (Environment.getCached() == EnvironmentType.PRODUCTION) ReleaseSubject.stable else
        ReleaseSubject.testing

    private fun getDeployStage() =
        when (Environment.getCached()) {
            EnvironmentType.PRODUCTION -> DeployStage.PRODUCTION.label
            EnvironmentType.TESTING -> DeployStage.TESTING.label
            else -> DeployStage.DEVTEST.label
        }


    private val appsByTrackerComponent: Map<String, DirectAppsConfEntry>
        get() = directAppsConfProvider.getDirectAppsConf()
            .associateBy { it.trackerComponent }

    override fun execute() {
        val releasesForDeploy = releasesDeployQueueRepository.getReleasesForDeploy()
        logger.info("Got releases ${releasesForDeploy.keys}")
        val currentTrace = Trace.current()
        for (releaseForDeploy in releasesForDeploy) {
            val thread = Thread {
                Trace.push(currentTrace)
                handleTask(releaseForDeploy.key, releaseForDeploy.value)
                Trace.pop()
            }
            thread.name = "${releaseForDeploy.key}-deploy-release"
            thread.isDaemon = true
            thread.start()
            try {
                thread.join()// IS-NOT-COMPLETABLE-FUTURE-JOIN
            } catch (e: InterruptedException) {
                logger.error("Got interrupted exception", e)
                Thread.currentThread().interrupt()
                throw InterruptedRuntimeException(e)
            }
        }
    }

    private fun handleTask(releaseTicket: String, state: State) {
        try {
            val status =
                try {
                    releaseAndDeploy(releaseTicket, state)
                } catch (e: Exception) {
                    logger.error("Handling ticket $releaseTicket failed with exception", e)
                    state.errors = e.message
                    ReleasesDeployQueueStatus.InProgress
                }

            releasesDeployQueueRepository.saveState(releaseTicket, status, state)
        } catch (e: Throwable) {
            logger.error("Fail to handleTask $releaseTicket with throwable", e)
        }
    }

    private fun releaseAndDeploy(releaseTicket: String, state: State): ReleasesDeployQueueStatus {
        state.errors = null
        logger.info("Start handle ticket $releaseTicket")
        state.started = true
        val issue = startrekSession.issues().get(releaseTicket)
        if (issue.status.key == StartrekStatusKey.CLOSED) {
            logger.info("Ticket closed")
            return ReleasesDeployQueueStatus.Deployed
        }
        if (issue.status.key != StartrekStatusKey.READY_TO_DEPLOY) {
            val errorMessage = "Ticket not in readyToDeployStatus, actual status ${issue.status.key}"
            logger.error(errorMessage)
            state.errors = errorMessage
            return ReleasesDeployQueueStatus.InProgress
        }

        val directAppComponents =
            issue.components.orEmpty().mapNotNull { it.display }.filter { it in appsByTrackerComponent }
        val appComponent = directAppComponents.getOrNull(0)
        if (appComponent == null) {
            val errorMessage = "Missing app component in release ticket $releaseTicket"
            logger.error(errorMessage)
            state.errors = errorMessage
            return ReleasesDeployQueueStatus.InProgress
        }

        val app = appsByTrackerComponent[appComponent]

        if (app == null) {
            val errorMessage = "No app for component $appComponent, release ticket $releaseTicket"
            logger.error(errorMessage)
            state.errors = errorMessage
            return ReleasesDeployQueueStatus.InProgress
        }

        val releaseVersion = getReleaseVersionString(issue.summary)
        if (releaseVersion == null) {
            val errorMessage = "Can't get release version for ticket ${app.name}"
            logger.error(errorMessage)
            state.errors = errorMessage
            return ReleasesDeployQueueStatus.InProgress
        }

        val stageNameList = try {
            app.yaDeployStages.getValue(deployStage)
        } catch (e: NoSuchElementException) {
            null
        }
        if (stageNameList == null || stageNameList.isEmpty()) {
            val errorMessage = "No stage name for app ${app.name}"
            logger.error(errorMessage)
            state.errors = errorMessage
            return ReleasesDeployQueueStatus.InProgress
        }
        val sandboxTaskId = issue.field<String>(StartrekField.SANDBOX_BUILD_TASK)?.toLong()
        if (sandboxTaskId == null) {
            val errorMessage = "No sandbox task in ticket $releaseTicket"
            logger.error(errorMessage)
            state.errors = errorMessage
            return ReleasesDeployQueueStatus.InProgress
        }
        val release = sandboxClient.getRelease(sandboxTaskId, releaseSubject)
        if (release == null) {
            logger.info("Releasing sandbox task $sandboxTaskId to $releaseSubject")
            sandboxClient.releaseTask(ReleaseRequest(releaseSubject, sandboxTaskId, releaseSubject))
            return ReleasesDeployQueueStatus.InProgress
        }
        state.released = true

        val statusList = stageNameList.filterNotNull().map { stageName -> commitDeployTicket(app, releaseTicket, stageName, releaseVersion, state) }

        val status = if (statusList.any { it == ReleasesDeployQueueStatus.Failed }) {
            ReleasesDeployQueueStatus.Failed
        } else if (statusList.all { it == ReleasesDeployQueueStatus.Deployed }) {
            ReleasesDeployQueueStatus.Deployed
        } else {
            ReleasesDeployQueueStatus.InProgress
        }

        return status
    }


    private fun commitDeployTicket(
        app: DirectAppsConfEntry,
        releaseTicket: String,
        stageName: String,
        version: String,
        state: State,
    ): ReleasesDeployQueueStatus {
        val deployTicket = deployService.getDeployTicketForVersion(stageName) { release ->
            release.spec.docker.imagesList
                .mapNotNull { resource -> resource.tag }
                .contains(version)
        }
        if (deployTicket == null) {
            val errorMessage = "No deploy ticket for app ${app.name} version $version"
            logger.error(errorMessage)
            state.errors = errorMessage
            return ReleasesDeployQueueStatus.InProgress
        }
        if (deployTicket.status.action.type == EDeployPatchActionType.DPAT_NONE) {
            logger.info("Committing deploy ticket ${deployTicket.meta.name}")
            val commit = TDeployTicketControl.TCommitAction.newBuilder().apply {
                options = TDeployTicketControl.TActionOptions.newBuilder().apply {
                    reason = "Commit from kotlin"
                    message = "release ${app.name}: $releaseTicket, version $version"
                    patchSelector = TDeployTicketControl.TPatchSelector.newBuilder()
                        .apply {
                            type = EDeployTicketPatchSelectorType.DTPST_FULL
                        }
                        .build()
                }.build()

            }.build()
            val update = YpObjectUpdate.builder(YpTypedId(deployTicket.meta.id, YpObjectType.DEPLOY_TICKET))
                .addSetUpdate(YpSetUpdate("/control/commit", null) { _ ->
                    YpPayload.protobuf(
                        commit
                    )
                }).build()
            ypClientCrossDC.objectService().updateObject(update).get()
            state.commited = true
            return ReleasesDeployQueueStatus.InProgress
        }
        if (deployTicket.status.action.type != EDeployPatchActionType.DPAT_COMMIT) {
            val errorMessage = "Unexpected action type ${deployTicket.status.action.type}"
            logger.error(errorMessage)
            state.errors = errorMessage
            return ReleasesDeployQueueStatus.Failed
        }
        if (deployTicket.status?.progress?.closed?.status == ru.yandex.yp.client.api.EConditionStatus.CS_TRUE) {
            logger.info("Release completed")
            state.deployed = true
            return ReleasesDeployQueueStatus.Deployed
        }
        if (deployTicket.status.action.type == EDeployPatchActionType.DPAT_COMMIT) {
            val deployUnitStatus = deployService.getDeployUnitStatus(stageName)
            val podsReady = deployUnitStatus.progress?.podsReady
            val podsTotal = if (deployUnitStatus.currentTarget?.hasMultiClusterReplicaSet() == true)
                deployUnitStatus.currentTarget?.multiClusterReplicaSet?.replicaSet?.clustersList?.sumOf { it.spec.replicaCount }
            else {
                deployUnitStatus.currentTarget?.replicaSet?.perClusterSettingsMap?.map { it.value.podCount }?.sum()
            }
            state.progress = ((podsReady!! * 1.0 / podsTotal!!) * 100).toInt()
        }

        return ReleasesDeployQueueStatus.InProgress
    }
}
