package ru.yandex.direct.chassis.entity.deploy.repository

import org.jooq.Record6
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.deploy.model.DeployNotification
import ru.yandex.direct.dbschema.chassis.tables.DeployNotifications.DEPLOY_NOTIFICATIONS
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapperhelper.InsertHelper

@Component
class DeployNotificationRepository(
    private val dslContextProvider: DslContextProvider,
) {
    private fun booleanToTinyint(finished: Boolean): Long = if (finished) 1 else 0
    private fun tinyintToBoolean(tinyint: Long): Boolean = tinyint != 0L

    private fun notificationMapper(it: Record6<Long, String, String, String, Long, Long>) =
        DeployNotification(
            id = it[DEPLOY_NOTIFICATIONS.ID],
            app = it[DEPLOY_NOTIFICATIONS.APP],
            releaseId = it[DEPLOY_NOTIFICATIONS.RELEASE_ID],
            issueKey = it[DEPLOY_NOTIFICATIONS.ISSUE_KEY],
            messageId = it[DEPLOY_NOTIFICATIONS.MESSAGE_ID].toInt(),
            finished = tinyintToBoolean(it[DEPLOY_NOTIFICATIONS.FINISHED]),
        )

    fun getNotification(app: String, issueKey: String): DeployNotification? {
        return dslContextProvider.chassis()
            .select(
                DEPLOY_NOTIFICATIONS.ID,
                DEPLOY_NOTIFICATIONS.APP,
                DEPLOY_NOTIFICATIONS.RELEASE_ID,
                DEPLOY_NOTIFICATIONS.ISSUE_KEY,
                DEPLOY_NOTIFICATIONS.MESSAGE_ID,
                DEPLOY_NOTIFICATIONS.FINISHED
            )
            .from(DEPLOY_NOTIFICATIONS)
            .where(DEPLOY_NOTIFICATIONS.APP.eq(app))
            .and(DEPLOY_NOTIFICATIONS.ISSUE_KEY.eq(issueKey))
            .fetchOne { notificationMapper(it) }
    }

    fun getUnfinishedNotifications(): List<DeployNotification> {
        return dslContextProvider.chassis()
            .select(
                DEPLOY_NOTIFICATIONS.ID,
                DEPLOY_NOTIFICATIONS.APP,
                DEPLOY_NOTIFICATIONS.RELEASE_ID,
                DEPLOY_NOTIFICATIONS.ISSUE_KEY,
                DEPLOY_NOTIFICATIONS.MESSAGE_ID,
                DEPLOY_NOTIFICATIONS.FINISHED
            )
            .from(DEPLOY_NOTIFICATIONS)
            .where(DEPLOY_NOTIFICATIONS.FINISHED.eq(0))
            .fetch { notificationMapper(it) }
    }

    fun insertNotification(notification: DeployNotification): DeployNotification {
        val notificationId = InsertHelper(dslContextProvider.chassis(), DEPLOY_NOTIFICATIONS)
            .set(DEPLOY_NOTIFICATIONS.APP, notification.app)
            .set(DEPLOY_NOTIFICATIONS.RELEASE_ID, notification.releaseId)
            .set(DEPLOY_NOTIFICATIONS.ISSUE_KEY, notification.issueKey)
            .set(DEPLOY_NOTIFICATIONS.MESSAGE_ID, notification.messageId.toLong())
            .set(DEPLOY_NOTIFICATIONS.FINISHED, booleanToTinyint(notification.finished))
            .executeIfRecordsAddedAndReturn(DEPLOY_NOTIFICATIONS.ID)
            .single()
        return notification.copy(id = notificationId)
    }

    fun updateNotification(notification: DeployNotification) {
        dslContextProvider.chassis()
            .update(DEPLOY_NOTIFICATIONS)
            .set(DEPLOY_NOTIFICATIONS.FINISHED, booleanToTinyint(notification.finished))
            .execute()
    }
}
