package ru.yandex.direct.chassis.entity.regression.restart

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.startrek.client.model.Issue
import kotlin.math.max

/**
 * Действие, которое может мутировать состояние. Если состояние было изменено, возвращает новое состояние. Иначе `null`
 */
typealias StateAction = (ReleaseRegressionInfo) -> ReleaseRegressionInfo?

internal const val AUTORESTART_RESET = "autorestart_reset"
internal const val AUTORESTART_ONE_MORE = "autorestart_one_more"

/**
 * Теги, которые надо сбросить при обновлении состояния
 */
internal val ACTION_TAGS = listOf(
    REGRESSION_DONE_TAG,
    RESTARTER_STOPPED_TAG,
    AUTORESTART_RESET,
    AUTORESTART_ONE_MORE,
)

@Component
class AutotestsActions(
    val startrek: StartrekHelper,
) {

    private val logger = LoggerFactory.getLogger(AutotestsActions::class.java)

    /**
     * Изменяем состояние при изменении тега запуска, либо при наличии специальных тегов на тикете
     */
    fun processTags(
        regressionTickets: List<Issue>,
        tag: String,
        state: ReleaseRegressionInfo
    ): ReleaseRegressionInfo {

        fun comment(content: String) {
            regressionTickets.forEach { issue ->
                // TODO: Применять действие только к пакам нужного профиля?
                startrek.addComment(issue, content, notify = false)
            }
        }

        val actions = listOf(
            booleanAction(tag != state.tag) {
                logger.info("Changed launch tag from ${state.tag} на $tag, clearing pack info")
                // TODO: Менять тег на всех тикетах?
                comment(
                    """
                        Тег аквы был изменен с %%${state.tag}%% на %%$tag%%.
                        Новая ссылка: $AQUA_LAUNCH_PREFIX$tag
                    """.trimIndent()
                )
                state.copy(tag = tag, packs = listOf())
            },
            tagAction(AUTORESTART_RESET, regressionTickets) {
                logger.info("Clearing restart state")
                comment("Состояние рестарта тестов очищено. Тесты будут рестартится с чистого листа.")
                state.copy(packs = listOf())
            },
            tagAction(AUTORESTART_ONE_MORE, regressionTickets) {
                val newState = state.copy(packs = state.packs.map { pack ->
                    if (pack.stopReason in listOf(StopReason.COUNT_LIMIT_REACHED, StopReason.NO_DIFF)) {
                        pack.copy(
                            restartCount = max(pack.restartCount - 1, 0),
                            stopReason = null,
                            // для причины "не уменьшилось множество упавших"
                            lastFinishedLaunch = null,
                        )
                    } else {
                        pack
                    }
                })

                if (newState != state) {
                    logger.info("Restarting packs one more time")
                    comment("Упавшие паки будут перезапущены еще один раз.")
                } else {
                    logger.info("Removing $AUTORESTART_ONE_MORE because there is nothing to restart")
                    comment(
                        "Не нашлось ни одного остановленного пака, который можно порестартить. " +
                                "Тег ##$AUTORESTART_ONE_MORE## удален без изменений в состоянии."
                    )
                }

                newState
            },
        )

        return applyActions(state, regressionTickets, actions)
    }

    private fun applyActions(
        state: ReleaseRegressionInfo,
        issues: List<Issue>,
        actions: List<StateAction>
    ): ReleaseRegressionInfo {
        for (action in actions) {
            val newState = action(state)
            if (newState != null) {
                resetTags(issues)
                return newState
            }
        }
        return state
    }

    private fun booleanAction(condition: Boolean, block: StateAction): StateAction {
        return { state ->
            if (condition) {
                block(state)
            } else {
                null
            }
        }
    }

    private fun tagAction(tag: String, issues: List<Issue>, block: StateAction): StateAction {
        return { state ->
            if (regressionHasTag(tag, issues)) {
                block(state)
            } else {
                null
            }
        }
    }

    private fun resetTags(issues: List<Issue>) {
        issues.forEach { issue ->
            startrek.removeTags(issue, ACTION_TAGS)
        }
    }

    private fun regressionHasTag(tag: String, issues: List<Issue>): Boolean {
        return issues.any { issue ->
            startrek.hasTag(issue, tag)
        }
    }
}
