package ru.yandex.direct.chassis.entity.regression.restart

import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.entity.startrek.affectedApps
import ru.yandex.direct.chassis.entity.startrek.howToTest
import ru.yandex.direct.chassis.entity.startrek.testedApps
import ru.yandex.direct.chassis.properties.PropertiesLightSupport
import ru.yandex.direct.chassis.repository.RegressionRestartRepository
import ru.yandex.direct.chassis.util.DirectAppsConfEntry
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.util.DirectReleaseUtils
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.aqua.AquaClient
import ru.yandex.direct.chassis.util.aqua.Launch
import ru.yandex.direct.chassis.util.aqua.LaunchStatus
import ru.yandex.direct.chassis.util.nonCf
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.model.Issue

/**
 * TODO
 * - Профили для перла
 * - Отправка в графит ?
 * - Ограничение на количество одновременно запущенных тестов (не актуально для api5?)
 * - Останавливать "зависшие" запуски по таймауту
 * - Если мы закончили рестартить тесты, но потом они прошли (из-за ручных рестартов) &mdash; писать об этом в тикет
 * - Отчет после 8 часов?
 * - "Тесты почти прошли"?
 */

private const val AUTOTESTS_RESTART_JOB_ENABLED = "autotests_restart_job_enabled"
internal const val AQUA_LAUNCH_PREFIX = "https://aqua.yandex-team.ru/#/launches-tag?tag="

internal const val REGRESSION_DONE_TAG = "regression_done"
internal const val RESTARTER_STOPPED_TAG = "restarter_stopped"

private val TICKET_KEY_REGEX = """(DIRECT|TESTIRT)-\d+""".toRegex()
private val AQUA_LAUNCH_REGEX = """https://aqua\.yandex-team\.ru/#/launches-tag\?tag=(.*)""".toRegex()

private const val REPORT_FAILED_TESTS_LIMIT = 20

@Hourglass(periodInSeconds = 3 * 60, needSchedule = ProductionOnly::class)
class AutotestsRestartJob(
    private val autotestsActions: AutotestsActions,
    private val aquaClient: AquaClient,
    private val startrek: StartrekHelper,
    private val regressionRestartRepository: RegressionRestartRepository,
    private val propertiesLightSupport: PropertiesLightSupport,
    private val directAppsConfProvider: DirectAppsConfProvider,
    regressionConfigurations: List<RegressionConfig>,
) : DirectJob() {

    private val logger = LoggerFactory.getLogger(AutotestsRestartJob::class.java)

    private val appByName: Map<String, DirectAppsConfEntry>
        get() = directAppsConfProvider.getDirectAppsConf()
            .associateBy { it.name }

    private val configurationByApp = regressionConfigurations.associateBy { it.app }

    private val isJobEnabled: Boolean
        get() = propertiesLightSupport[AUTOTESTS_RESTART_JOB_ENABLED] in listOf("true", "1")

    override fun execute() {
        if (!isJobEnabled) {
            logger.info("Autotests restart job is disabled by property")
            return
        }

        // Любые незакрытые релизные тикеты
        val query = """
            Queue: DIRECT
            Release
            (Components: "Releases: Direct" OR Components: "Releases: JavaDirect")
            Status: !Closed
            "Sort by": key desc
        """.trimIndent()

        val releaseTickets: List<Issue> = startrek.session.issues().find(query).toList().nonCf()

        val releasesByApp = releaseTickets
            .groupBy { issue -> DirectReleaseUtils.getReleaseApp(issue, appByName.values) }

        releasesByApp.forEach { (app, issues) ->
            if (app != null) {
                val config = configurationByApp[app.name]
                if (config != null) {
                    issues.forEach { release ->
                        val regressionTickets = findRegressionTickets(config, release)
                        processRegression(app, config, release, regressionTickets)
                    }
                }
            }
        }
    }

    private fun processRegression(
        app: DirectAppsConfEntry,
        config: RegressionConfig,
        release: Issue,
        regressionTickets: List<Issue>,
    ) {
        logger.info("Processing release {} with app {}", release.key, app.name)
        logger.info("Regression tickets are: {}", regressionTickets.map { it.key })

        val tag = findLaunchTag(regressionTickets)
        if (tag == null) {
            logger.warn("Could not find launch tag")
            return
        }
        logger.info("Found launch tag {}", tag)

        var state = regressionRestartRepository.getRestartState(config.app, release.key)
            ?: ReleaseRegressionInfo(config.app, release.key, tag, listOf())

        val launches = aquaClient.getLaunches(tag).launches
        if (launches.isEmpty()) {
            logger.warn("No launches with tag {}", tag)
            return
        }

        if (config.actionsEnabled) {
            state = autotestsActions.processTags(regressionTickets, tag, state)
        }

        logger.info("Restart iteration for tag {}", tag)
        state = restartLaunch(config, state, tag, launches)

        logger.info("Saving new state")
        regressionRestartRepository.saveRestartState(state)

        logger.info("Comment iteration for tag {}", tag)
        commentIssues(app, config, regressionTickets, launches, state)
    }

    private fun commentIssues(
        app: DirectAppsConfEntry,
        config: RegressionConfig,
        regressionTickets: List<Issue>,
        launches: List<Launch>,
        state: ReleaseRegressionInfo,
    ) {
        if (config.profiles != null) {
            config.profiles.forEach { profile ->
                logger.info("Comment iteration for profile ${profile.issueTag}")

                commentSingleProfile(
                    app = app,
                    config = config,
                    regressionTickets = regressionTickets
                        .filter { profile.issueTag in it.tags.nonCf() },
                    launches = launches
                        .filter { it.pack.name in profile.packNames },
                    packs = state.packs
                        .filter { it.name in profile.packNames },
                )
            }
        } else {
            commentSingleProfile(app, config, regressionTickets, launches, state.packs)
        }
    }

    private fun commentSingleProfile(
        app: DirectAppsConfEntry,
        config: RegressionConfig,
        regressionTickets: List<Issue>,
        launches: List<Launch>,
        packs: List<PackRestartState>,
    ) {
        regressionTickets.forEach { issue ->
            if (app.trackerAffectedApp != null && app.trackerAffectedApp !in issue.affectedApps) {
                startrek.appendAffectedApp(issue, app.trackerAffectedApp)
            }

            if (packs.isEmpty()) {
                logger.info("No finished launches")
            } else if (packs.all { it.stopReason == StopReason.PASSED }) {
                comment(issue, REGRESSION_DONE_TAG, summon = false) {
                    """!!(зел)**Тесты успешно прошли!**!!
                        |
                        |<{Отчет
                        |${buildReport(packs, launches)}
                        |}>
                    """.trimMargin()
                }

                if (app.trackerAffectedApp != null && app.trackerAffectedApp !in issue.testedApps) {
                    startrek.appendTestedApp(issue, app.trackerAffectedApp)
                }

                startrek.closeIssue(issue, force = true)
            } else if (packs.all { it.stopReason != null }) {
                var text =
                    """!!**Тесты больше не будут перезапускаться**!!
                        |
                        |${buildReport(packs, launches, failedOnly = true)}
                    """.trimMargin()


                if (config.actionsEnabled) {
                    text += """
                        |
                        |Для ручного перезапуска проставьте один из тегов:
                        |%%autorestart_one_more%% - еще одна итерация для упавших тестов, достигших лимита перезапусков
                        |%%autorestart_reset%% - сбросить счетчик перезапусков для всех тестов
                    """.trimMargin()
                }

                comment(issue, RESTARTER_STOPPED_TAG, summon = true, text)
            } else {
                logger.info("Some packs are not finished")
            }
        }
    }

    private fun buildReport(
        packs: List<PackRestartState>,
        launches: List<Launch>,
        failedOnly: Boolean = false,
    ): String {
        val stoppedPacks = packs
            .filter { it.stopReason != null }
            .sortedBy { it.lastFinishedLaunch?.name }

        val reportPacks = if (failedOnly) {
            stoppedPacks.filter { it.stopReason != StopReason.PASSED }
        } else {
            stoppedPacks
        }

        var message = "===== Результаты:"
        fun appendLine(line: String) {
            message += "\n" + line
        }

        reportPacks.forEach { pack ->
            val name = pack.lastFinishedLaunch?.name
            val reason = when (pack.stopReason) {
                StopReason.PASSED -> "Тесты успешно прошли."
                StopReason.NO_DIFF -> "Последний перезапуск не уменьшил множество упавших тестов."
                StopReason.COUNT_LIMIT_REACHED -> "Достигнут лимит перезапусков."
                else -> ""
            }

            val launch = launches
                .find { it.id == pack.lastFinishedLaunch?.launchId }
            val launchUrl = launch?.launchUrl
            val reportUrl = launch?.reportUrl

            appendLine("%%$name%% $reason (($launchUrl Запуск)), (($reportUrl отчет)).")

            val packLaunches = launches
                .filter { it.pack.id == pack.lastFinishedLaunch?.packId }
                .sortedBy { it.startTime }

            val failedSuites = pack.lastFinishedLaunch?.failedSuites.orEmpty()
            failedSuites.take(REPORT_FAILED_TESTS_LIMIT).forEach { suite ->
                val logUrls = packLaunches
                    .flatMap { it.pack.projects }
                    .flatMap { it.launchSuites }
                    .filter { it.suite.id == suite.id }
                    .map { it.logUrl }
                    .withIndex()
                    .joinToString(prefix = "Запуски: ", separator = ", ") { (index, url) ->
                        "(($url №${index + 1}))"
                    }

                appendLine("  ❌ %%${suite.name}%% $logUrls.")
            }

            if (failedSuites.size > REPORT_FAILED_TESTS_LIMIT) {
                appendLine("  Показаны только первые $REPORT_FAILED_TESTS_LIMIT. Всего падений: ${failedSuites.size}.")
            }
        }

        return message
    }

    private fun comment(issue: Issue, issueTag: String, summon: Boolean = false, text: String) {
        comment(issue, issueTag, summon) { text }
    }

    private fun comment(issue: Issue, issueTag: String, summon: Boolean = false, commentBuilder: () -> String) {
        if (!startrek.hasTag(issue, issueTag)) {
            val summonee = if (summon) issue.assignee.orNull else null
            startrek.addComment(issue, commentBuilder(), true, summonee = summonee)
            startrek.addTag(issue, issueTag)
        }
    }

    /**
     * Ищем тег в поле `howToTest` (Как тестировать), или в комментарии от робота.
     * Найденный тег проставляем в поле.
     * Предполагается, что можно руками менять тег в поле тикета
     */
    private fun findLaunchTag(tickets: List<Issue>): String? {
        val foundTags = tickets.map { issue ->
            // Тег из поля
            val fieldTag = """(?:aqua_tag:)(.*)""".toRegex()
                .matchEntire(issue.howToTest ?: "")
                ?.groupValues?.get(1)

            // Последний тег из комментариев от робота
            val commentTag = issue.comments.toList().nonCf()
                .filter { it.createdBy.login in listOf("direct-handles", "robot-direct-chassis") }
                .mapNotNull { it.text.orNull }
                .flatMap { text -> AQUA_LAUNCH_REGEX.findAll(text) }
                .map { match -> match.groupValues[1] }
                .lastOrNull()

            fieldTag ?: commentTag
        }

        // Выбираем тег, который указан в большинстве тикетов
        val selectedTag = foundTags
            .groupingBy { it }
            .eachCount()
            .maxByOrNull { (_, count) -> count }
            ?.key

        // Проставляем тег в поле тикетов
        if (selectedTag != null) {
            val value = "aqua_tag:$selectedTag"
            tickets
                .filter { it.howToTest != value }
                .forEach { it.howToTest = value }
        }

        return selectedTag
    }

    private fun findRegressionTickets(config: RegressionConfig, issue: Issue): List<Issue> {
        val issueKeys = issue.comments.toList().nonCf()
            .mapNotNull { it.text.orNull }
            .filter { text ->
                text.contains("Созданы тикеты на регрессию:")
            }
            .flatMap { text ->
                TICKET_KEY_REGEX.findAll(text)
                    .map { match -> match.value }
            }
            .distinct()

        val issues = issueKeys
            .map { key -> startrek.session.issues().get(key) }

        return issues
            .filter { it.status.key != StartrekStatusKey.CLOSED }
            .filter { config.filter == null || !config.filter.matches(it.summary) }
    }

    /**
     * Для каждого пака, пробуем рестартить последний запуск
     */
    private fun restartLaunch(
        config: RegressionConfig,
        state: ReleaseRegressionInfo,
        tag: String,
        launches: List<Launch>,
    ): ReleaseRegressionInfo {
        val currentPackStates = createState(launches)

        val previousPacksByPackId = state.packs.associateBy { it.packId }

        val restartedPacks = currentPackStates
            .associateWith {
                previousPacksByPackId[it.packId]
                    ?: PackRestartState(packId = it.packId, name = it.name)
            }
            .map { (current, previous) -> restartSinglePack(config, tag, previous, current) }

        return state.copy(packs = restartedPacks)
    }

    /**
     * Запуск рестартится, если он
     * - завершен (`REPORT_READY`)
     * - лимит попыток рестарта не исчерпан
     * - есть упавшие тесты
     * - есть дифф в упавших тестах относительно прошлого запуска
     */
    private fun restartSinglePack(
        config: RegressionConfig,
        tag: String,
        previous: PackRestartState,
        current: PackLaunch,
    ): PackRestartState {
        if (previous.stopReason != null ||
            current.status !in listOf(LaunchStatus.REPORT_READY, LaunchStatus.REPORT_FAILED)
        ) {
            return previous
        }

        val testsPassed = current.failedSuites.isEmpty()
        val noDiff = config.noDiffEnabled && current.failedSuites == previous.lastFinishedLaunch?.failedSuites
        val restartLimitReached = previous.restartCount >= config.maxRestartCount

        // Порядок определяет приоритет
        val stopReason = when {
            testsPassed -> StopReason.PASSED
            noDiff -> StopReason.NO_DIFF
            restartLimitReached -> StopReason.COUNT_LIMIT_REACHED
            else -> null
        }

        return if (stopReason == null) {
            logger.info("Restarting pack {}", current.name)

            val restartedLaunch = aquaClient.restartLaunch(current.launchId)
            aquaClient.addLaunchTag(restartedLaunch.id, tag)

            previous.copy(lastFinishedLaunch = current, restartCount = previous.restartCount + 1)
        } else {
            logger.info("Will no longer restart pack '{}' with reason {}", current.name, stopReason)

            previous.copy(lastFinishedLaunch = current, stopReason = stopReason)
        }
    }

    /**
     * Для каждого пака выбираем последний запуск
     */
    private fun createState(launches: List<Launch>): List<PackLaunch> {
        return launches
            // Выбираем максимальные по startTime запуски паков
            .sortedByDescending { it.startTime }
            .distinctBy { it.pack.id }
            .map { launch ->
                PackLaunch(
                    packId = launch.pack.id,
                    name = launch.pack.name,
                    launchId = launch.id,
                    startTime = launch.startTime,
                    status = launch.launchStatus,
                    failedSuites = launch.pack.projects
                        .flatMap { it.launchSuites }
                        .filter { it.launchStatus in listOf(LaunchStatus.FAILED, LaunchStatus.REVOKED) }
                        .map { SuiteInfo(it.suite.id, it.suite.name) }
                        .toSet()
                )
            }
    }
}
