package ru.yandex.direct.chassis.entity.regression.start

import org.joda.time.DateTime
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.abc.ScheduleSlug
import ru.yandex.direct.chassis.util.nonCf
import ru.yandex.startrek.client.model.Issue

/**
 * Как в описании тикета будет выглядеть подпись для истории регрессий
 */
private const val REGRESSION_HISTORY_TEMPLATE = "Тикеты на предыдущие запуски можно найти по ((%s ссылке))."

data class Release(
    val issue: Issue,
    val type: ReleaseType,
) {
    /**
     * Возвращает тег запусков аквы для релиза
     */
    fun releaseTag(): String {
        val date = DateTime.now().toLocalDate()
        return "${type.prefix}-${issue.key}-$date"
    }
}

data class ReleaseType(
    /**
     * Название приложения
     */
    val app: String,

    /**
     * Компоненты, которые должны быть у релизного тикета
     */
    val components: List<String>,

    /**
     * Регрессионные тикеты, которые надо создать для релиза
     */
    val issueTemplates: List<RegressionIssueTemplate> = listOf(),

    /**
     * Путь к Jenkins джобе, запускающей релизные тесты
     */
    val jenkinsJob: String? = null,

    /**
     * Дополнительные параметры Jenkins джобы
     */
    val jenkinsParams: Map<String, String> = mapOf(),

    /**
     * Префикс тега запусков релизных тестов в акве
     */
    val prefix: String = "direct-release-$app",
) {
    /**
     * Возвращает `true`, если релизный тикет принадлежит типу
     */
    fun test(issue: Issue): Boolean {
        return issue.components.nonCf()
            .map { it.display }
            .containsAll(components)
    }
}

data class RegressionIssueTemplate(
    val queue: String,
    val summary: String,
    val description: String? = null,
    val tags: List<String> = listOf(),
    val components: List<String> = listOf(),
    val needAquaComment: Boolean = true,
    val duty: ScheduleSlug? = null,
) {
    /**
     * Ожидается [summary] в формате "Запуск автотестов в релизе %s"
     */
    fun summary(releaseKey: String) = summary.format(releaseKey)

    /**
     * Добавляет в описание тикета с регрессиями ссылку на историю регрессий
     */
    fun descriptionWithLink(startrekUrl: String): String {
        val filterLink: String = createHistoryFilterQueryLink(startrekUrl)
        val regressionHistoryDescription = REGRESSION_HISTORY_TEMPLATE.format(filterLink)
        return if (description == null || description.isEmpty()) {
            regressionHistoryDescription
        } else {
            "$description\n---\n$regressionHistoryDescription"
        }
    }

    /**
     * Создает ссылку на историю регрессий
     */
    private fun createHistoryFilterQueryLink(startrekUrl: String): String {
        var historyFilterQuery = "Queue: $queue"
        val filterPart: String = if (components.isNotEmpty()) {
            " Components: \"${components.first()}\""
        } else if (tags.isNotEmpty()) {
            " Tags: \"${tags.first()}\""
        } else {
            " Summary: \"${summary("")}\""
        }
        historyFilterQuery += "$filterPart \"Sort by\": key desc"
        val filterLink: String = StartrekHelper.filterLink(historyFilterQuery, startrekUrl)
        return filterLink
    }
}
