package ru.yandex.direct.chassis.entity.regression.start

import org.slf4j.LoggerFactory
import ru.yandex.direct.chassis.entity.startrek.DIRECT_RELEASE_TESTS_TICKET_TAG
import ru.yandex.direct.chassis.properties.PropertiesLightSupport
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.startrek.StartrekHelper.Companion.components
import ru.yandex.direct.chassis.util.Utils
import ru.yandex.direct.chassis.util.abc.AbcClient
import ru.yandex.direct.chassis.util.abc.ServiceSlug
import ru.yandex.direct.chassis.util.aqua.AquaUtils
import ru.yandex.direct.chassis.util.jenkins.JenkinsClient
import ru.yandex.direct.chassis.util.jenkins.JenkinsException
import ru.yandex.direct.chassis.util.nonCf
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.direct.chassis.util.startrek.StartrekTransition
import ru.yandex.direct.env.ProductionOnly
import ru.yandex.direct.scheduler.Hourglass
import ru.yandex.direct.scheduler.support.DirectJob
import ru.yandex.startrek.client.model.Issue
import ru.yandex.startrek.client.model.IssueCreate
import ru.yandex.startrek.client.model.IssueRef
import ru.yandex.startrek.client.model.LocalLink
import ru.yandex.startrek.client.model.Relationship
import ru.yandex.startrek.client.model.UserRef
import java.time.Duration

/**
 * Пропертя для включения джобы
 */
private const val REGRESSION_START_JOB_ENABLED = "regression_start_job_enabled"

/**
 * Этот тег на релизном тикете значит, что тикет уже был обработан радаром
 */
private const val RADAR_TAG = "radar"

/**
 * Тег используется для запуска джобы в Jenkins
 */
private const val REGRESSION_STARTED_TAG = "regression_started"

/**
 * Джоба создает для новых релизов тикеты на автоматическую регрессию и запускает необходимые паки в Jenkins
 */
@Hourglass(periodInSeconds = 5 * 60, needSchedule = ProductionOnly::class)
class RegressionStartJob(
    private val startrek: StartrekHelper,
    private val abcClient: AbcClient,
    private val jenkinsClient: JenkinsClient,
    private val propertiesLightSupport: PropertiesLightSupport,
) : DirectJob() {

    private val logger = LoggerFactory.getLogger(RegressionStartJob::class.java)

    // При isDevelopment = true, будут подгружаться два захардкоженных тикета DIRECT-42926 и DIRECT-42913.
    // Если в конфиге параметр startrek.startrek_api_url настроен на тестовый стартрек, то тикеты будут получены.
    // Если в конфиге параметр startrek.startrek_api_url настроен на прод, то не вернется ничего, так как в продакшене
    // тикеты с этими номерами не релизные. Таким образом запуск с isDevelopment=true со стартреком,
    // настроенным на прод, не сделает вообще ничего.
    // вместо true для безопасности ставьте Environment.getCached() == EnvironmentType.DEVELOPMENT
    private val isDevelopment: Boolean = false

    private val isJobEnabled: Boolean
        get() = propertiesLightSupport[REGRESSION_START_JOB_ENABLED] in listOf("true", "1")

    override fun execute() {
        if (!isJobEnabled) {
            logger.info("Regression start job is disabled by property")
            return
        }

        val releases = findReleases()
        logger.info("Found: $releases")

        for (release in releases) {
            process(release)
        }
    }

    private fun findReleases(): List<Release> {
        val query = if (isDevelopment) {
            // Для локальной разработки нужно не забыть поменять адреса стартрека на тестовые в app-development.conf
            """
                key: DIRECT-42926,DIRECT-42913
                Type: Release
                "Parent Issue": empty()
                "Sort by": key DESC
            """.trimIndent()
        } else {
            """
                Queue: DIRECT
                Type: Release
                "Parent Issue": empty()
                (Status: "Ready For Test" OR Status: "Testing")
                Created: <= week()
                Tags: !$RADAR_TAG
                Tags: !$REGRESSION_STARTED_TAG
                "Sort by": key desc
            """.trimIndent()
        }

        val releaseTickets = startrek.session.issues().find(query).toList().nonCf()

        return releaseTickets.mapNotNull { issue -> toRelease(issue) }
    }

    private fun toRelease(issue: Issue): Release? {
        val releaseType = RELEASE_TYPES
            .firstOrNull { it.test(issue) }

        if (releaseType == null) {
            logger.info("Unknown release type for issue ${issue.key}, skipping it")
            return null
        }

        return Release(issue, releaseType)
    }

    private fun process(release: Release) {
        changeIssueStatus(release)

        val issues = createTickets(release)

        if (runJenkinsJob(release)) {
            // Комментарии оставляем только если получилось запустить джобу в Jenkins (либо запуск не нужен), чтобы
            // избежать дублирующихся запусков. Если запрос с комментарием упадет, то не все комментарии будут созданы
            addLaunchComments(release, issues)
        }
    }

    private fun changeIssueStatus(release: Release) {
        if (release.issue.status.key == StartrekStatusKey.READY_FOR_TEST) {
            if (!startrek.updateStatus(release.issue, StartrekStatusKey.TESTING)) {
                logger.error("Failed to change issue status")
            }
        }
    }

    data class RegressionIssue(
        val issue: IssueRef,
        val template: RegressionIssueTemplate,
    )

    private fun createTickets(release: Release): List<RegressionIssue> {
        val regressionIssues = mutableListOf<RegressionIssue>()

        val existingIssuesBySummary = release.issue.links.nonCf()
            .associateBy { it.`object`.display }

        for (template in release.type.issueTemplates) {
            val assigneeLogin: String? = if (template.duty == null) {
                release.issue.assignee.orElse(null as UserRef?)?.login
            } else {
                abcClient.getCurrentDutyLogins(ServiceSlug.DIRECT_APP_DUTY, template.duty).randomOrNull()?.login
            }

            val summary = template.summary(release.issue.key)

            val existingIssue: LocalLink? = existingIssuesBySummary[summary]
            if (existingIssue != null) {
                regressionIssues += RegressionIssue(
                    issue = existingIssue.`object`,
                    template = template,
                )

                logger.info("Issue with summary \"$summary\" already exists, skipping")
                continue
            }

            val issue = createRegressionTicket(summary, template, release, assigneeLogin)

            logger.info("Created issue ${issue.key} with summary ${issue.summary}")

            startrek.updateStatus(issue, StartrekTransition.START_PROGRESS)

            regressionIssues += RegressionIssue(
                issue = issue,
                template = template,
            )
        }

        return regressionIssues
    }

    private fun createRegressionTicket(
        summary: String,
        template: RegressionIssueTemplate,
        release: Release? = null,
        assigneeLogin: String? = null,
    ): Issue {
        val tags: List<String> = template.tags + listOf(DIRECT_RELEASE_TESTS_TICKET_TAG)

        var issueBuilder: IssueCreate.Builder = IssueCreate.builder()
            .queue(template.queue)
            .summary(summary)
            .description(template.descriptionWithLink(startrek.startrekUrl))
            .tags(*tags.toTypedArray())
            .components(template.components)

        if (release != null) {
            issueBuilder = issueBuilder.link(release.issue, Relationship.RELATES)
        }
        if (assigneeLogin != null) {
            issueBuilder.assignee(assigneeLogin)
        }

        val newIssueCreate: IssueCreate = issueBuilder.build()

        val issue = Utils.retryThrowing(3, pause = Duration.ofSeconds(3)) {
            startrek.session.issues().create(newIssueCreate)
        }
        return issue
    }

    /**
     * Запускает джобу в Jenkins. При успехе оставляет комменты со ссылкой на запуск и ставит
     * тег [REGRESSION_STARTED_TAG]
     * @return `true`, если тесты были запущены
     */
    private fun runJenkinsJob(release: Release): Boolean {
        if (startrek.hasTag(release.issue, REGRESSION_STARTED_TAG)) {
            return false
        }

        if (release.type.jenkinsJob != null) {
            try {
                jenkinsClient.buildJob(
                    release.type.jenkinsJob,
                    release.type.jenkinsParams + mapOf(
                        "tag" to release.releaseTag(),
                    )
                )
            } catch (e: JenkinsException) {
                logger.error("Failed to start jenkins job")
                return false
            }
        }

        startrek.addTag(release.issue, REGRESSION_STARTED_TAG)
        return true
    }

    /**
     * Добавляет комментарий со ссылкой на регрессионные тикеты в релизные тикет.
     * А так-же комментарии со ссылкой на акву в каждый регрессионный тикет
     */
    private fun addLaunchComments(release: Release, issues: List<RegressionIssue>) {
        val aquaUrl = AquaUtils.getLaunchesUrl(release.releaseTag())

        var comment = "Созданы тикеты на регрессию:\n"
        comment += issues.joinToString("\n") { it.issue.key }

        startrek.addComment(release.issue, comment, false)

        for ((issue, template) in issues) {
            if (!template.needAquaComment) {
                continue
            }

            startrek.addComment(issue, """
                Запуски в aqua:
                $aquaUrl
            """.trimIndent(), false)
        }
    }
}
