package ru.yandex.direct.chassis.entity.release.service

import com.fasterxml.jackson.annotation.JsonProperty
import org.springframework.beans.factory.annotation.Value
import org.springframework.stereotype.Component
import ru.yandex.direct.common.liveresource.zookeeper.ZookeeperLiveResourceProvider
import ru.yandex.direct.libs.curator.CuratorFrameworkProvider
import ru.yandex.direct.utils.JsonUtils
import java.net.URI

/**
 * Сервис про `direct-release`
 */
@Component
class DirectReleaseService(
    @Value("\${zookeeper.test_config_servers}") servers: String,
    @Value("\${zookeeper.lock_path}") lockPath: String,
) {
    private val zookeeperLiveResourceProvider = ZookeeperLiveResourceProvider(
        CuratorFrameworkProvider(servers, lockPath)
    )

    enum class Stage {
        STABLE,
        WAITING_1,
    }

    private class ReleaseState(
        @JsonProperty("ticket")
        val ticket: String,
    )

    /**
     * Возвращает тикет релиза приложения [app], который находится в стадии [stage] из zookeeper
     */
    fun getTicketByStage(app: String, stage: Stage): String? {
        val path = when (stage) {
            Stage.STABLE -> "/direct/release-state/stable/$app"
            Stage.WAITING_1 -> "/direct/release-state/waiting/$app/1"
        }

        val content = try {
            zookeeperLiveResourceProvider.get(URI("zk://$path")).content
        } catch (e: Exception) {
            return null
        }

        if (content.isEmpty()) {
            return null
        }

        val state = JsonUtils.fromJson(content, ReleaseState::class.java)
        return state.ticket
    }

}
