package ru.yandex.direct.chassis.entity.release.service

import ru.yandex.direct.chassis.entity.release.model.ReleaseError
import ru.yandex.direct.chassis.entity.startrek.DependencyGraph
import ru.yandex.direct.chassis.entity.startrek.ReleaseDependencyService
import ru.yandex.direct.chassis.entity.telegram.TelegramUtils
import ru.yandex.direct.chassis.entity.telegram.TelegramUtils.htmlLink
import ru.yandex.direct.chassis.entity.release.service.DirectReleaseService.Stage
import ru.yandex.direct.chassis.util.DirectAppsConfEntry
import ru.yandex.direct.chassis.util.DirectAppsConfIgnoredFeatures
import ru.yandex.direct.chassis.util.DirectReleaseUtils
import ru.yandex.direct.chassis.util.DistUtils
import ru.yandex.direct.chassis.util.TabulaUtils
import ru.yandex.direct.chassis.util.Utils.currentLogViewerLink
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.startrek.StartrekQueue
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.startrek.client.model.Issue

class ReleaseChecker(
    private val release: Issue,
    private val app: DirectAppsConfEntry,

    private val dependencyGraph: DependencyGraph,
    private val unclosedReleases: List<Issue>,

    private val startrek: StartrekHelper,
    private val distUtils: DistUtils,
    private val releaseDependencyService: ReleaseDependencyService,
    private val directReleaseService: DirectReleaseService,
) {
    companion object {
        private val DIRECTMIGR_REGEX = """${StartrekQueue.DIRECTMIGR}-\d+""".toRegex()
    }

    fun shouldNotify(): Boolean {
        return DirectAppsConfIgnoredFeatures.RELEASES_FOR_DEPLOY_NOTIFICATION !in app.ignoredFeatures
    }

    fun isNewCiApp(): Boolean {
        return app.newCiGraph != null
    }

    fun buildNotification(): String {
        var result = "(${app.name}) https://st.yandex-team.ru/${release.key}"

        val migrationsCount = migrationsCount()
        if (migrationsCount != 0) {
            result += " (миграции: $migrationsCount)"
        }

        val errors: List<ReleaseError> = collectErrors()
        if (errors.isNotEmpty()) {
            result += "\n"
            result += errors.joinToString("\n")
        }

        return result
    }

    private fun migrationsCount(): Int {
        val migrationKeys = DIRECTMIGR_REGEX
            .findAll(release.description.orElse(""))
            .map { it.value }.toList()
        return startrek.findIssuesByKeys(migrationKeys).size
    }

    fun collectErrors(): List<ReleaseError> {
        val errors: MutableList<ReleaseError> = mutableListOf()
        checkIssueStatus(errors)
        checkPreviousRelease(errors)
        checkPackageInStable(errors)
        checkDirectReleaseStage(errors)
        checkDependencies(errors)
        checkConsistency(errors)
        return errors
    }

    private fun checkIssueStatus(errors: MutableList<ReleaseError>) {
        when (release.status.key) {
            StartrekStatusKey.READY_TO_DEPLOY -> {}
            StartrekStatusKey.RM_ACCEPTANCE -> {
                if (!isNewCiApp()) {
                    errors += ReleaseError.warn("не забудьте нажать Accept до выкладки")
                }
            }
            else -> {
                errors += ReleaseError.error("неожиданный статус тикета (${release.status.display})")
            }
        }
    }

    private fun checkPreviousRelease(errors: MutableList<ReleaseError>) {
        val oldestUnclosedRelease = unclosedReleases
            .minByOrNull { it.createdAt }

        if (oldestUnclosedRelease != null
            && oldestUnclosedRelease.key != release.key
        ) {
            val prevReleaseLink = TelegramUtils.issueLink(oldestUnclosedRelease)
            errors += ReleaseError.error("предыдущий релиз ($prevReleaseLink) не выложен")
        }
    }

    private fun checkPackageInStable(errors: MutableList<ReleaseError>) {
        if (app.deployType != "yadeploy" && app.packageName != null) {
            val version = DirectReleaseUtils.getReleaseVersionString(release.summary)
                ?: return
            val environment = distUtils.getPackageEnvironment(app.packageName, version)
            if (environment == null) {
                val logLink = htmlLink("лог", currentLogViewerLink())
                errors += ReleaseError.warn("не получилось найти пакет на dist, $logLink")
            } else if (environment != "stable") {
                errors += ReleaseError.warn("пакет не в stable")
            }
        }
    }

    private fun checkDirectReleaseStage(errors: MutableList<ReleaseError>) {
        if (DirectAppsConfIgnoredFeatures.DIRECT_RELEASE !in app.ignoredFeatures
            && !isNewCiApp()
        ) {
            val isStableOrWaiting1 = listOf(Stage.STABLE, Stage.WAITING_1)
                .map { stage -> directReleaseService.getTicketByStage(app.name, stage) }
                .any { ticket -> ticket == release.key }

            if (!isStableOrWaiting1) {
                errors += ReleaseError.warn("в direct-release данный релиз не в stable, waiting-1")
            }
        }
    }

    private fun checkDependencies(errors: MutableList<ReleaseError>) {
        if (!releaseDependencyService.checkReleaseDependenciesSatisfied(dependencyGraph, release)) {
            val dependenciesLink = htmlLink("зависимостей", TabulaUtils.OBSERVATORIUM_LINK)
            errors += ReleaseError.error("нельзя выкладывать из-за $dependenciesLink")
        }
    }

    private fun checkConsistency(errors: MutableList<ReleaseError>) {
        if (false) {
            errors += ReleaseError.warn("проверка релиза упала (запустите dt-check-release-consistency %s на ppcdev'е)")
        }
    }
}
