package ru.yandex.direct.chassis.entity.release.service

import org.slf4j.LoggerFactory
import org.springframework.stereotype.Component
import ru.yandex.direct.chassis.entity.startrek.DependencyGraph
import ru.yandex.direct.chassis.entity.startrek.ReleaseDependencyService
import ru.yandex.direct.chassis.util.DirectAppsConfProvider
import ru.yandex.direct.chassis.util.DirectReleaseUtils
import ru.yandex.direct.chassis.util.DistUtils
import ru.yandex.direct.chassis.util.nonNullKeys
import ru.yandex.direct.chassis.util.startrek.StartrekHelper
import ru.yandex.direct.chassis.util.startrek.StartrekIssueTypeKey
import ru.yandex.direct.chassis.util.startrek.StartrekQueue
import ru.yandex.direct.chassis.util.startrek.StartrekStatusKey
import ru.yandex.startrek.client.model.Issue

@Component
class ReleaseCheckerFactory(
    private val startrek: StartrekHelper,
    private val distUtils: DistUtils,
    private val directAppsConfProvider: DirectAppsConfProvider,
    private val releaseDependencyService: ReleaseDependencyService,
    private val directReleaseService: DirectReleaseService,
) {
    private val logger = LoggerFactory.getLogger(ReleaseCheckerFactory::class.java)

    fun getUnclosedReleasesByApp(): Map<String, List<Issue>> {
        val releases: List<Issue> = startrek.findIssues(
            """
                Queue: ${StartrekQueue.DIRECT}
                Type: ${StartrekIssueTypeKey.RELEASE}
                Status: !${StartrekStatusKey.CLOSED}
                "Sort by": Created desc
            """.trimIndent()
        )

        return releases
            .groupBy { DirectReleaseUtils.getReleaseApp(it, directAppsConfProvider.getDirectAppsConf()) }
            .nonNullKeys()
            .mapKeys { (app, _) -> app.name }
    }

    private fun getUnclosedReleases(app: String): List<Issue> {
        return getUnclosedReleasesByApp()[app].orEmpty()
    }

    fun createReleaseContext(
        release: Issue,
        dependencyGraph: DependencyGraph? = null,
        unclosedReleasesByApp: Map<String, List<Issue>>? = null,
    ): ReleaseChecker? {
        val app = DirectReleaseUtils.getReleaseApp(release, directAppsConfProvider.getDirectAppsConf())
        if (app == null) {
            logger.warn("Unknown app for release ${release.key}")
            return null
        }

        val unclosedReleases = unclosedReleasesByApp
            ?.let { it[app.name].orEmpty() }
            ?: getUnclosedReleases(app.name)

        return ReleaseChecker(
            release = release,
            app = app,
            dependencyGraph = dependencyGraph
                ?: releaseDependencyService.buildDependencyGraph(),
            unclosedReleases = unclosedReleases,
            startrek = startrek,
            distUtils = distUtils,
            releaseDependencyService = releaseDependencyService,
            directReleaseService = directReleaseService,
        )
    }
}

